/*
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 * 
 * fig2e -- make fig format file into curve list
 * 
 * fig2e [ file ...] produces file.e for each file.
 * 
 * with no args, reads stdin, writes stdout
 */
#include "edgemap.h"
#include "fig.h"

char    Usage[] = "\
Usage: %s [fig_file ...]\n\
	converts fig file to edgemap; produces `fig_file.e'\n";

static int coord_type;

#define ORIGIN_LL 1		/* lower-left = 0,0 */
#define ORIGIN_UL 2		/* upper-left = 0,0 */

main(argc, argv)
int     argc;
char  **argv;
{
    char    c;
    extern char *optarg;
    extern int optind;

    PgmName = argv[0];

    while ((c = getopt(argc, argv, " ")) != -1) {

	switch (c) {

	case '?':		/* getopt() arg not in list */
	    fprintf(stderr, Usage, PgmName);
	    exit(1);

	}
    }

    if (optind < argc) {
	while (optind < argc) {
	    fig2e(argv[optind]);
	    optind++;
	}
    } else
	fig2e(NULL);

    return Status;
}

fig2e(file)
char   *file;
{
    char    buf[512];
    FILE   *fin = stdin, *fout = stdout;

    CurrentFile = file;
    CurrentLine = 1;

    if (file != NULL) {
	if ((fin = fopen(file, "r")) == NULL) {
	    fprintf(stderr,
		    "%s: no access to `%s', not processed\n",
		    PgmName, file);
	    Status = 1;
	    return;
	}
	strcpy(buf, FileName(file, ".e"));
	if ((fout = fopen(buf, "w")) == NULL) {
	    fprintf(stderr,
		    "%s: can't create `%s'; not processed\n",
		    PgmName, buf);
	    Status = 1;
	    return;
	}
    }
    if (ReadFigContours(fin) == FALSE) {
	if (file != NULL)
	    fclose(fin);
	Status = 1;
	return;
    }
    PrintContours(fout, "converted from fig");

    if (file != NULL) {
	fclose(fin);
	fclose(fout);
	fprintf(stderr, "%s: Wrote edge map to %s\n",
		PgmName, buf);
    }
    /* reset */
    FreeContours();
}

/*
 * returns TRUE for success
 */
int
ReadFigContours(f)
FILE   *f;
{
    PointList *p, *ReadFigContour();
    Contour *c;

    if (!ReadFigHeader(f))
	return FALSE;

    while ((p = ReadFigContour(f)) != NULL) {

	/* new contour */
	if (NContours >= MAXCONTOUR) {
	    fprintf(stderr,
		    "%s: file `%s', line %d: too many curves (%d) for me; ignoring the rest.\n",
		    PgmName, CurrentFile, CurrentLine, NContours);
	    Status = 2;
	    return NULL;
	}
	c = &Contours[NContours];
	NContours++;

	c->p = p;
    }

    ComputeSize();

    if (coord_type == ORIGIN_LL)
	FlipWhy();		/* flip all y coords about Height/2 */

    return TRUE;
}

ComputeSize()
{
    Contour *C;
    real    maxx = 0, maxy = 0;
    double  floor();

    for (C = &Contours[0]; C <= &Contours[NContours - 1]; C++) {

	PointList *p = C->p;
	int     i;

	for (i = 0; i < p->n; i++) {
	    if (maxx < p->x[i])
		maxx = p->x[i];
	    if (maxy < p->y[i])
		maxy = p->y[i];
	}
    }

    Height = (int) floor(maxy) + 1;
    Width = (int) floor(maxx) + 1;
}

FlipWhy()
{
    Contour *C;
    double  floor();

    for (C = &Contours[0]; C <= &Contours[NContours - 1]; C++) {

	PointList *p = C->p;
	int     i;

	for (i = 0; i < p->n; i++)
	    p->y[i] = Height - p->y[i];
    }
}

int
ReadFigHeader(stream)
FILE   *stream;
{
    char    buf[512];
    int     dpi;

    if (fgets(buf, 512, stream) == NULL) {
err:
	fprintf(stderr, "%s: %s line %d, doesn't look like a fig file\n",
		PgmName, CurrentFile, CurrentLine);
	return FALSE;
    }
    CurrentLine++;

    if (strncmp(buf, FIG_header_1st_line, strlen(FIG_header_1st_line)))
	goto err;

    do {
	if (fgets(buf, 512, stream) == NULL)
	    goto err;

	CurrentLine++;
    } while (buf[0] == '#');

    if (sscanf(buf, "%d%d", &dpi, &coord_type) != 2)
	goto err;
    if (dpi != 80)
	fprintf(stderr, "%s: warning: %s says %d dpi (fig 2.1 expects 80)\n",
		PgmName, CurrentFile, dpi);
    if (coord_type != ORIGIN_LL && coord_type != ORIGIN_UL) {
	fprintf(stderr,
		"%s: %s says coordinate type %d; should be %d or %d\n",
		PgmName, CurrentFile, coord_type, ORIGIN_LL, ORIGIN_UL);
	return FALSE;
    }
    return TRUE;
}

PointList *
ReadFigContour(stream)
FILE   *stream;
{
    static PointList *contour = NULL;
    PointList *AllocCopyPointList();
    char   *fgets();
    char    buf[512];
    int     point;

    /* the FIG polyline object variables */
    int     object_code;	/* (always 2) */
    int     sub_type;		/* (1 : polyline */

    /* 3 : polygon, closed contour) */
    int     line_style;		/* (0 : solid */

    /* 1 : dashed = continuation */
    /* 2 : dotted = image boundary) */
    int     thickness;		/* (-1 = not used) */
    int     color;		/* (-1 = not used) */
    int     depth;		/* (-1 = not used) */
    int     pen;		/* (-1 = not used) */
    int     area_fill;		/* (0 = unfilled) */
    float   style_val;		/* (0 if solid line_style */

    /* 4.0 = dash length if dashed */
    /* 3.2 = dot gap if dotted) */
    int     radius;		/* (0 = not used) */
    int     forward_arrow;	/* (0 = off) */
    int     backward_arrow;	/* (0 = off) */

    if (feof(stream))
	return NULL;

    do {
	if (fgets(buf, 512, stream) == NULL)
	    return NULL;
	CurrentLine++;

    } while (buf[0] == '#' || buf[0] == '\n');

    /* read the contour line */
    if (sscanf(buf, FIG_contour_scan_fmt,
	       &object_code,
	       &sub_type,
	       &line_style,
	       &thickness,
	       &color,
	       &depth,
	       &pen,
	       &area_fill,
	       &style_val,
	       &radius,
	       &forward_arrow,
	       &backward_arrow) != 12) {
not_polyline:
	fprintf(stderr,
		"%s: file `%s', line %d: doesn't look like a fig polyline\n",
		PgmName, CurrentFile, CurrentLine);
	return NULL;
    }
    if (object_code != 2 ||
	(sub_type != 1 && sub_type != 3) ||
	(line_style != 0 && line_style != 1 && line_style != 2) ||
	forward_arrow != 0 || backward_arrow != 0)
	goto not_polyline;

    if (contour == NULL)
	contour = NewPointList(10000);

    point = 0;

    /*
     * WARNING:  no comment line processing within a contour!
     */
    for (;;) {
	int     figx, figy;

	if (fscanf(stream, "%d%d", &figx, &figy) == NULL)
	    break;

	if (figx == 9999 && figy == 9999)
	    break;

	contour->x[point] = FIG_unpix(figx);
	contour->y[point] = FIG_unpix(figy);
	contour->theta[point] = 0;
	contour->k[point] = 0;

	point++;
    }
    if (point == 0)
	return NULL;
    if (point == 1) {
	fprintf(stderr,
		"%s: `%s' line %d one-point contour ignored\n",
		PgmName, CurrentFile, CurrentLine);
	return ReadFigContour(stream);
    }
    contour->n = point;

    contour->contin = (line_style == 1 ? TRUE : FALSE);
    contour->boundary = (line_style == 2 ? TRUE : FALSE);

    /* is last point repeated? */
    contour->closed = FALSE;
    if (contour->x[contour->n - 1] == contour->x[0]
	&& contour->y[contour->n - 1] == contour->y[0]) {

	contour->n--;
	contour->closed = TRUE;
	contour->withcorner = (sub_type == 1 ? TRUE : FALSE);
    }
    /* make a copy of the contour */
    return AllocCopyPointList(contour, 0, contour->n - 1);
}

/* copy p from from to to into an allocated point list */
PointList *
AllocCopyPointList(p, from, to)
PointList *p;
int     from, to;
{
    int     n;
    PointList *newp;

    if (from < 0 || from >= p->size
	|| to < 0 || to >= p->size
	|| from > to) {
	fprintf(stderr, "AllocCopyPointList called with bad args--fail\n");
	abort();
    }
    n = to - from + 1;
    newp = NewPointList(n);
    newp->n = n;
    newp->closed = p->closed;
    newp->withcorner = p->withcorner;
    newp->contin = p->contin;
    newp->boundary = p->boundary;

    CopyPointList(p, from, newp, 0, n);
    return newp;
}

CopyPointList(p, pfrom, p2, p2from, n)
PointList *p, *p2;
int     pfrom, p2from;
int     n;
{
    n *= sizeof (real);
    bcopy((char *) &p->x[pfrom], (char *) &p2->x[p2from], n);
    bcopy((char *) &p->y[pfrom], (char *) &p2->y[p2from], n);
    bcopy((char *) &p->theta[pfrom], (char *) &p2->theta[p2from], n);
    bcopy((char *) &p->k[pfrom], (char *) &p2->k[p2from], n);
}
