/*
 * Copyright (c) 1990,1,2 Mark Nitzberg
 * and President and Fellows of Harvard College
 * All rights reserved.
 * 
 * give values from elastica-based prior on contours; build elastica
 */
#include "edgemap.h"

real    ContourEnergy();
real    AngleFix();
real    Distance();

/*
 * NB. theta1 and theta2 point IN THE DIRECTION OF TRAVEL
 */

real
InitialElastLength(x1, y1, theta1, x2, y2, theta2, alpha, nu)
real    x1, y1, theta1, x2, y2, theta2, alpha, nu;
{
    real    dist = Distance(x1, y1, x2, y2);

    /* distance is increased:  triple if theta1 is u-turn to theta2 */
    dist += dist * 2 * (fabs(AngleFix(theta2 - theta1)) / M_PI);

    return dist;
}

real
BuildElast(p, x1, y1, theta1, x2, y2, theta2, alpha, nu)
PointList *p;
real    x1, y1, theta1, x2, y2, theta2, alpha, nu;
{
    static float SplineEnergy(), MinSpeed();
    real    dist = InitialElastLength(x1, y1, theta1, x2, y2, theta2, alpha, nu);
    real    speed;

    /* find speed near dist which minimizes length + curvature squared */
    MinSpeed(dist / 5.0, dist, dist * 5.0, SplineEnergy, .001, &speed,
	     p, x1, y1, theta1, x2, y2, theta2, alpha, nu);

    return SplineEnergy(speed, p, x1, y1, theta1, x2, y2, theta2, alpha, nu);
}

static float
SplineEnergy(speed, p, x1, y1, theta1, x2, y2, theta2, alpha, nu)
float   speed;
PointList *p;
real    x1, y1, theta1, x2, y2, theta2, alpha, nu;
{
    int     i;
    double  sint1, cost1, sint2, cost2;
    real    a, b, c, d, e, f, g, h;

    sincos(theta1, &sint1, &cost1);
    sincos(theta2, &sint2, &cost2);

    /* x(t) = a + bt + ct^2 + dt^3 */
    a = x1;
    b = speed * cost1;
    c = 3 * (x2 - x1) - speed * (2 * cost1 + cost2);
    d = 2 * (x1 - x2) + speed * (cost1 + cost2);

    /* y(t) = e + ft + gt^2 + ht^3 */
    e = y1;
    f = speed * sint1;
    g = 3 * (y2 - y1) - speed * (2 * sint1 + sint2);
    h = 2 * (y1 - y2) + speed * (sint1 + sint2);

    for (i = 0; i < p->n; i++) {
	real    t = (real) i / (real) (p->n - 1);
	real    t2 = t * t;	/* t^2 */
	real    t3 = t2 * t;	/* t^3 */
	real    xp, yp, xpp, ypp;	/* 1st & 2nd deriv's */
	real    norm;

	p->x[i] = d * t3 + c * t2 + b * t + a;
	p->y[i] = h * t3 + g * t2 + f * t + e;

	/* tangent angle */
	xp = 3 * d * t2 + 2 * c * t + b;
	yp = 3 * h * t2 + 2 * g * t + f;
	p->theta[i] = atan2(yp, xp);

	/* curvature */
	xpp = 6 * d * t + 2 * c;
	ypp = 6 * h * t + 2 * g;
	norm = xp * xp + yp * yp;
	norm = sqrt(norm);
	norm = norm * norm * norm;	/* to the 3/2 power */

	p->k[i] = (xp * ypp - yp * xpp) / norm;
    }
    return ContourEnergy(p, alpha, nu);
}

real
ContourEnergy(p, alpha, nu)
PointList *p;
real    alpha, nu;
{
    int     i;
    real    e = 0, lastds = 0;

    /* integral of nu + alpha curv squared !!ds!! */
    for (i = 0; i < p->n; i++) {
	real    ds = (i < p->n - 1
		      ? Distance(p->x[i], p->y[i], p->x[i + 1], p->y[i + 1])
		      : 0);
	real    avgds = (ds + lastds) / 2;

	e += (nu + alpha * p->k[i] * p->k[i]) * avgds;
	lastds = ds;
    }
    return e;
}


/* highly non-portable adaptation from numerical recipes "golden" */

#define R 0.61803399
#define C (1.0-R)
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);

static float
MinSpeed(ax, bx, cx, f, tol, xmin,
	 _p, _x1, _y1, _t1, _x2, _y2, _t2, _alpha, _nu)
float   ax, bx, cx, tol, *xmin;
float   (*f) ();
PointList *_p;
real    _x1, _y1, _t1, _x2, _y2, _t2, _alpha, _nu;
{
    float   f0, f1, f2, f3, x0, x1, x2, x3;

    x0 = ax;
    x3 = cx;
    if (fabs(cx - bx) > fabs(bx - ax)) {
	x1 = bx;
	x2 = bx + C * (cx - bx);
    } else {
	x2 = bx;
	x1 = bx - C * (bx - ax);
    }
    f1 = (*f) (x1, _p, _x1, _y1, _t1, _x2, _y2, _t2, _alpha, _nu);
    f2 = (*f) (x2, _p, _x1, _y1, _t1, _x2, _y2, _t2, _alpha, _nu);
    while (fabs(x3 - x0) > tol * (fabs(x1) + fabs(x2))) {
	if (f2 < f1) {
	    SHFT(x0, x1, x2, R * x1 + C * x3)
		SHFT(f0, f1, f2,
		   (*f) (x2, _p, _x1, _y1, _t1, _x2, _y2, _t2, _alpha, _nu))
	} else {
	    SHFT(x3, x2, x1, R * x2 + C * x0)
		SHFT(f3, f2, f1,
		   (*f) (x1, _p, _x1, _y1, _t1, _x2, _y2, _t2, _alpha, _nu))
	}
    }
    if (f1 < f2) {
	*xmin = x1;
	return f1;
    } else {
	*xmin = x2;
	return f2;
    }
}

#undef C
#undef R

/*
 * The precomputed prior file contains a 3-d prior:
 * 
 * base b = 1.0551131; ln b = .05364796497
 * 
 * logprior[logb r][theta/6.25 degrees][phi/12 degrees]
 * 
 * logb r = [0..29]	    r = [1 .. 4.7388] dist. units theta/6.25 =
 * [0..15]	theta = [0 .. 93.75] degrees phi/12 = [0..29]      phi = [0
 * .. 348] degrees
 */
static char readcmd[] = "PATH=/usr/ucb:$PATH;zcat logprior";

#define lnb 0.05364796497

#define NR 30
#define MaxR 4.7388
#define MinAlphaNuRatio 0
#define MaxAlphaNuRatio 16
#define sqrtMaxAlphaNuRatio 4

#define NTheta 16
#define NPhi 30

#define Rint(x) ((int)((x)+0.5))/* round up */

static double Prior[NR][NTheta][NPhi];

static int readlogprior = FALSE;

static void
ReadLogPrior()
{
    FILE   *f;
    int     r, theta, phi;
    double  sum;

    if ((f = popen(readcmd, "r")) == NULL) {
	fprintf(stderr, "Can't get log prior file via `%s'\n", readcmd);
	exit(2);
    }
    sum = 0.0;

    for (r = 0; r < NR; r++) {
	for (theta = 0; theta < NTheta; theta++) {
	    for (phi = 0; phi < NPhi; phi++) {
		int     c;

		for (;;) {
		    c = getc(f);
		    if (c == '#') {
			while ((c = getc(f)) != '\n' && c != EOF);
		    } else {
			ungetc(c, f);
			break;
		    }
		}
		if (fscanf(f, "%lg", &Prior[r][theta][phi]) != 1) {
		    fprintf(stderr, "Command `%s' didn't work--giving up.\n",
			    readcmd);
		    exit(2);
		}
		sum += Prior[r][theta][phi];
	    }
	}
    }
    pclose(f);			/* return value = exit status of zcat */

    /* normalize */
    for (r = 0; r < NR; r++)
	for (theta = 0; theta < NTheta; theta++)
	    for (phi = 0; phi < NPhi; phi++)
		Prior[r][theta][phi] /= sum;

    readlogprior = TRUE;
}


real
ElastPrior(x, y, phi, alpha, nu)
real    x, y, phi, alpha, nu;
{
    double  r, theta, lambda;
    double  alphanuratio;

    y = fabs(y);

    if (x == 0 && y == 0)
	return 0;

    if (!readlogprior)
	ReadLogPrior();

    theta = atan2(y, x);
    r = sqrt(x * x + y * y);

    theta = Rint(fabs(theta) / RAD(6.25));
    if (theta >= NTheta)
	return 0;

    if (phi < 0)
	phi += 2 * M_PI;
    phi = Rint(phi / RAD(12.0));

    /* scale r according to dimensions of picture and alpha/nu */
    if (nu == 0)
	alphanuratio = MaxAlphaNuRatio;
    else
	alphanuratio = alpha / nu;
    if (alphanuratio > MaxAlphaNuRatio)
	alphanuratio = MaxAlphaNuRatio;

    lambda = MAX(Height, Width) / MaxR;
    lambda *= sqrt(alphanuratio) / sqrtMaxAlphaNuRatio;
    r *= lambda;

    /* now put in log scale */
    r = Rint(log(r) / lnb);
    if (r < 0)
	return 0;
    if (r > NR - 1)
	r = NR - 1;
    if (theta > NTheta - 1)
	theta = NTheta - 1;
    if (phi > NPhi - 1)
	phi = NPhi - 1;

    return Prior[(int) r][(int) theta][(int) phi];
}
