/* Xanima.c      Distribution 1.2   91/1/28   Scry */

/*   The Scry system is copyright (C) 1988-1991 Regents  of  the
University  of  California.   Anyone may reproduce ``Scry'',
the software in this distribution, in whole or in part, pro-
vided that:

(1)  Any copy  or  redistribution  of  Scry  must  show  the
     Regents  of  the  University of California, through its
     Lawrence Berkeley Laboratory, as the source,  and  must
     include this notice;

(2)  Any use of this software must reference this  distribu-
     tion,  state that the software copyright is held by the
     Regents of the University of California, and  that  the
     software is used by their permission.

     It is acknowledged that the U.S. Government has  rights
in  Scry  under  Contract DE-AC03-765F00098 between the U.S.
Department of Energy and the University of California.

     Scry is provided as a professional  academic  contribu-
tion  for  joint exchange.  Thus it is experimental, is pro-
vided ``as is'', with no warranties of any kind  whatsoever,
no  support,  promise  of updates, or printed documentation.
The Regents of the University of California  shall  have  no
liability  with respect to the infringement of copyrights by
Scry, or any part thereof. */


/* programmers:  David Robertson and Antony Courtney */

/**********************************************************

Routines within this file:

  anima_init:           initialize Anima footer variables
  parse_args:           get command-line arguments
  init_vars:            initialize sequence-viewing variables
  file_text_proc:       input Anima file name
  gamma_proc:           set gamma setting for next frame
  sunrast_proc:         save current Anima frame as Sun rasterfile
  postscr_proc:         save current Anima frame as Postscript file
  quit_proc:            quit Xanima
  number_text_proc:     display frame number
  goto_proc:            seek to frame entered in "Number:" field
  frame_text_proc:      input Postscript dpi or frame number
  zoom_proc:            set zoom or unzoom for next frame
  step_back_proc:       step backward one frame
  step_forw_proc:       step forward one frame
  stop_proc:            still frame
  full_proc:            view sequence at full speed
  speed_proc:           set speed for viewing sequence
  backward_proc:        view sequence by decreasing frame number
  forward_proc:         view sequence by increasing frame number
  frame_monitor:        when timer expires, display a frame
  advance_proc:         used to unstill frame

**************************************************************/


#include <stdio.h>
#include <math.h>
#include <scry_anima.h>
#include <scry_image.h>
#include <scry_limits.h>
#include "xvwin.h"
#include "Xanima_ui.h"
#include "image_ui.h"

extern image_win_objects *image_win ;	/* display window */

int S_dont_repaint = 1 ;			/* if true, don't repaint */
int S_do_zoom = 0 ;			/* zoom or not */
float scry_gamma = S_DEF_GAMMA_VALUE ;		/* gamma value */
int gamma_changed = 0 ;			/* gamma value has changed or not */

Notify_value frame_monitor() ;	/* called to display frame on expiration
				   of timer */

struct itimerval itimevalue ;	/* timer expiration value */

Xanima_control_objects	*Xanima_control;	/* control panel */

/*
 * Instance XV_KEY_DATA key.  An instance is a set of related
 * user interface objects.  A pointer to an object's instance
 * is stored under this key in every object.  This must be a
 * global variable.
 */
Attr_attribute	INSTANCE;




void
main(argc, argv)

int argc;
char **argv;

{
	    /* initialize Xview */
	xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);
	INSTANCE = xv_unique_key();
	
	    /* initialize control panel */
	Xanima_control = Xanima_control_objects_initialize(NULL, NULL);
	    /* default image width and height */
	S_image_info.s_height = S_DEF_WINSERV_HEIGHT ;
	S_image_info.s_width = S_DEF_WINSERV_WIDTH ;
	    /* initialize Anima variables */
	init_vars() ;
	anima_init() ;
	    /* parse command-line arguments */
	parse_args (argc,argv) ;
	    /* initialize Anima display window */
	image_init(Xanima_control->control);
	    /* show window */
	(void) xv_set(image_win->win, XV_SHOW, TRUE, NULL) ;
	    /* if an Anima file has been entered on the command line */
	if (S_anima_filename[0] != NULL)
	{
		/* get it */
	    if (getfile())
	    {
                S_avars.frame.last = S_anima_count - 1;
                S_avars.start = TRUE ;
	    }
        }	
	/*
	 * Turn control over to XView.
	 */
	xv_main_loop(Xanima_control->control);
	exit(0);
}




/* initialize Anima footer variables */

anima_init()

{
    S_anima_file = NULL ;
    S_anima_filename[0] = '\0' ;
    S_anima_bytes = 0 ;
    S_anima_count = 0 ;
}




/* parse command-line arguments */

parse_args(argc, argv)

int argc;
char *argv[];

{
    int i;
    
    for(i = 1; i < argc; i++)
    {
        if(*argv[i] != '-')
        {
            fprintf(stderr,"bad command line:  use %s -h for help\n", argv[0]);
            exit(-1);
        }
        switch(*(argv[i]+1))
        {
            case 'i':		/* compressed data file */
                strcpy(S_anima_filename, argv[++i]);
                break;
            case 'c':		/* maximum number of colors to display */
		S_maxcol = atoi(argv[++i]) ;
		if ((S_maxcol < 1) || (S_maxcol > S_MAX_COL_SIZE))
		{
		    fprintf (stderr,"incorrect color entry maximum setting\n") ;
                    fprintf (stderr,"using default number of colors\n") ;
		    S_maxcol = S_MAX_COL_DISPLAY ;
		}
                break;
            case 'h':
                fprintf(stderr,"animation previewer.\n");
                fprintf(stderr,"Usually invoke with %s [args]\n",argv[0]);
                fprintf(stderr,"-i file : compressed image input file\n");
                fprintf(stderr,"-h : this message\n");
                exit(-1);
            default:
                printf("bad command line:  use %s -h for help\n", argv[0]);
                exit(-1);
        }
    }
}




/* initialize sequence-viewing variables */

init_vars()
{
    S_avars.stop = TRUE;		/* Stay put */
    S_avars.advance = FALSE;	/* Stay put */
    S_avars.speed = 5;		/* Half speed */
    S_avars.search = FALSE;	/* Stay put */
    S_avars.current = 0;		/* Start at 0 */
    S_avars.start = FALSE;		/* Nothing yet... */
    S_avars.direction = S_FORWARD;
    S_avars.quit = FALSE;		/* Don't die! */
    S_avars.pid = getpid();

    S_avars.frame.first = 0 ;

        /* set timer */
    itimevalue.it_value.tv_sec = 0 ; 
    itimevalue.it_interval.tv_sec = 0 ;
    itimevalue.it_value.tv_usec = (S_DELAY_INC * (10 - S_avars.speed)) ; 
    itimevalue.it_interval.tv_usec = (S_DELAY_INC * (10 - S_avars.speed)) ;
    xv_set(Xanima_control->speed_slider, PANEL_VALUE, S_avars.speed, 0) ;
    notify_set_itimer_func(Xanima_control->control, frame_monitor, ITIMER_REAL, 
			   &itimevalue, ((struct itimerval *) 0));
}




/*
 * Notify callback function for `file_input_text'.
 */

/* input Anima file name */

Panel_setting
file_text_proc(item, event)
	Panel_item	item;
	Event		*event;
{
    Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
    char *	value = (char *) xv_get(item, PANEL_VALUE);
	
    strcpy(S_anima_filename,value);

	/* get Anima file */
    if (!getfile())
	return panel_text_notify(item, event);

    xv_set(Xanima_control->file_input_text, PANEL_VALUE, "", 0) ;
    xv_set(Xanima_control->control,
	       PANEL_CARET_ITEM, Xanima_control->frame_text,
	       0);
    S_avars.frame.last = S_anima_count - 1;
    S_avars.start = TRUE ;

    return panel_text_notify(item, event);
}




/*
 * Notify callback function for `gamma_text'.
 */

/* set gamma setting for next frame */

Panel_setting
gamma_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	char *	value = (char *) xv_get(item, PANEL_VALUE);
	float temp_gamma ;
	
	temp_gamma = (float) atof(value) ;
	    /* only change if Anima is in stopped mode */
	if ((S_avars.stop) && ((temp_gamma > 0.1) && (temp_gamma <= 4.0)))
	{
            scry_gamma = temp_gamma ;
	    gamma_changed = 1 ;
            xv_set(Xanima_control->control,
	           PANEL_CARET_ITEM, Xanima_control->frame_text,
	           0);
	}
	else
            xv_set(Xanima_control->gamma_text, PANEL_VALUE, "", 0) ;
        return panel_text_notify(item, event);
}




/*
 * Notify callback function for `sunrast_button'.
 */

/* save current Anima frame as Sun rasterfile */

void
sunrast_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	
#ifdef TELL_WHERE
	fputs("Xanima: sunrast_proc\n", stderr);
#endif
	anima_to_sunrast() ;
}




/*
 * Notify callback function for `postscr_button'.
 */

/* save current Anima frame as Postscript file */

void
postscr_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	void anima_to_post();
	float tmp ;
	
#ifdef TELL_WHERE
	fputs("Xanima: postscr_proc\n", stderr);
#endif
	tmp = (float) S_avars.frame.first ;
	    /* default is 60 dpi */
	if (tmp == 0)
	    tmp = 60.0 ;
	anima_to_post(tmp) ;
	S_avars.frame.first = 0 ;
        xv_set(Xanima_control->frame_text, PANEL_VALUE, "", 0) ;
}




/*
 * Notify callback function for `quit_button'.
 */
void
quit_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	
#ifdef TELL_WHERE
	fputs("anima: quit_proc\n", stderr);
#endif
    notify_set_itimer_func(Xanima_control->control, NOTIFY_FUNC_NULL, ITIMER_REAL, 
			   NULL, NULL);
    fclose(S_anima_file);
    xv_set(Xanima_control->control, FRAME_NO_CONFIRM, TRUE, 0);
    xv_destroy_safe(Xanima_control->control);
    xv_set(image_win->anima_canvas, FRAME_NO_CONFIRM, TRUE, 0);
    xv_destroy_safe(image_win->anima_canvas);
    free(S_image_info.data) ;
}




/*
 * Notify callback function for `number_text'.
 */

/* display frame number (not user writeable) */

Panel_setting
number_text_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	char *	value = (char *) xv_get(item, PANEL_VALUE);
	
	return panel_text_notify(item, event);
}




/*
 * Notify callback function for `goto_button'.
 */

/* seek to frame entered in "Number:" field */

void
goto_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

	int temp ;
	
    temp = S_avars.frame.first ;
    if((temp < 0) || (temp >= S_anima_count))	/* frame number is invalid */
      xv_set(Xanima_control->frame_text, PANEL_VALUE, "", 0) ;
    else
    {
        S_avars.stop = FALSE;
        S_avars.advance = TRUE;
        S_avars.search = TRUE;
        S_avars.current = temp;
    }
    S_avars.frame.first = 0 ;
}




/*
 * Notify callback function for `frame_text'.
 */

/* input Postscript dpi or frame number from "Number:" field */

Panel_setting
frame_text_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	char *	value = (char *) xv_get(item, PANEL_VALUE);
	
	S_avars.frame.first = atoi(value);
	return panel_text_notify(item, event);
}




/*
 * Notify callback function for `zoom_button'.
 */

/* zoom or unzoom by a factor of 2 */

void
zoom_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	
        S_do_zoom = !S_do_zoom ;
	if (S_do_zoom)
	{
	    S_image_info.s_width *= 2 ;
	    S_image_info.s_height *= 2 ;
	}
	else
	{
	    S_image_info.s_width /= 2 ;
	    S_image_info.s_height /= 2 ;
	}
	S_zoom_changed = 1 ;
}




/*
 * Notify callback function for `step_back_button'.
 */

/* step backward one frame */

void
step_back_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	void advance_proc() ;
	
#ifdef TELL_WHERE
	fputs("anima: step_back_proc\n", stderr);
#endif
    S_avars.direction = S_BACKWARD;
    advance_proc();
}




/*
 * Notify callback function for `step_forw_button'.
 */

/* step forward one frame */

void
step_forw_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	void advance_proc() ;
	
#ifdef TELL_WHERE
	fputs("anima: step_forw_proc\n", stderr);
#endif
    S_avars.direction = S_FORWARD;
    advance_proc();
}




/*
 * Notify callback function for `stop_button'.
 */

/* still frame */

void
stop_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	
#ifdef TELL_WHERE
	fputs("anima: stop_proc\n", stderr);
#endif
    S_avars.stop = TRUE;
    xv_set(Xanima_control->frame_text, PANEL_VALUE, "", 0);
    xv_set(Xanima_control->control, PANEL_CARET_ITEM, Xanima_control->frame_text, 0);
}




/*
 * Notify callback function for `full_button'.
 */

/* view sequence at full speed */

void
full_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	
#ifdef TELL_WHERE
	fputs("anima: full_proc\n", stderr);
#endif
    /* set to maximum speed value */
    S_avars.stop = FALSE;
    S_avars.speed = 10;
    xv_set(Xanima_control->speed_slider, PANEL_VALUE, S_avars.speed, 0) ;
    itimevalue.it_value.tv_usec = S_FULL_SPEED ; 
    itimevalue.it_interval.tv_usec = S_FULL_SPEED ;
    notify_set_itimer_func(Xanima_control->control, frame_monitor, ITIMER_REAL, 
			   &itimevalue, ((struct itimerval *) 0));
}




/*
 * Notify callback function for `speed_slider'.
 */

/* set speed for viewing sequence */

void
speed_proc(item, value, event)
	Panel_item	item;
	int		value;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
        S_avars.speed = value ;
	
    if (S_avars.speed == 10)
    {
        itimevalue.it_value.tv_usec = S_FULL_SPEED ; 
        itimevalue.it_interval.tv_usec = S_FULL_SPEED ;
    }
    else if (S_avars.speed != 0)
    {
        itimevalue.it_value.tv_usec = (S_DELAY_INC * (10 - S_avars.speed)) ; 
        itimevalue.it_interval.tv_usec = (S_DELAY_INC * (10 - S_avars.speed)) ;
    }
    else
    {
        itimevalue.it_value.tv_usec = S_SLOWEST_SPEED ; 
        itimevalue.it_interval.tv_usec = S_SLOWEST_SPEED ;
    }
    notify_set_itimer_func(Xanima_control->control, frame_monitor, ITIMER_REAL, 
			   &itimevalue, ((struct itimerval *) 0));
}




/*
 * Notify callback function for `backward_button'.
 */

/* view sequence by decreasing frame number */

void
backward_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	
#ifdef TELL_WHERE
	fputs("anima: backward_proc\n", stderr);
#endif
    S_avars.direction = S_BACKWARD;
    S_avars.stop = FALSE;
}




/*
 * Notify callback function for `forward_proc'.
 */

/* view sequence by increasing frame number */

void
forward_proc(item, event)
	Panel_item	item;
	Event		*event;
{
	Xanima_control_objects	*ip = (Xanima_control_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
	
#ifdef TELL_WHERE
	fputs("anima: forward_proc\n", stderr);
#endif
    S_avars.direction = S_FORWARD;
    S_avars.stop = FALSE;
}




/* when timer expires, display a frame */

Notify_value
frame_monitor(client,which)

{
    char s[12] ;

    if ((!S_avars.stop) || (S_avars.start))
        display_frame();
    return (NOTIFY_DONE) ;
}




/* used to get out of still mode */
void
advance_proc()
{
    if(S_avars.stop == TRUE)
	{
	    S_avars.advance = TRUE;
	    S_avars.stop = FALSE;
	}
    else S_avars.stop = TRUE;
}
