#ifndef RpImageTile_CLASS
#define RpImageTile_CLASS

#include <RpType.h>
#include <RpReferenced.h>
#include <RpBinaryBuffer.h>
#include <RpImageArea.h>

#define		__TILE_MEMMGR_SIZE		(256)

// ************************************************************************
// CLASS: RpImageTile
// DESC:
//    This class implements a slab of image data.  The tile is specified
//    by an (x,y,z,c) coordinate which gives the lower-left corner.
//    It extends in the x and y directions.
//
class RpImageTile : public RpReferenced {
   public:              
      RpImageTile( const RpImageArea& setArea, RpType setType=RpFilm );
      RpImageTile( const RpImageTile& source );
      
      ~RpImageTile( void ) { }
      
      // ------------------------------------------------------------------
      // post: returns the size, in bytes, of the tile buffer.
      //
      unsigned
      getRAMSize( void ) const
      { return( _maArea.width * _maArea.height * bytesize( _mtType ));}
      
      // ------------------------------------------------------------------
      // post: returns the data type of the stored data.
      // 
      RpType
      getType( void ) const
      { return ( _mtType ); }
      
      // ------------------------------------------------------------------
      // post: returns the position and dimensions of the slab in image 
      //       space.
      // 
      const RpImageArea&
      getArea( void ) const
      { return ( _maArea ); }
      
      // ------------------------------------------------------------------
      // post: resizes the internal buffer & sets new data type
      // post: destroys current contents
      // note: use typecast() for a version to preserve contents.
      // post: on error returns a non-zero.
      int
      alloc( const RpImageArea& setArea, RpType setType);
      
      // ------------------------------------------------------------------
      // post: returns a pointer to the buffer at a specific coordinate.
      // post: returns 0 on error or for out-of-domain coordinate.
      //
      const void *
      getBufferAt( signed iColumn, signed iRow ) const;
      void *
      getBufferAt( signed iColumn, signed iRow );
      
      // ------------------------------------------------------------------
      // post: returns a pointer to the data buffer
      // note: this allows fast iteration over buffer contents
      // note: be sure to confirm type of data with getType()
      //
      const void *
      getBuffer(void) const;
      void *
      getBuffer( void );
      
      // ------------------------------------------------------------------
      // desc: fast object instantiator for RpImageTiles.
      // post:
      //    returns a pointer to an allocated tile.
      // 
      static RpImageTile *
      newTile( const RpImageArea& setArea, RpType setType = RpFilm );
      
      // ------------------------------------------------------------------
      // pre: (*this) was acquired by a C++ new operation
      //    or a ??::newTile() method
      // post: dereferenced a smart pointer.
      // note: It is important to use this to avoid memory leaks!!
      // 
      void
      deleteTile( void );

      // ------------------------------------------------------------------
      // post: resizes the internal buffer & sets new data type
      // post: typecasted contents to match new data type
      // post: on error returns a non-zero.
      // note: use alloc() for a fast version that destroys contents.
      //
      int
      typecast( RpType type );
      
      // ------------------------------------------------------------------
      // note: simply a convenience method.
      // post: on error, returns a non-zero.
      int
      floodFill( float rValue );
            
   protected:
      RpImageArea		_maArea;
      RpType			_mtType;
      RpBinaryBuffer		_buffer;

   protected:
   
      static unsigned		_memmgr_uCount;
      static RpImageTile	*_memmgr_ppStore[ __TILE_MEMMGR_SIZE ];
      
   protected:
   
      void
      forceInstantiation( void );
};


//
// copies to (dest) the data from (src) 
// which fills the intersection between ( area, dest.getArea(), src.getArea() )
//
// pre: dest.getType() == src.getType()
//
void
tilecpy( RpImageTile& dest, const RpImageTile& src, const RpImageArea& area );

#endif

