// ************************************************************************
//
// Here's an image processing work graph of non-trivial complexity.
// Count the number of convolutions.
//
// The purpose of this program is to remove grainy noise from images
// scanned from film negatives or prints. The process applies blurs
// to areas of low color-gradients.  Small blur kernels are used
// iteratively, rather than one large kernel, to prevent colors
// from bleeding across edge boundaries.  This implementation works
// well for clean images.  Noisy images may require additional blur
// passes.
//
// This program demonstrates a wide variety of Rampage's features:
//
// (1) Almost all the work (at the application level) takes
//     place with constructors.  Instantiating large
//     work graphs this way may feel awkward, so you may
//     wish to implement a graph-management data stucture.
//     Then you could instantiate image-op objects dynamically.
//     (If you do, then use RpInputImage::unregisterReference()
//     to deallocate them! See RpReferenced.h )
//
// (2) Notice how a RpTypeCaster is used in conjunction with forced
//     caching to identify a known bottleneck (the edge detection).
//
// (3) RpBinaryOp is used to implement simple, channel-uniform 
//     point-processing operations.  A wide variety of imaging
//     operations, particularly application-specific tweaks, can
//     be quickly implemented this way. See also RpUnaryOp.h
//
// (4) Image tiles serve as convolution kernels.
//     Though you can always create and validate convolution kernels
//     yourself, RpConvolver provides convenience methods for constructing
//     the most common kind of kernel, an origin-centered matrix.
//     RpConvolver::newKernel() has a corresponding ::deleteKernel()
//     function.
//
// (5) It is a good idea to manually set the image manager type and
//     the amount of RAM you'd like to use.  Use RpManReference
//     to do this.
//
// (6) Fill strategies can be used to fill out-of-frame areas on
//     sampled images.  Here, RpFillPadding is used to prevent
//     artifacts along the edges of the images when doing convolutions.
//     Without RpFillPadding, the default fill strategy (RpFillConstant)
//     will cause darkened edges in convolution outputs.  If this doesn't
//     bother you, then RpFillConstant is faster in execution...
//
// (c) 1997 Mayur Patel
// ************************************************************************

#include <RpSGIReader.h>
#include <RpSGIWriter.h>

#include <RpFillPadding.h>
#include <RpConvolver.h>
#include <RpCompositer.h>
#include <RpTypeCaster.h>
#include <RpBinaryOp.h>

#include <RpImageMan.h>

#include <assert.h>
#include <stdlib.h> // atol()
#include <string.h> // strcpy()
#include <unistd.h> // getopt()
#include <iostream.h>

//
// Here is a functoid class for blending partial edge-detect
// plates.  If you are familiar with component-programming
// (or generic programming or whatever), as is used in STL,
// then you will recognize how such a class is used:
// (See RpBinaryOp.h or RpUnaryOp.h for specifics)
//
class PixelBlend
{
   public:
      float
      operator()( float pix1, float pix2 ) const
      { return( ((pix1 * pix1) + (pix2 * pix2)) * 7.0 ); }
};

const float
fpEdgeKernel1[6] = 
{
   -1.0, 1.0,
   -1.0, 1.0,
   -1.0, 1.0,
};

const float
fpEdgeKernel2[6] =
{
   -1.0, -1.0, -1.0,
   1.0, 1.0, 1.0
};

const float
fpEdgeKernel3[6] = 
{
   1.0, -1.0,
   1.0, -1.0,
   1.0, -1.0,
};

const float
fpEdgeKernel4[6] =
{
   1.0, 1.0, 1.0,
   -1.0, -1.0, -1.0
};


const float
fpBlurKernel[3] =
{
   0.32, 0.36, 0.32
};


void
degrain( char *cpMB, char *cpFilename )
{
   char 		cpOutFile[1024];
   unsigned		uLoop;

   //
   // do degrain:
   //
   assert( cpFilename && cpMB );
   if ( cpFilename[0] )
   {
      //
      // set output filename:
      //
      strcpy( cpOutFile, "degrain." );
      strcpy( cpOutFile+8, cpFilename );
      
      //
      // initialize the image manager:
      //
      RpManReference		r;
      r.setManagerType( RpImageMan::SoftPull );
      r->setRAMBound( atol( cpMB ) << 20 );

      //
      // create edge detection kernels:
      //
      RpImageTile
         *pEdgeKernel1 = RpConvolver::newKernel( fpEdgeKernel1, 2, 3 );
         
      RpImageTile
         *pEdgeKernel2 = RpConvolver::newKernel( fpEdgeKernel2, 3, 2 );

      RpImageTile
         *pEdgeKernel3 = RpConvolver::newKernel( fpEdgeKernel3, 2, 3 );
         
      RpImageTile
         *pEdgeKernel4 = RpConvolver::newKernel( fpEdgeKernel4, 3, 2 );
      
      //
      // create blur kernels:
      //
      RpImageTile
         *pBlurKernel1 = RpConvolver::newKernel( fpBlurKernel, 1, 3 );
         
      RpImageTile
         *pBlurKernel2 = RpConvolver::newKernel( fpBlurKernel, 3, 1 );
         
      //
      // input image
      //
      RpFillPadding padder;
      RpSGIReader
         in( cpFilename );
         in.setFillStrategy( (RpFillStrategy *) &padder );
       
      //
      // create edge-detect matte
      //
      RpConvolver
         edge1( (RpSampledImage *) &in, pEdgeKernel1 );         

      RpConvolver
         edge2( (RpSampledImage *) &in, pEdgeKernel2 );

      RpConvolver
         edge3( (RpSampledImage *) &in, pEdgeKernel3 );

      RpConvolver
         edge4( (RpSampledImage *) &in, pEdgeKernel4 );
         
      RpBinaryOp< PixelBlend >
	 mixEdges1( (RpInputImage *)&edge1, (RpInputImage *)&edge2 );

      RpBinaryOp< PixelBlend >
         mixEdges2( (RpInputImage *)&edge3, (RpInputImage *)&edge4 );

      RpBinaryOp< PixelBlend >
         finalMix( (RpInputImage *)&mixEdges1, (RpInputImage *)&mixEdges2 );
      
      //
      // edge detect is a severe bottleneck, so
      // force caching of the edge image:
      //
      // Depth of the matte isn't too important, so use 
      // RpUChar to conserve RAM.
      //
      RpTypeCaster
         edges( (RpInputImage*) &finalMix, in.getArea(), RpUChar );
         edges.setFillStrategy( (RpFillStrategy *) &padder );
         r->forceCache( (RpSampledImage*) &edges );

      //
      // first-pass blur
      //
      RpConvolver
         softEdge1( (RpSampledImage *) &edges, pBlurKernel1 );
         softEdge1.setFillStrategy( (RpFillStrategy *) &padder );
         
      RpConvolver
         softEdges( (RpSampledImage *) &softEdge1, pBlurKernel2 );
         softEdges.setFillStrategy( (RpFillStrategy *) &padder );
         
      RpConvolver
         firstblur1( (RpSampledImage *) &in, pBlurKernel1 );
         firstblur1.setFillStrategy( (RpFillStrategy *) &padder );
         
      RpConvolver
         firstblur( (RpSampledImage *) &firstblur1, pBlurKernel2 );
         firstblur.setFillStrategy( (RpFillStrategy *) &padder );
         
      //
      // first-pass composite
      //
      RpCompositer
         firstpass( 
            (RpInputImage*) &in,
            (RpInputImage*) &firstblur,  
            (RpInputImage*) &softEdges,
            0,
            in.getArea().c
         );
         
      //
      // last-pass blur
      //
      RpConvolver
         lastblur1( (RpInputImage *) &firstpass, in.getArea(), pBlurKernel1 );
         lastblur1.setFillStrategy( (RpFillStrategy *) &padder );
         
      RpConvolver
         lastblur( (RpSampledImage *) &lastblur1, pBlurKernel2 );
      
      //
      // last-pass composite
      //
      RpCompositer
         degrained( 
            (RpInputImage*) &in,
            (RpInputImage*) &lastblur,  
            (RpInputImage*) &edges,
            0,
            in.getArea().c
         );
         
      //
      // write output file:
      //
      RpSGIWriter
       out( cpOutFile, (RpInputImage *) &degrained, in.getArea(), RpUChar );
       
      //
      // cleanup on exit:
      //
      RpConvolver::deleteKernel( pEdgeKernel1 );
      RpConvolver::deleteKernel( pEdgeKernel2 );
      RpConvolver::deleteKernel( pEdgeKernel3 );
      RpConvolver::deleteKernel( pEdgeKernel4 );
      
      RpConvolver::deleteKernel( pBlurKernel1 );
      RpConvolver::deleteKernel( pBlurKernel2 ); 

   }
   
   return;
}




int
main ( int argc, char *argv[] )
{
   char		cpMB[16];
   char		cpFilename[1024];
   
   int		iC;
   
   //
   // default:
   //
   strcpy( cpMB, "4" );
   cpFilename[0] = 0;
   
   //
   // get args:
   //
   while( (iC = getopt( argc, argv, "m:f:h")) != -1 )
   {
      switch( iC )
      {
         case( 'f' ):
            strcpy( cpFilename, optarg);
            break;
            
         case( 'm' ):
            strcpy( cpMB, optarg );
            break;
            
         case( 'h' ):
         default:
            cout << endl << argv[0] << endl;
            cout << '\t' << "-f 'SGI-filename'" << endl;
            cout << '\t' << "[-m 'megabytes'] (default == 4)" << endl;
            cout << '\t' << "[-h ] (this)" << endl;
            cout << endl;
            
            return( -1 );
      };
   }

   //
   // do degrain:
   //
   degrain( cpMB, cpFilename ); 
   
   return ( -1 );
}
