# vec.tcl - Tcl Cartesian vector utilities
#
# $Id: vec.tcl,v 1.1 96/02/11 21:35:04 leech Exp $
#
# Copyright (C) 1996, Jonathan P. Leech
#
# This software may be freely copied, modified, and redistributed,
# provided that this copyright notice is preserved on all copies.
#
# There is no warranty or other guarantee of fitness for this software,
# it is provided solely "as is". Bug reports or fixes may be sent
# to the author, who may or may not act on them as he desires.
#
# You may not include this software in a program or other software product
# without supplying the source, or without informing the end-user that the
# source is available for no extra charge.
#
# If you modify this software, you should include a notice giving the
# name of the person performing the modification, the date of modification,
# and the reason for such modification.
#
set __rcslog {
$Log:	vec.tcl,v $
Revision 1.1  96/02/11  21:35:04  leech
Initial revision

}

proc vector { {x 0} {y 0} {z 0} } {
    list $x $y $z
}

# Return infinity norm of v
proc vmax { v } {
    set max [abs [lindex $v 0]]

    set y [abs [lindex $v 1]]
    if { $y > $max } {
	set max $y
    }

    set z [abs [lindex $v 2]]
    if { $z > $max } {
	set max $z
    }

    return $max
}

proc vmag { v } {
    set x [lindex $v 0]
    set y [lindex $v 1]
    set z [lindex $v 2]
    sqrt [expr $x * $x + $y * $y + $z * $z ]
}

proc vnorm { v } {
    set x [lindex $v 0]
    set y [lindex $v 1]
    set z [lindex $v 2]
    set mag [expr $x * $x + $y * $y + $z * $z]
    if { $mag > 0 } {
	set maginv [expr 1.0 / [sqrt $mag]  ]
	list [expr $x * $maginv] [expr $y * $maginv] [expr $z * $maginv]
    } else {
	return $v
    }
}

proc vdot { a b } {
    expr [lindex $a 0] * [lindex $b 0] + \
	 [lindex $a 1] * [lindex $b 1] + \
	 [lindex $a 2] * [lindex $b 2]
}

proc vcross { a b } {
    set x1 [lindex $a 0]
    set y1 [lindex $a 1]
    set z1 [lindex $a 2]
    set x2 [lindex $b 0]
    set y2 [lindex $b 1]
    set z2 [lindex $b 2]

    list [expr $y1 * $z2 - $y2 * $z1 ] \
	 [expr $x2 * $z1 - $x1 * $z2 ] \
	 [expr $x1 * $y2 - $x2 * $y1 ]
}

proc vadd { a b } {
    set x1 [lindex $a 0]
    set y1 [lindex $a 1]
    set z1 [lindex $a 2]
    set x2 [lindex $b 0]
    set y2 [lindex $b 1]
    set z2 [lindex $b 2]

    list [expr $x1 + $x2] [expr $y1 + $y2] [expr $z1 + $z2]
}

proc vsub { a b } {
    set x1 [lindex $a 0]
    set y1 [lindex $a 1]
    set z1 [lindex $a 2]
    set x2 [lindex $b 0]
    set y2 [lindex $b 1]
    set z2 [lindex $b 2]

    list [expr $x1 - $x2] [expr $y1 - $y2] [expr $z1 - $z2]
}

# Vector * Scalar
proc vsmul { v s } {
    set x [lindex $v 0]
    set y [lindex $v 1]
    set z [lindex $v 2]
    list [expr $x * $s] [expr $y * $s] [expr $z * $s]
}

# Matrix creation
proc matrix {} {
    list 0 0 0 0 0 0 0 0 0 0 0 0
}

# mget list row col -> float
proc mget { mat row col } {
    set index [expr $row * 4 + $col]
    lindex $mat $index
}

# mset list row col float -> list
proc mset { mat row col value } {
    set index [expr $row * 4 + $col]
    lreplace $mat $index $index $value
}

proc mprint { mat } {
    for { set row 0 } { $row < 3 } { incr row } {
	puts "[mget $mat $row 0] [mget $mat $row 1] [mget $mat $row 2] [mget $mat $row 3]"
    }
}

# Matrix multiplication
proc matmul { a b } {
    set res [matrix]

    for { set row 0 } { $row < 3 } { incr row } {
	for { set col 0 } { $col < 4 } { incr col } {
	    set sum 0
	    for { set k 0 } { $k < 3 } { incr k } {
		set sum [expr $sum + [mget $a $row $k] * [mget $b $k $col]]
	    }
	    if { $row == 3 } {
		set sum [expr $sum + [mget $res $row $col] * [mget $a $row 3]]
	    }
	    set res [mset $res $row $col $sum]
	}
    }
    return $res
}

# Returns { d l }
#   d = distance from point v to line passing through p1 and p2
#   l = distance along line segment to projection of v on p1-p2
proc linedist {v p1 p2} {
    # Compute normalized line direction
    set n [vnorm [vsub $p2 $p1]]
    echo n = $n
    # Compute distance
    set p [vsub $v $p1]
    echo p = $p
    set d [vmag [vcross $n $p]]
    set l [vdot $p $n]
    return [list $d $l]
}

# Distance between two points
proc distance { a b } {
    return [vmag [vsub $b $a]]
}
