/* FILE: misc.c

   Miscellaneous generic functions. A pseudo random number generator
   and routine to convert RGB values to HSV values and vice versa.

   Written by Petri Kuittinen, last modifications 14th July 1993.
*/ 

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include "misc.h"

/* Routines to generate pseudo random numbers.

   Original code by James E. Wilson 
   
   This algorithm is from:
   Stephen K. Park and Keith W. Miller, "Random Number Generators:
   Good ones are hard to find", Communications of the ACM, October 1988,
   vol 31, number 10, pp. 1192-1201.

   Written by Petri Kuittinen, last modifications 14th July 1993.
*/

#define RNG_M 0x7fffffff  /* m = 2^31 - 1 */
#define RNG_A 16807
#define RNG_Q 127773      /* m div a */
#define RNG_R 2836        /* m mod a */

/* FUNCTION:     int set_rnd_seed ()
   DESCRIPTION:  set current random generator seed value
   PARAMETERS:   new seed value
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 14th July 1993
*/
void set_rnd_seed (int new_seed)
{
  rnd_seed = (new_seed%(RNG_M-1))+1; /* Set seed to value between 1 and m-1 */
}


/* FUNCTION:     int rnd_int ()
   DESCRIPTION:  get a pseudo random number.
   PARAMETERS:   none
   RETURNS:      integer random number in range between 1 and m-1.
   
   Written by Petri Kuittinen, last modifications 14th July 1993
*/
int rnd_int ()
{
  register long low, high, test;

  high = rnd_seed/RNG_Q;
  low = rnd_seed%RNG_Q;
  test = RNG_A*low-RNG_R*high;
  rnd_seed = test;
  if (test<0) rnd_seed += RNG_M;

  return rnd_seed;
}


/* FUNCTION:     double rnd_double ()
   DESCRIPTION:  get a pseudo random number.
   PARAMETERS:   none
   RETURNS:      double floating point random number in range between 0 and 1.
   
   Written by Petri Kuittinen, last modifications 16th July 1993
*/
double rnd_double ()
{
  return (rnd_int()/(double)RNG_M);
}


/* FUNCTION:     char *duplicate_string (char *s)
   DESCRIPTION:  duplicates a string.
   PARAMETERS:
   char *s       pointer to a string to be duplicates         
   RETURNS:      pointer to duplicated string. Returns NULL if malloc fails.
   
   Written by Petri Kuittinen, last modifications 12nd July 1993
*/
char *duplicate_string (char *s)
{
  register char *new;

  new = malloc(strlen(s)+1);
  if (new==NULL) return NULL;

  strcpy (new, s);
  return new;
}


/* FUNCTION:     void rgb_to_hsv(double r, double g, double b,
   double *h, double *s, double *v)
   DESCRIPTION:  Convert RGB values to HSV values.
   PARAMETERS:
   double r      red value [0,1]
   double g      green value [0,1]
   double b      blue value [0,1]
   double *h     pointer to hue value [0,360]
   double *s     pointer to saturation value [0,1]
   double *v     pointer to v value [0,1]

   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 21th July 1993
*/
void rgb_to_hsv(double r, double g, double b, double *h, double *s, double *v)
{
  register double max, min, delta;

  max = MAX(r, g); max = MAX(max, b);
  min = MIN(r, g); min = MIN(min, b);

  *v = max; /* value for v */
  
  /* Determine saturation */
  if (max)
    *s = (max-min)/max;
  else
    *s = 0.0;

  /* if (*s==0.0)
   *h = ; */
  /* Saturation is zero, so hue cannot be determined */
  if (*s!=0.0)
    {
      delta = max-min;
      if (r==max)
	*h = (g-b)/delta;
      else if (g==max)
	*h = 2+(b-r)/delta;
      else
	*h = 4+(r-g)/delta;
      *h *= 60;
      if (*h<0) *h += 360;
    }
}


/* FUNCTION:     void hsv_to_rgb(double h, double s, double v,
   double *r, double *g, double *b)
   DESCRIPTION:  Convert HSV values to RGB values.
   PARAMETERS:
   double h     hue value [0,360]
   double s     saturation value [0,1]
   double v     v value [0,1]
   double *r    pointer to red value [0,1]
   double *g    pointer to green value [0,1]
   double *b    pointer to blue value [0,1]

   RETURNS:     nothing
   
   Written by Petri Kuittinen, last modifications 20th July 1993
*/

void hsv_to_rgb(double h, double s, double v, double *r, double *g, double *b)
{
  register double p1, p2, p3, f;
  register int i;

  if (h==360) h=0;
  h /= 60;
  i = h;
  f = h-i;
  p1 = v*(1-s);
  p2 = v*(1-s*f);
  p3 = v*(1-(s*(1-f)));
	  
  switch(i)
    {
    case 0:
      *r = v; *g = p3; *b = p1; break;
    case 1:
      *r = p2; *g = v; *b = p1; break;
    case 2:
      *r = p1; *g = v; *b = p3; break;
    case 3:
      *r = p1; *g = p2; *b = v; break;
    case 4:
      *r = p3; *g = p1; *b = v; break;
    default:
      *r = v; *g = p1; *b = p2; break;
    }
}
