/* FILE:   fill.c
 
   Routines for filling polygons are are here.

   Written by Petri Kuittinen, last modifications 25th August 1993.
*/

#include <stdlib.h>
#include "misc.h"
#include "fill.h"

/* FUNCTION:     struct polygon *create_polygon()
   DESCRIPTION:  Create an empty polygon.
   PARAMETERS:
   none
   RETURNS:      pointer to polygon struct, NULL if allocating memory fails.
   
   Written by Petri Kuittinen, last modifications 14th August 1993
*/
struct polygon *create_polygon()
{
  struct polygon *new_poly;

  new_poly = malloc(sizeof(struct polygon));
  if (new_poly == NULL) return NULL;

  new_poly->edge_list = NULL;
  new_poly->n_edges = 0;
  
  return new_poly;
}

/* FUNCTION:     void add_polygon_edge (struct polygon *poly, int x1, int y1,
   int x2, int y2)
   DESCRIPTION:  Add one edge to polygon.
   PARAMETERS:
   struct polygon *poly  pointer to polygon struct
   int x1        end coordinates of the edge (need to be inside screen!)
   int y1
   int x2
   int y2
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 25th August 1993
*/
void add_polygon_edge (struct polygon *poly, int x1, int y1, int x2, int y2)
{
  register struct poly_edge *p;

  if (y2==y1) return;

  if (y2<y1)
    {
      SWAP (x1, x2);
      SWAP (y1, y2);
    }
  
  /* no clipping implemented yet! */

  p = poly->edge_list;

  if (poly->n_edges==0)
    {
      if (x1<=x2)
	{
	  poly->min_x = x1;
	  poly->max_x = x2;
	}
      else
	{
	  poly->min_x = x2;
	  poly->max_x = x1;
	}
      poly->min_y = y1;
      poly->max_y = y2;
    }

  /* Allocate more space every 256th polygon edge */
  if (((poly->n_edges)&255)==0)
    {
      poly->edge_list = realloc (poly->edge_list,
				 (poly->n_edges+256)*sizeof(struct poly_edge));
      if (poly->edge_list==NULL)
        {
          poly->n_edges = 0;
          return;
        }
    }
  
  /* insert new polygon edge */
  p = poly->edge_list+poly->n_edges;

  p->x1 = x1;
  p->y1 = y1;
  p->x2 = x2;
  p->y2 = y2;

  p->sx = SGN(x2-x1);
  if (x2<x1) {SWAP(x1, x2);}
  p->ax = (x2-x1)<<1;
  p->d = p->ax-(y2-y1); /* <-- there seems to be something wrong with this
			   don't know what! */
  p->dy = (y2-y1);

  /* shorten intersection points */
  if (poly->n_edges>0)
    {
      if ((p-1)->y1<p->y1 && (p-1)->y2<p->y2)
	(p-1)->dy--;
      else if ((p-1)->y1>p->y1 && (p-1)->y2>p->y2)
	p->dy--;
    }
  poly->n_edges ++;

  if (x1<poly->min_x) poly->min_x = x1;
  if (x2>poly->max_x) poly->max_x = x2;
  if (y1<poly->min_y) poly->min_y = y1;
  if (y2>poly->max_y) poly->max_y = y2;
}


/* FUNCTION:     void delete_polygon(struct polygon *poly)
   DESCRIPTION:  Create an empty polygon.
   PARAMETERS:
   struct polygon *poly  pointer to polygon struct.
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 14th August 1993
*/
void delete_polygon(struct polygon *poly)
{
  if (poly->n_edges)
    {
      free (poly->edge_list);
      poly->edge_list = NULL;
      poly->n_edges = 0;
    }
  free (poly);
}

/* A macro to swap to polygon edges */
#define SWAP_EDGES(E1, E2) \
SWAP(E1->x1,E2->x1); \
SWAP(E1->y1,E2->y1); \
SWAP(E1->x2,E2->x2); \
SWAP(E1->y2,E2->y2); \
SWAP(E1->d,E2->d); \
SWAP(E1->ax,E2->ax); \
SWAP(E1->sx,E2->sx); \
SWAP(E1->dy,E2->dy);

/* FUNCTION:     void fill_polygon (struct polygon *poly,
   void (*put_pixel) (int x, int y))
   DESCRIPTION:  Fills the polygon using a shit algorithm by me which
   doesn't always work!
   PARAMETERS:
   struct polygon *poly              pointer to polygon struct.
   void (*put_pixel) (int x, int y)  pointer to the pixel drawing routine.
   RETURNS:      nothing
   
   Written by Petri Kuittinen, last modifications 25th August 1993
*/
void fill_polygon (struct polygon *poly, void (*put_pixel) (int x, int y))
{
  register int x1, x2, y;
  register struct poly_edge *p, *p2, *p3, *head, *tail, *end;
  
  if (poly->n_edges==0) return;  /* empty polygon */

  head = tail = poly->edge_list; /* pointer to first and last items in
				    active edge list */
  end = head+poly->n_edges-1;    /* points to end of edge list */


  /* shorten first/last intersection point */
  if (head!=end)
    {
      if (head->y1<end->y1 && head->y2<end->y2)
	head->dy--;
      else if (head->y1>end->y1 && head->y2>end->y2)
	end->dy--;
    }

  /* Sort edges by increasing y1 order */
  p = head;
  while (p!=end)
    {
      p2 = p3 = p;
      y = p->y1;
      while (p2++!=end)
	{
	  if (p2->y1<y)
	    {
	      y = p2->y1;
	      p3 = p2;
	    }
      	}
      if (p!=p3)
	{
	  SWAP_EDGES(p, p3);
	  if (p->y1==p->y2) /* horizontal line? */
	    {
	      x1 = p->x1;
	      x2 = p->x2;
	      do
		(*put_pixel) (x1++,y);
	      while (x1<=x2);

	      p->dy = -1;
	    }
	}
      p++;
    }

  y = head->y1;

  /* Do filling until all edges processed */
  while (head<=end)
    {
      /* remove edges from active edge list */
      p = p2 = head;
      do
	{
	  if (p->dy<0)
	    {
	      SWAP_EDGES(p, head);
	      head++;
	    }
	}
      while(p++!=tail);

      /* update activate edge list */
      while (tail<end && (tail+1)->y1==y) 
	tail++;
      while (head<=end && head->dy<0)
	head++;
      if (head>end) break;
 
      /* Sort active edge list by smallest x1, using selection sort */
      p = head;
      do
	{
	  p2 = p3 = p;
	  x1 = p->x1;
	  do
	    {
	      if (p2->x1<x1)
		{
		  x1 = p2->x1;
		  p3 = p2;
		}
	    }
	  while(p2++!=tail);
	    
	  if (p!=p3) {SWAP_EDGES(p, p3);}
	}
      while(p++!=tail);

      /* Draw hlines */
      p = head;
      do
	{
	  x1 = p->x1;
	  if (p!=tail)
	    {
	      p++;
	      x2 = p->x1;
	      do
		(*put_pixel) (x1++,y);
	      while (x1<=x2);
	    }
	  else
	    {
	      (*put_pixel) (x1,y);
	    }
	}
      while(p++!=tail);

      /* Update edge list */
      p = head;
      do
	{
	  while (p->d>=0)
	    {
	      p->d -= (p->y2-p->y1)<<1;
	      p->x1 += p->sx;
	    }
	  p->d += p->ax;
	  p->dy--;
	}
      while(p++!=tail);

      y++;
    }
} 
