/*
$Header: error.c,v 1.5 93/08/30 11:01:23 hmallat Exp $
*/

#include <stdio.h>
#include <stdlib.h>
#include "pixel.h"
#include "error.h"
#include "list.h"
#include "undo.h"
#include "X_misc.h"
#include "file_access.h"
#include "tools1.h"

struct view *kill_view = NULL, *v_p = NULL;

/* FUNCTION:       panic(char *s, int flag)
   DESCRIPTION:    Do a panic save of the unsaved images, print an error
                   message to stderr and exit, if the error was a fatal one.
		   If not, display an X alertbox with the errormessage.
   PARAMETERS:
   char *s         error message
   flag            TRUE == fatal
   RETURNS:        nothing
   
   Written by Hannu Mallat, last modifications 24th August 1993.
*/   
void panic(char *s, int flag)
{
  extern struct view *head_p;
  struct view *v_p = head_p;

  while(v_p != NULL)
    {
      if((v_p->pic_p->flags & TOUCHED) != 0)
	{
	  v_p->pic_p->name = realloc(v_p->pic_p->name, 
				     strlen(v_p->pic_p->name)+
				     strlen(".panic")+1);
	  strcat(v_p->pic_p->name, ".panic");
	  save_file(v_p->pic_p, FALSE);
	}
      v_p = v_p->next;
    }
  if(flag == TRUE)
    {
      fprintf(stderr, "phoenix: FATAL: %s\n", s);
      really_exit();
    }
  else
    {
      fprintf(stderr, "phoenix: ERROR: %s\n", s);
      show_errormsg(s, "Go on...", NULL);
    }
}

/* FUNCTION:       abandon_q()
   DESCRIPTION:    Prompt the user with a question if he really wants to 
                   abandon a picture.
   PARAMETERS:
   none
   RETURNS:        Nothing
   
   Written by Hannu Mallat, last modifications 19th July 1993.
*/   
void abandon_q()
{
  extern struct view *cur_view;
  extern struct picture *cur_pic;
  char *s, *q="Do you really want to abandon all views to ";

  if(cur_pic != NULL && cur_view != NULL)
    {
      s = (char *)malloc((strlen(q)+strlen(cur_pic->name)+2)*sizeof(char));
      if(s != NULL)
	{
	  sprintf(s, "%s\n%s", q, cur_pic->name);
	  ask_question(s, "Abandon", "Cancel", abandon_q_func);
	  free(s);
	}
      else
	panic("Mallocing a string failed (abandon_q)", TRUE);
    }
  else
    show_errormsg("No active picture to abandon.", "OK", NULL);
}

/* FUNCTION:       abandon_q_func()
   DESCRIPTION:    Function where to return after questioning
   PARAMETERS:
   none
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 20th July 1993.
*/   
void abandon_q_func(int client_data)
{
  switch(client_data)
    {
    case B_OK:
      all_busy();
      abandon_picture();
      all_busy_nomore();
      break;
    case B_CANCEL:
    default:
      break;
    }
}

/* FUNCTION:       exit_init()
   DESCRIPTION:    Initialize the question sequence for unsaved pictures
                   when exiting program.
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 2nd August 1993.
*/   
void exit_init()
{
  extern struct view *head_p, *v_p;
  char *s, *q = " has been modified but not saved.\n Do you want to save it?";

  v_p = head_p;
  skip_untouched();
  if(v_p != NULL)
    {
      s = (char *)malloc((strlen(q)+strlen(v_p->pic_p->name)+1)
			 *sizeof(char));
      if(s != NULL)
	{
	  sprintf(s, "%s%s", v_p->pic_p->name, q);
	  ask_question3(s, "Save", "Abandon", "Cancel", exit_q_func);
	  free(s);
	}
      else
	panic("Mallocing a string failed (exit_init)", TRUE);
    }
  else
    really_exit();
}

/* FUNCTION:       really_exit()
   DESCRIPTION:    Clear the allocated memory and exit
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 2nd August 1993.
*/   
void really_exit()
{
  all_busy();
  abandon_all();
  del_filters();
  all_busy_nomore();
  exit(0);
}

/* FUNCTION:       skip_untouched()
   DESCRIPTION:    Skip over pictures that have not been modified after the 
                   last save
   PARAMETERS:
   none
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 2nd August 1993.
*/   
void skip_untouched()
{
  extern struct view *v_p;

  while(v_p != NULL)
    if((v_p->pic_p->flags & TOUCHED) == 0)
      v_p = v_p->next;
    else
      return;
}

/* FUNCTION:       exit_q_func(int client_data)
   DESCRIPTION:    Save a picture if so requested and proceed to next 
                   unsaved one until every one has been processed
   PARAMETERS:
   client_data     which button was pressed in the question?
   RETURNS:        nothing

   Written by Hannu Mallat, last modifications 2nd August 1993.
*/   
void exit_q_func(int client_data)
{
  extern struct view *v_p;
  char *s, *q = " has been modified but not saved.\n Do you want to save it?";

  switch(client_data)
    {
    case B_OK:
      save_file(v_p->pic_p, TRUE);
      break;
    case B_NO:
      untouch(v_p->pic_p);
      break;
    case B_CANCEL:
      return;
      break;
    }
  skip_untouched();
  if(v_p != NULL)
    {
      s = (char *)malloc((strlen(q)+strlen(v_p->pic_p->name)+1)
			 *sizeof(char));
      if(s != NULL)
	{
	  sprintf(s, "%s%s", v_p->pic_p->name, q);
	  ask_question3(s, "Save", "Abandon", "Cancel", exit_q_func);
	  free(s);
	}
      else
	panic("Mallocing a string failed (exit_q_func)", TRUE);
    }
  else
    really_exit();
}

/* FUNCTION:       kill_window_q(struct view *v_p)
   DESCRIPTION:    Currently, kills a window. Should be changed to ask
                   the user a question if the window is the last view to
		   an image.
   PARAMETERS:
   *v_p            Pointer to the view
   RETURNS:        Nothing

   Written by Hannu Mallat, last modifications 19th July 1993.
*/   
/*
void kill_window_q(struct view *v_p)
{
  del_view(v_p);
  update_menu();
}
*/
void kill_window_q(struct view *u_p)
{
  extern struct view *head_p, *kill_view;
  struct view *w_p = head_p;
  int n = 0;

  if(u_p != NULL)
    {
      while(w_p != NULL)
	{
	  if(u_p->pic_p == w_p->pic_p) n++;
	  w_p = w_p->next;
	}
      if(n > 1)
	{
	  kill_view = NULL;
	  del_view(u_p);
	  update_menu();
	}
      else
	{
	  kill_view = u_p;
	  ask_question("Killing the last view means abandoning the image!",
		       "Abandon", "Cancel", kill_q_func);
	}
    }
}

void kill_q_func(int client_data)
{
  extern struct view *kill_view;

  switch(client_data)
    {
    case B_OK:
      del_view(kill_view);
      kill_view = NULL;
      update_menu();
      break;
    case B_CANCEL:
    default:
      break;
    }
}
