/*
 * file:     cmapObject.c
 * author:   Wes Barris
 * date:     5/05/92
 * purpose:  handles all cmap functions (drawing, creating, editing)
 *
 * copyright info:
 *
 *    @Copyright 1992
 *    Research Equipment Inc. dba Minnesota Supercomputer Center
 *
 * RESTRICTED RIGHTS LEGEND
 *
 * Use, duplication, or disclosure of this software and its documentation
 * by the Government is subject to restrictions as set forth in subdivision
 * { (b) (3) (ii) } of the Rights in Technical Data and Computer Software
 * clause at 52.227-7013.
 */

#include "desi.h"
#include "proto.h"
#include <sys/stat.h>
#include <gl/gl.h>
#include <X11/Xirisw/GlxMDraw.h>
#include <Xm/Xm.h>
#include <Xm/FileSB.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>
#include "SelfM/SelfM.h"
#include "Umsc/List.h"

extern DESI		desi;
extern UmscList		theList;
extern int		needtosave;
extern char		systype[10];
extern int		currentFrame;
extern GenericInfo	defaultInfo;

static Widget	csb = NULL,
		dw,
		left,
		right,
		flip;
static int	c_or_e;
static GLXconfig glxConfig [] = {
    { GLX_NORMAL, GLX_RGB, TRUE },
    { 0, 0, 0 }
};


/*
 * This routine is called any time a cmap object needs to be re-drawn.
 */
void
DrawCmap(GenericInfo *thisObject)
{
   Arg			args[10];
   Dimension		wwidth, wheight;
   FILE			*pixin;
   char			cmd1[256], cmd2[256], flip;
   float		xscale, yscale;
   int			xl, xr, yu, yd;
   int			i, bufsiz, n, extra;
   short		cv[3];
   short		vert[2];
   struct stat		buf;
/*
 * We want to draw into the correct window.
 */
   GLXwinset(XtDisplay(desi.toplevel), XtWindow(thisObject->w));
   if (thisObject->decoration == DROP_SHADOW)
      extra = thisObject->dec_size;
   else if (thisObject->decoration == BORDER)
      extra = 2*thisObject->dec_size;
   else
      extra = 0;
   viewport(0, (Screencoord)(thisObject->width+extra-1),
            0, (Screencoord)(thisObject->height+extra-1));
   myortho2(-0.5, (Coord)(thisObject->width+extra-0.5),
            -0.5, (Coord)(thisObject->height+extra-0.5));
/*
 * Get the size of the cmap widget and compare it to the size required
 * by thisObject to see if we have to adjust for decorations.
 */
   n = 0;
   XtSetArg(args[n], XmNwidth, &wwidth); n++;
   XtSetArg(args[n], XmNheight, &wheight); n++;
   XtGetValues(thisObject->w, args, n);
   if (thisObject->width+extra != (int)wwidth  ||
       thisObject->height+extra != (int)wheight) {
/*
 * The difference in size must be accounted for by the decoration.
 */
      n = 0;
      XtSetArg(args[n],XmNwidth,(int)thisObject->width+extra);n++;
      XtSetArg(args[n],XmNheight,(int)thisObject->height+extra);n++;
      XtSetValues(thisObject->w, args, n);
      viewport(0, (Screencoord)thisObject->width-1+extra,
               0, (Screencoord)thisObject->height-1+extra);
      myortho2(-0.5, (Coord)thisObject->width-0.5+extra,
               -0.5, (Coord)thisObject->height-0.5+extra);
      }
/*
 * Is this color map available?
 */
   if (stat(thisObject->u.image.filename, &buf)) {
      fprintf(stderr, "Can't find %s\n", thisObject->u.image.filename);
      c3s(thisObject->dec);
      rectangle(0,(int)thisObject->width-1,0,(int)thisObject->height-1);
      return;
      }
/*
 * Build a command line that will create this colormap image.
 */
   if (buf.st_size == 768)
      sprintf(cmd1, "rlebg -s 100 256 -l -v 0.0 1.0 255 255 255\
 | rleswap -f 0 | rleldmap -u %s | applymap", thisObject->u.image.filename);
   else
      sprintf(cmd1, "rlebg -s 100 256 -l -v 0.0 1.0 255 255 255\
 | rleswap -f 0 | rleldmap -t %s | applymap", thisObject->u.image.filename);
/*
 * Do we need to flip it?
 */
   if (thisObject->u.image.keepaspect == LEFT) {
      sprintf(cmd2, " | rleflip -l");
      strcat(cmd1, cmd2);
      }
   else if (thisObject->u.image.keepaspect == RIGHT) {
      sprintf(cmd2, " | rleflip -r");
      strcat(cmd1, cmd2);
      }
   else if (thisObject->u.image.keepaspect == FLIP) {
      sprintf(cmd2, " | rleflip -v");
      strcat(cmd1, cmd2);
      }
/*
 * Do we need to scale it?
 */
   if (((int)thisObject->width != (int)thisObject->u.image.orig_width) ||
       ((int)thisObject->height != (int)thisObject->u.image.orig_height)) {
      xscale = (double)thisObject->width/thisObject->u.image.orig_width;
      yscale = (double)thisObject->height/thisObject->u.image.orig_height;
      sprintf(cmd2, " | fant -p 0 0 -s %11.8f %11.8f", xscale, yscale);
      strcat(cmd1, cmd2);
      }
   sprintf(cmd2, " | rletoraw -r");
   strcat(cmd1, cmd2);
/*
 * Draw the drop shadow?
 */
   if (thisObject->decoration == DROP_SHADOW) {
      c3s(thisObject->dec);
      rectangle(thisObject->dec_size,(int)thisObject->width+thisObject->dec_size,
                0,thisObject->height-1);
/*
 * Fill in those little rectangles in the upper right and lower left corners.
 */
      xr = thisObject->x - desi.back.xoffset +
           thisObject->width - 1 + thisObject->dec_size;
      yu = thisObject->y - desi.back.yoffset;
      getrgb(xr, yu, cv);
      c3s(cv);
      rectangle(thisObject->width-1, thisObject->width  + thisObject->dec_size,
                thisObject->height-1,thisObject->height + thisObject->dec_size);

      xl = thisObject->x - desi.back.xoffset;
      yd = thisObject->y - desi.back.yoffset +
           thisObject->height - 1 + thisObject->dec_size;
      getrgb(xl, yd, cv);
      c3s(cv);
      rectangle(0,thisObject->dec_size,0,thisObject->dec_size);
      }
   else if (thisObject->decoration == BORDER) {
/*
 * Draw a rectangle that is "dec_size" larger than the image all the way around.
 */
      c3s(thisObject->dec);
      rectangle(0,thisObject->width+extra,0,thisObject->height+extra);
      }
/*
 * Draw the colormap.
 */
   if (!thisObject->u.cmap.imgbuf) {
      bufsiz = thisObject->width*thisObject->height*4;
      thisObject->u.cmap.imgbuf = (unsigned long *) malloc(bufsiz);
      if ((pixin = popen(cmd1, "r")) == NULL)
         fprintf(stderr, "Cannot open pipe.\n");
      for (i=0; i<bufsiz/4; i++) {
         if ((fread(&thisObject->u.cmap.imgbuf[i], 3, 1, pixin)) != 1) {
            fprintf(stderr, "Cannot read command from pipe.\n");
            exit(1);
            }
         thisObject->u.cmap.imgbuf[i] =
                               (thisObject->u.cmap.imgbuf[i] & 0xFFFFFF00) >> 8;
         }
      pclose(pixin);
      }
   lrectwrite((thisObject->decoration == BORDER?thisObject->dec_size:0),
              (thisObject->decoration == 0?0:thisObject->dec_size),
               thisObject->width-1 +
              (thisObject->decoration == BORDER?thisObject->dec_size:0),
               thisObject->height-1 +
              (thisObject->decoration == 0?0:thisObject->dec_size),
                 thisObject->u.image.imgbuf);
}


/*
 * This routine is called whenever a cmap object is resized.
 */
void
ResizeCmapCB(w, client_data, call_data)
Widget                  w;
XtPointer               client_data;
GlxDrawCallbackStruct   *call_data;
{
   GenericInfo	*thisObject;
   int		extra;

   thisObject = WidgetToObject(w);
   switch (thisObject->decoration) {
      case DROP_SHADOW:
         extra = thisObject->dec_size;
         break;
      case BORDER:
         extra = 2*thisObject->dec_size;
         break;
      default:
         extra = 0;
      }
   if (thisObject->width+extra != (int)call_data->width ||
       thisObject->height+extra != (int)call_data->height) {
      thisObject->width = (int)call_data->width-extra;
      thisObject->height= (int)call_data->height-extra;
      }
}

/*
 * This routine is called from the CmapHandler if we are creating
 * or if we are editing and a new file name was selected.
 */
static GenericInfo*
GetCmapFname(Widget w, XtPointer client_data, XmSelectionBoxCallbackStruct *call_data)
{
   GenericInfo	*thisObject;
   float	xscale, yscale;
   extern int	needtosave;
/*
 * Create space to hold this object.
 */
   if (c_or_e == CREATING) {
      if ((thisObject = (GenericInfo *)malloc(sizeof(GenericInfo))) == NULL) {
         fprintf(stderr, "ERROR -- Unable to get memory for cmap.\n");
         exit(1);
         }
      }
   else
      thisObject = UmscListGetCurrent(theList);
   XmStringGetLtoR((XmString)call_data->value, XmSTRING_DEFAULT_CHARSET,
                   &thisObject->u.cmap.filename);
   if ((int)thisObject->u.cmap.filename[strlen(thisObject->u.cmap.filename)-1] == '/') {
      free(thisObject);
      thisObject = NULL;
      return thisObject;
      }
   else {
      needtosave = 1;
      if (c_or_e == CREATING) {
         thisObject->selected = False;
         thisObject->frame = currentFrame;
         thisObject->objtype = MAPOBJ;
         thisObject->decoration = defaultInfo.decoration;
         thisObject->dec_size = defaultInfo.dec_size;
         thisObject->dec[0] = defaultInfo.dec[0];
         thisObject->dec[1] = defaultInfo.dec[1];
         thisObject->dec[2] = defaultInfo.dec[2];
         thisObject->u.cmap.imgbuf = NULL;
/*
 * I have to set the current pointer to the last item in the list
 * so that the new node is placed at the end (after the current).
 */
         UmscListSetCurrent(theList, UmscLAST);
         UmscListInsert(theList, thisObject, UmscNEXT);
         UmscListSetCurrent(theList, UmscLAST);
         }
/*
 * Set the initial size, etc.
 */
      if (XmToggleButtonGetState(right)) {
         thisObject->u.cmap.orientation = RIGHT;
         thisObject->u.cmap.orig_width = 256;
         thisObject->u.cmap.orig_height = 100;
         }
      else if (XmToggleButtonGetState(left)) {
         thisObject->u.cmap.orientation = LEFT;
         thisObject->u.cmap.orig_width = 256;
         thisObject->u.cmap.orig_height = 100;
         }
      else if (XmToggleButtonGetState(flip)) {
         thisObject->u.cmap.orientation = FLIP;
         thisObject->u.cmap.orig_width = 100;
         thisObject->u.cmap.orig_height = 256;
         }
      else {
         thisObject->u.cmap.orientation = NORMAL;
         thisObject->u.cmap.orig_width = 100;
         thisObject->u.cmap.orig_height = 256;
         }
      thisObject->width = thisObject->u.cmap.orig_width;
      thisObject->height = thisObject->u.cmap.orig_height;
      if (c_or_e == CREATING) {
         thisObject->x = desi.back.xoffset + (desi.back.width - thisObject->width)/2;
         thisObject->y = desi.back.yoffset + (desi.back.height - thisObject->height)/2;
         }
      }
   return thisObject;
}


/*
 * This routine is called when the OK button is clicked after:
 *	- creating a new cmap
 *	- editing an existing cmap
 */
static void
CmapHandlerCB(w, client_data, call_data)
Widget				w;
XtPointer			client_data;
XmSelectionBoxCallbackStruct	*call_data;
{
   Arg		args[5];
   GenericInfo	*thisObject;
   char		junk[80];
   int		lastformat, n;
   int		found;

   if (c_or_e == CREATING) {
      thisObject = GetCmapFname(w, client_data, call_data);
      if (thisObject)
         CreateCmap(thisObject);
      }
   else { /* if EDITING */
      thisObject = UmscListSetCurrent(theList, UmscFIRST);
      found = 0;
      while (!found) {
         if (!thisObject->w)
            thisObject = UmscListGetNext(theList);
         else if (XtParent(thisObject->w) != dw)
            thisObject = UmscListGetNext(theList);
         else
            found = 1;
         }
      strcpy(junk, thisObject->u.cmap.filename);
      XmStringGetLtoR(call_data->value, XmSTRING_DEFAULT_CHARSET,
                                            &thisObject->u.cmap.filename);
/*
 * They removed the file name -- delete it.
 */
      if ((int)thisObject->u.cmap.filename[strlen(thisObject->u.cmap.filename)-1] == '/')
         CutCB((Widget)0, (XtPointer)0, (char *)0);
/*
 * The cmap file name has changed.
 */
      else if (strcmp(thisObject->u.cmap.filename, junk)) {
         thisObject = GetCmapFname(w, client_data, call_data);
         /*selectedImage.filename = thisObject->u.cmap.filename;*/
         free(thisObject->u.image.imgbuf);
         thisObject->u.image.imgbuf = NULL;
         DrawCmap(thisObject);
         }
/*
 * Set the initial size, etc.
 */
      if (thisObject->u.cmap.orig_width == 100)
         lastformat = NORMAL;
      else
         lastformat = FLIP;
      if (XmToggleButtonGetState(right)) {
         thisObject->u.cmap.orientation = RIGHT;
         thisObject->u.cmap.orig_width = 256;
         thisObject->u.cmap.orig_height = 100;
         }
      else if (XmToggleButtonGetState(left)) {
         thisObject->u.cmap.orientation = LEFT;
         thisObject->u.cmap.orig_width = 256;
         thisObject->u.cmap.orig_height = 100;
         }
      else if (XmToggleButtonGetState(flip)) {
         thisObject->u.cmap.orientation = FLIP;
         thisObject->u.cmap.orig_width = 100;
         thisObject->u.cmap.orig_height = 256;
         }
      else {
         thisObject->u.cmap.orientation = NORMAL;
         thisObject->u.cmap.orig_width = 100;
         thisObject->u.cmap.orig_height = 256;
         }
      if ((thisObject->u.cmap.orig_width == 100 && lastformat == FLIP) ||
          (thisObject->u.cmap.orig_width == 256 && lastformat == NORMAL)) {
         lastformat = thisObject->width;
         thisObject->width = thisObject->height;
         thisObject->height = lastformat;
         n = 0;
         XtSetArg(args[n], XmNwidth, thisObject->width); n++;
         XtSetArg(args[n], XmNheight, thisObject->height); n++;
         XtSetValues(thisObject->w, args, n);
         }
      }
}


/*
 * This routine needs to be called when the cmap is rotated.  It is necessary
 * because the imgbuf needs to be recomputed for the rotation.
 */
static void
RotatedCB(w, client_data, call_data)
Widget				w;
XtPointer			*client_data;
XmToggleButtonCallbackStruct	*call_data;
{
   GenericInfo	*thisObject;
/*
 * Get out-a-here if we're creating -- an imgbuf does not yet exist.
 */
   if (c_or_e == CREATING)
      return;
/*
 * Free the imgbuf
 */
   thisObject = UmscListGetCurrent(theList);
   if (thisObject)
      if (thisObject->u.cmap.imgbuf) {
         free(thisObject->u.cmap.imgbuf);
         thisObject->u.cmap.imgbuf = NULL;
         }
}


/*
 * This routine creates the cmap selection box (csb).
 */
static void
CmapInfoPanelInit(void)
{
   int		n;
   Arg		args[10];
   Widget	warea,
		frame,
		rc;
/*
 * Let the user pick an cmap file.
 */
   n = 0;
   csb = XmCreateFileSelectionDialog(desi.toplevel, "csb", args, n);
   XtUnmanageChild(XmFileSelectionBoxGetChild(csb, XmDIALOG_HELP_BUTTON));
   XtAddCallback(csb, XmNokCallback, (XtCallbackProc)unmanageCB, (char *)csb);
   XtAddCallback(csb, XmNokCallback, (XtCallbackProc)BusyCursorCB, NULL);
   XtAddCallback(csb, XmNokCallback, (XtCallbackProc)CmapHandlerCB, NULL);
   XtAddCallback(csb, XmNokCallback, (XtCallbackProc)UnBusyCursorCB, NULL);
   XtAddCallback(csb, XmNcancelCallback, (XtCallbackProc)unmanageCB, (char *)csb);
/*
 * Add a rowcolumn thing to this cmap selection box.
 */
   n = 0;
   warea = XmCreateWorkArea(csb, "warea", args, n);
   XtManageChild(warea);
/*
 * Inside the rowcolumn put a frame.
 */
   n = 0;
   frame = XmCreateFrame(warea, "frame", args, n);
   XtManageChild(frame);
/*
 * Put a rowcolumn thing inside this frame.
 */
   n = 0;
   XtSetArg(args[n], XmNradioBehavior, True); n++;
   XtSetArg(args[n], XmNradioAlwaysOne, False); n++;
   rc = XmCreateRowColumn(frame, "rc", args, n);
   XtManageChild(rc);
/*
 * Inside the row column put a couple thingies.
 */
   left = XtCreateManagedWidget("left", xmToggleButtonWidgetClass, rc, args, 0);
   right= XtCreateManagedWidget("right",xmToggleButtonWidgetClass, rc, args, 0);
   flip = XtCreateManagedWidget("flip", xmToggleButtonWidgetClass, rc, args, 0);

   XtAddCallback(left, XmNvalueChangedCallback, (XtCallbackProc)RotatedCB, NULL);
   XtAddCallback(right,XmNvalueChangedCallback, (XtCallbackProc)RotatedCB, NULL);
   XtAddCallback(flip, XmNvalueChangedCallback, (XtCallbackProc)RotatedCB, NULL);
}


void
GetCmapInfoCB(w, client_data, call_data)
Widget				w;
XtPointer			client_data;
XmSelectionBoxCallbackStruct	*call_data;
{
   Arg		args[10];
   GenericInfo	*thisObject;
   int		n, found;

   c_or_e = (int)client_data;
   if (csb == NULL)
      CmapInfoPanelInit();
/*
 * If we are editing the cmap, stuff put the info of the selected object into
 * the file selection box.
 */
   if (c_or_e == EDITING) {
      dw = w;	/* this is the sm widget that was double clicked */
/*
 * Find this object in the list.
 */
      thisObject = UmscListSetCurrent(theList, UmscFIRST);
      found = 0;
      while (!found) {
         if (!thisObject->w)
            thisObject = UmscListGetNext(theList);
         else if (XtParent(thisObject->w) != dw)
            thisObject = UmscListGetNext(theList);
         else
            found = 1;
         }
/*
 * Put the colormap's filename into the colormap selection box.
 */
      n = 0;
      XtSetArg(args[n], XmNtextString,
                        XmStringCreateSimple(thisObject->u.cmap.filename)); n++;
      XtSetValues(csb, args, n);
/*
 * Set the toggle buttons to match the orientation of the object.
 */
      if (thisObject->u.cmap.orientation == RIGHT)
         XmToggleButtonSetState(right, True, False);
      else if (thisObject->u.cmap.orientation == LEFT)
         XmToggleButtonSetState(left, True, False);
      else if (thisObject->u.cmap.orientation == FLIP)
         XmToggleButtonSetState(flip, True, False);
      else if (thisObject->u.cmap.orientation == NORMAL) {
         XmToggleButtonSetState(right, False, False);
         XmToggleButtonSetState(left, False, False);
         XmToggleButtonSetState(flip, False, False);
         }
      }
   XtManageChild(csb);
}


/*
 * This routine will create a colormap object.
 */
void
CreateCmap(GenericInfo *thisObject)
{
   Arg		args[10];
   Widget	selfMoving = CreateSelfM();
   int		n;

/*
 * Get the current object.
 */
   thisObject = UmscListGetCurrent(theList);
/*
 * Create a glx widget for the image.
 */
   n = 0;
   XtSetArg(args[n], GlxNglxConfig, glxConfig); n++;
   XtSetArg(args[n], XmNwidth, thisObject->width); n++;
   XtSetArg(args[n], XmNheight, thisObject->height); n++;
   thisObject->w = GlxCreateMDraw(selfMoving, "cmap", args, n);
/*
 * Add some callbacks.
 */
   XtAddCallback(selfMoving, XtNselectCallback, (XtCallbackProc)SelectCB, (XtPointer)thisObject);
   XtAddCallback(selfMoving, XtNunselectCallback, (XtCallbackProc)UnselectCB, (XtPointer)thisObject);
   XtAddCallback(selfMoving, XtNdoubleClickCallback, (XtCallbackProc)GetCmapInfoCB, (XtPointer)EDITING);
   XtAddCallback(selfMoving, XtNendResizeCallback, (XtCallbackProc)SelfCB, (XtPointer)MAPOBJ);
   XtAddCallback(selfMoving, XtNendMoveCallback, (XtCallbackProc)SelfCB, (XtPointer)NULL);

   XtAddCallback(thisObject->w, GlxNresizeCallback, (XtCallbackProc)ResizeCmapCB, NULL);
   XtAddCallback(thisObject->w, GlxNexposeCallback, (XtCallbackProc)BusyCursorCB, NULL);
   XtAddCallback(thisObject->w, GlxNexposeCallback, (XtCallbackProc)DrawObjectCB, NULL);
   XtAddCallback(thisObject->w, GlxNexposeCallback, (XtCallbackProc)UnBusyCursorCB, NULL);
/*
 * Place the image on the background.
 */
   n = 0;
   XtSetArg(args[n], XtNxChild, thisObject->x); n++;
   XtSetArg(args[n], XtNyChild, thisObject->y); n++;
   XtSetValues(selfMoving, args, n);

   XtManageChild(thisObject->w);
   XtManageChild(selfMoving);
}
