/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/VideoObject/RCS/Pioneer6000Driver.c,v 0.20 92/09/28 12:40:00 drapeau Exp $ */
/* $Log:	Pioneer6000Driver.c,v $
 * Revision 0.20  92/09/28  12:40:00  drapeau
 * PlayFromTo() was modified to reflect the new semantics of the VideoObject
 * library.
 * 
 * Revision 0.19  92/07/30  15:27:18  drapeau
 * Several changes:
 * * Re-formatted all function declarations to conform to ANSI function
 *   prototype standards.
 * * Replaced hard-coded references to 30 frame-per-second frame rates with
 *   new definition "FrameRate".  All math is based on this definition now,
 *   allowing the driver to be used in places where the frame rate is not
 *   30 fps.
 * * Improved diagnostic messages.  Diagnostics now report the serial port
 *   being used for the command when possible.
 * 
 * Revision 0.18  91/09/30  17:07:05  lim
 * Implemented Pioneer6000Ping.
 * Renamed Power as MyPower.
 * 
 * Revision 0.17  91/08/24  17:50:10  lim
 * Implemented PrintDiagnostics.
 * 
 * Revision 0.16  91/08/24  13:38:09  lim
 * 1. Updated to use status codes in new PlayerStatus.h
 * 2. Clear Marker() removed as part of video object.
 * 
 * Revision 0.15  91/08/17  20:40:44  lim
 * 1. Removed 'stopped'. Video mute is turned off whenever a command
 * requiring video output is issued.
 * 2.  Stop and Pause implemented with interrupt capability.
 * 3. Clear Marker implemented to do as specs in videoObj.h require.
 * 
 * Revision 0.14  91/08/08  17:18:05  lim
 * 1. PlayAtSpeedDir() - try to input device-dependent speed rather than an arbitrary speed.
 *    ie use CalcSpeed before calling PlayAtSpeedDir()
 * 2. CalcSpeed() - one parameter added, 'playMode'. If playMode is 1, it is used to calculate
 *    device speed for segment play. If playMode is 0, it is used to calculate device speed for
 *    normal play.
 * 
 * Revision 0.13  91/08/07  13:16:41  lim
 * 1. Made as part of VideoLib.a. Removed references to vEdit application.
 * 2. Added instance pointer, "theObject" to all public functions. 
 * 
 * Revision 0.12  91/08/02  12:52:43  lim
 * 1. SetDefaults function changed to take in 4 parameters"
 * audio, addressingMode, addressDisplayOnOff, and displayMode
 * 2. xv_set is no longer called within driver code, but from
 * vEdit application.
 * 3. Pause is removed.
 * 
 * Revision 0.11  91/07/29  22:28:27  lim
 * Status codes updated to conform with specs.
 * 
 * Revision 0.10  91/07/27  22:18:35  lim
 * Initial revision.
 *  */

#include "Pioneer6000Driver.h"

static char pioneer6000Rcs[] = "$Header: /Source/Media/collab/VideoObject/RCS/Pioneer6000Driver.c,v 0.20 92/09/28 12:40:00 drapeau Exp $";

static int	sliderControl = 1;
static char	diagMsg[64];


/* Convert integer to Pioneer6000 code. */

void
  IntToCode(int num, char* code)
{
  char	tempString[11];
  int	i = 0;
  int	first = 1;
  int	count;
  
  while ((num != 0) || (first != 0))				    /* Convert digit by digit to code, starting from ... */
  {								    /* ... least significant and storing into tempString */
    switch (num%10)
    {
     case 0:
      strcpy(&tempString[i], "3F");
      break;
     case 1:
      strcpy(&tempString[i], "0F");
      break;
     case 2:
      strcpy(&tempString[i], "8F");
      break;
     case 3:
      strcpy(&tempString[i], "4F");
      break;
     case 4:
      strcpy(&tempString[i], "2F");
      break;
     case 5:
      strcpy(&tempString[i], "AF");
      break;
     case 6:
      strcpy(&tempString[i], "6F");
      break;
     case 7:
      strcpy(&tempString[i], "1F");
      break;
     case 8:
      strcpy(&tempString[i], "9F");
      break;
     case 9:
      strcpy(&tempString[i], "5F");
      break;
    }
    num/=10;
    first = 0;
    i+=2;
  }
  tempString[i] = '\0';
  count = strlen(tempString) - 2;
  for (i = 0; i <= count; i+=2)					    /* tempString is reversed, so we put into code in right ... */
    strncpy(&code[i], &tempString[count-i], 2);			    /* ... order */
  code[strlen(tempString)] = '\0';
  return;
}								    /* end function IntToCode */


/* Calculates base to the power of expon */
int
  MyPower(int base, int expon)
{
  int i;
  int p = 1;
  
  for (i = 1; i <= expon; i++)
    p = p*base;
  return p;
}								    /* end function MyPower */


/* Converts character "hex code" for frame address to integer */

int
  CodeToInt(char* code)
{
  int i;
  int frame = 0;
  
  for (i = 4; i > 0; i--)
  {
    if ((code[i-1] > 64) && (code[i-1] < 71))			    /* Character is between 'A' and 'F' */
      frame += (((int) code[i-1]) - 55) * MyPower(16, (4-i));
    else							    /* Character is between '0' and '9' */
      frame += (((int) code[i-1]) - 48) * MyPower(16, (4-i));
  }
  return frame;
}								    /* end function CodeToInt */


void
  Pioneer6000LoadSegmentPlayProgram(VideoObject* theObject)
{
  
  /* Program                      - Start programming mode
     0 Get                        - Put 0 in Register 0
     5 Recall                     - Recall and activate Register 5 (it stores whether address display is on/off)
     5 Arg Com                    - Use Register 5 as an argument and compare with Register 0
     19 Branch                    - If Register 0 > Register 5, branch to address 19 (Off)
     19 Branch                    - If Register 0 = Register 5, branch to address 19  "
     16 Branch                    - If Register 0 < Register 5, branch to address 16 (On)
     CharacterGeneratorEnable     - Address 16: Set character generator on
     1 Display                    - Set address display on
     6 Recall                     - Address 19: Recall and activate Register 6 (it stores start frame)
     6 Arg Com                    - Use Register 6 as an argument and compare with Register 0
     36 Branch                    - If Register 0 > Register 6, branch to address 36 (0)
     36 Branch                    - If Register 0 = Register 6, branch to address 36 (0)
     33 Branch                    - If Register 0 < Register 6, branch to address 33 (Non-zero)
     6 Arg Search                 - Address 33: Use Register 6 to search 
     7 Arg Com                    - Use Register 7 as an argument and compare with Register 0
     52 Branch                    - If Register 0 > Register 7, branch to address 52 (Play continuously)
     52 Branch                    - If Register 0 = Register 7, branch to address 52 (Play continuously)
     48 Branch                    - If Register 0 < Register 7, branch to address 48 (Play segment)
     7 Arg MultiSpeedForward      - Address 48: Use Register 7 as argument and play forward
     Halt                         - End automatic mode and return to manual mode
     Play                         - Address 52: Play
     Halt                         - End automatic mode and return to manual mode
     End                          - End programming mode and return to manual mode
     
     The code according to the lines above becomes :
     DF
     0:   3F 08
     AF 7F
     AF 0A 04
     0F 5F CF
     0F 5F CF
     0F 6F CF
     16:  E0
     0F F1
     19:  6F 7F
     6F 0A 04
     4F 6F CF
     4F 6F CF
     4F 4F CF
     33:  6F 0A F7
     36:  1F 0A 04
     AF 8F CF
     AF 8F CF
     2F 9F CF
     48:  1F 0A F2
     BF
     52:  FD F2 F7			   - This is the only way we can enter 'PLAY' into the RAM
     BF
     EF
     
     Register 5 : Address Display
     Register 6 : Start address
     Register 7 : End address
     
     */
  Pioneer6000SendCode(theObject, "DF", 0);
  Pioneer6000SendCode(theObject, "3F08", 0);
  Pioneer6000SendCode(theObject, "AF7F", 0);
  Pioneer6000SendCode(theObject, "AF0A04", 0);
  Pioneer6000SendCode(theObject, "0F5FCF", 0);
  Pioneer6000SendCode(theObject, "0F5FCF", 0);
  Pioneer6000SendCode(theObject, "0F6FCF", 0);
  Pioneer6000SendCode(theObject, "E0", 0);
  Pioneer6000SendCode(theObject, "0FF1", 0);
  Pioneer6000SendCode(theObject, "6F7F", 0);
  Pioneer6000SendCode(theObject, "6F0A04", 0);
  Pioneer6000SendCode(theObject, "4F6FCF", 0);
  Pioneer6000SendCode(theObject, "4F6FCF", 0);
  Pioneer6000SendCode(theObject, "4F4FCF", 0);
  Pioneer6000SendCode(theObject, "6F0AF7", 0);
  Pioneer6000SendCode(theObject, "1F0A04", 0);
  Pioneer6000SendCode(theObject, "AF8FCF", 0);
  Pioneer6000SendCode(theObject, "AF8FCF", 0);
  Pioneer6000SendCode(theObject, "2F9FCF", 0);
  Pioneer6000SendCode(theObject, "1F0AF2", 0);
  Pioneer6000SendCode(theObject, "BF", 0);
  Pioneer6000SendCode(theObject, "FDF2F7", 0);
  Pioneer6000SendCode(theObject, "BF", 0);
  Pioneer6000SendCode(theObject, "EF", 0);
  return;
}								    /* end function Pioneer6000LoadSegmentPlayProgram */



/* Disables remote control unit */
void
  Pioneer6000SetRCUOff(VideoObject* theObject)
{
  Pioneer6000SendCode(theObject, "13", 0);
  return;
}								    /* end function Pioneer6000SetRCUOff */



/* Sets up default values for video options */

int
  Pioneer6000SetDefaults(VideoObject*	theObject,
			 int		audio,
			 int		addressingMode,		    /* Unused */
			 int		addressDisplayOnOff,
			 int		displayMode)
{  
  
  Pioneer6000SetAudio(theObject, audio);			    /* Set audio */
  Pioneer6000SetAddressDisplay(theObject, 
			       addressDisplayOnOff, displayMode);   /* Set display */
  
  /* Loads program to perform segment play. */
  /* This is done because in manual mode, there ... */
  /* ... is no buffering of commands and setting ... */
  /* ... an autostop after a search will cause the ... */
  /* ... search to be stopped, regardless of the ... */
  /* ... status of the search. Only in automatic mode ... */
  /* ... will the commands be executed sequentially. */
  Pioneer6000LoadSegmentPlayProgram(theObject);			
  
  Pioneer6000SetRCUOff(theObject);				    /* Turn off remote control */
  return PlayerOk;
}								    /* end function Pioneer6000SetDefaults */


int
  Pioneer6000SendCode(VideoObject*	theObject,
		      char*		code,
		      int		mode)
{
  char response[20];
  int  nr;
  
  sprintf(diagMsg, "%s :\tSending %s.\n",
	  theObject->DevConfig->serialPort,
	  code);
  PrintDiagnostics(diagMsg);
  write(theObject->DevConfig->fd, code, strlen(code));
  switch (mode)
  {
   case 0:							    /* No response expected */
    return PlayerOk;
   case 1:							    /* Frame number */
    nr = read(theObject->DevConfig->fd, response, 20);
    response[nr-2] = '\0';
    sprintf(diagMsg, "%s :\tReading %s.\n",
	    theObject->DevConfig->serialPort,
	    response);
    PrintDiagnostics(diagMsg);
    return(CodeToInt(response));
   case 2:							    /* Status */
    nr = read(theObject->DevConfig->fd, response, 20);
    response[nr-2] = '\0';
    sprintf(diagMsg, "%s :\tReading %s.\n",
	    theObject->DevConfig->serialPort,
	    response);
    PrintDiagnostics(diagMsg);
    if (strcmp(response, "4C") == 0)
      return PlayerScan;
    else if (strcmp(response, "50") == 0)
      return PlayerSearch;
    else if (strcmp(response, "55") == 0)
      return PlayerStop;
    else if (strcmp(response, "E0") == 0)
      return PlayerForwardPlay;
    else if (strcmp(response, "E1") == 0)
      return PlayerForwardStep;
    else if (strcmp(response, "64") == 0)
      return PlayerForwardPlay;
    else if (strcmp(response, "65") == 0)
      return PlayerForwardStep;
    else if (strcmp(response, "66") == 0)
      return PlayerMultiStepFwd;
    else if (strcmp(response, "67") == 0)
      return PlayerMultiStepRev;
    else if (strcmp(response, "68") == 0)
      return PlayerSpinUp;
    else if (strcmp(response, "78") == 0)
      return PlayerPark;
    else if (strcmp(response, "7C") == 0)
      return PlayerEject;
    else 
      return PlayerUnknownReturnCode;
  }								    /* end switch */
}								    /* end function Pioneer6000SendCode */


/*  Player functions  */


int
  Pioneer6000Still(VideoObject* theObject)
{
  Pioneer6000SendCode(theObject, "BF", 0);			    /* Interrupt program play */
  return(Pioneer6000SendCode(theObject, "FB", 0));
}								    /* end function Pioneer6000Still */


int
  Pioneer6000Stop(VideoObject* theObject)
{
  Pioneer6000SendCode(theObject, "BF", 0);			    /* Interrupt program play */
  Pioneer6000SetVideo(theObject, FeatureOff);
  return(Pioneer6000SendCode(theObject, "FB", 0));				
}								    /* end function Pioneer6000Stop */



int 
  Pioneer6000Eject(VideoObject* theObject)
{
  Pioneer6000SendCode(theObject, "F9", 0);			    /* Eject disc */
  return(Pioneer6000SetDefaults(theObject, PlayerAudioStereo,	    /* Reset defaults */
				PlayerFrameMode, FeatureOn,
				DisplayFrame));
}								    /* Pioneer6000Eject */



int
  Pioneer6000Play(VideoObject* theObject)
{
  Pioneer6000SetVideo(theObject, FeatureOn);
  return(Pioneer6000SendCode(theObject, "FD", 0));
}								    /* end function Pioneer6000Play */


int
  Pioneer6000SetAddress(VideoObject*	theObject,
			int		add,
			int		start)
{
  char addCode[11];
  char command[20];
  
  IntToCode(add, addCode);
  if (start)
    Pioneer6000SendCode(theObject, "6F7F", 0);			    /* Set Register 6 active */
  else
    Pioneer6000SendCode(theObject, "1F7F", 0);			    /* Set Register 7 active */
  sprintf(command, "%sF5", addCode);				    /* Set address */
  return(Pioneer6000SendCode(theObject, command, 0));
}								    /* end function Pioneer6000Play */


int
  Pioneer6000PlayFromTo(VideoObject*	theObject,
			int		from,
			int		to,
			int		speed)
{
  int deviceSpeed;
  
  if ((from == to) || ((from > 0) && (to == 0)))		    /* Case a & b: search to "from" and still.  Since search... */
  {								    /* ...speed is quick, both cases are equivalent here (i.e.,... */
    return(Pioneer6000Search(theObject, from));			    /* ...there is no non-blocking search) */
  }
  if ((from == 0) && (to > 0))					    /* Case c: play from current position until address... */
  {								    /* ..."to" has been reached. */
    Pioneer6000SetAddress(theObject, to, 0);			    /* Set end address */
    Pioneer6000SetAddress(theObject, 0, 1);			    /* Continue play from current position */
    sliderControl = 0;
    deviceSpeed = Pioneer6000CalcSpeed(theObject, speed, 0);
    return(Pioneer6000PlayAtSpeedDir(theObject, deviceSpeed, Forward));
  }
  if ((from > 0) && (to > 0) && (from < to))			    /* Case d: play from "from" to "to" */
  {
    return(Pioneer6000Search(theObject, from));			    /* Search to "from" address */
    Pioneer6000SetAddress(theObject, to, 0);			    /* Set end address */
    Pioneer6000SetAddress(theObject, 0, 1);			    /* Continue play from current position */
    sliderControl = 0;
    deviceSpeed = Pioneer6000CalcSpeed(theObject, speed, 0);
    return(Pioneer6000PlayAtSpeedDir(theObject, deviceSpeed, Forward));
  }  
  return(PlayerReturnError);					    /* If this code is reached, incorrect from & to were requested */
}								    /* end function Pioneer6000PlayFromTo */


int
  Pioneer6000Step(VideoObject*	theObject,
		  enum Direction dir)
{
  Pioneer6000SetVideo(theObject, FeatureOn);
  if (dir == Forward)
    return(Pioneer6000SendCode(theObject, "F6", 0));
  else
    return(Pioneer6000SendCode(theObject, "FE", 0));
}								    /* end function Pioneer6000Step */



int
  Pioneer6000FastForward(VideoObject* theObject)
{  
  
  int i;
  
  Pioneer6000SetVideo(theObject, FeatureOn);
  Pioneer6000Still(theObject);
  for (i = 0; i < 30; i++)
    Pioneer6000SendCode(theObject, "F0", 0);
  return (Pioneer6000Play(theObject));
}								    /* end function Pioneer6000FastForward */



int
  Pioneer6000Reverse(VideoObject* theObject)
{
  int i;
  
  Pioneer6000SetVideo(theObject, FeatureOn);
  Pioneer6000Still(theObject);
  for (i = 0; i < 30; i++)
    Pioneer6000SendCode(theObject, "F8", 0);
  return (Pioneer6000Play(theObject));
}								    /* end function Pioneer6000Reverse */


int
  Pioneer6000PlayAtSpeedDir(VideoObject*	theObject,
			    int			framesPerSecond,
			    enum Direction	direction)
{
  int		timesSpeed;
  int		denom;
  char		command[10];
  char		speed[7];
  static int	lastSpeed = 0;
  
  if (framesPerSecond == lastSpeed)				    /* Was a new speed requested? */
    return(PlayerOk);						    /* No, don't bother sending a new command to the player */
  
  lastSpeed = framesPerSecond;					    /* Update last speed requested */
  if (framesPerSecond == 0)
    return(Pioneer6000Still(theObject));
  if (MyAbs(framesPerSecond) < FrameRate)			    /* Set speeds */
  {
    denom = MyRound(FrameRate, framesPerSecond);
    IntToCode(denom, speed);
    sprintf(command, "%sED", speed);
  }
  else
  {
    timesSpeed = MyRound(framesPerSecond, FrameRate);
    if (timesSpeed > 3)
      timesSpeed = 3;
    IntToCode(timesSpeed, speed);
    sprintf(command, "%sEC", speed);
  }
  Pioneer6000SetVideo(theObject, FeatureOn);
  Pioneer6000SendCode(theObject, command, 0);			    /* Send command to set speed */
  if (direction == Forward)		
    if (sliderControl)
      return(Pioneer6000SendCode(theObject, "F2", 0));		    /* Fast forward if slider control */
    else
    {
      sliderControl = 1;
      return(Pioneer6000SendCode(theObject, "CF", 0));		    /* Run program otherwise */
    }
  else								    /* Rewind */
    return(Pioneer6000SendCode(theObject, "FA", 0));
}								    /* end function Pioneer6000PlayAtSpeedDir */



/* Returns number of frames per second.
 */
int
  Pioneer6000CalcSpeed(VideoObject*	theObject,
		       int		sliderValue,
		       int		playMode)		    /* unused */
{
  int	denom;
  int	deviceSpeed;						    /* in frames per second */
  
  if (sliderValue == 0)
    return(0);
  if (MyAbs(sliderValue) < FrameRate)
  {
    denom = MyRound(FrameRate, sliderValue);			    /* Bring to the nearest rounded denom */
    deviceSpeed = (FrameRate / denom);
  }
  else
  {
    deviceSpeed = MyRound(sliderValue, FrameRate) * FrameRate;	    /* Bring to nearest whole number */
    if (deviceSpeed > (3 * FrameRate))
      deviceSpeed = FrameRate;
  }
  sprintf(diagMsg, "%s :\tDevice speed : %d.\n",
	  theObject->DevConfig->serialPort,
	  deviceSpeed);
  PrintDiagnostics(diagMsg);
  return(deviceSpeed);
}								    /* end function Pioneer6000CalcSpeed */



int
  Pioneer6000Search(VideoObject*	theObject,
		    int			to)
{
  char command[13];
  char address[11];
  
  IntToCode(to, address);
  sprintf(command, "%sF7", address);
  Pioneer6000SetVideo(theObject, FeatureOn);
  return(Pioneer6000SendCode(theObject, command, 0));
}								    /* end function Pioneer6000Search */



int
  Pioneer6000SetAddressDisplay(VideoObject*	theObject,
			       int		on,
			       int		mode)		    /* unused */
{
  if (on == FeatureOn) 
  {
    Pioneer6000SendCode(theObject, "E0", 0);			    /* Set character generator on */
    return(Pioneer6000SendCode(theObject, "0FF1", 0));
  }
  else 
    return(Pioneer6000SendCode(theObject, "3FF1", 0));
}								    /* end function Pioneer6000SetAddressDisplay */



int
  Pioneer6000SetAudio(VideoObject*	theObject,
		      int		ac)
{
  switch (ac)
  {
   case PlayerAudioMute:
    return(Pioneer6000SendCode(theObject, "A0", 0));
   case PlayerAudioLeft:
    return(Pioneer6000SendCode(theObject, "A1", 0));
   case PlayerAudioRight:
    return(Pioneer6000SendCode(theObject, "A2", 0));
   case PlayerAudioStereo:
    return(Pioneer6000SendCode(theObject, "A3", 0));
  }
}								    /* end function Pioneer6000SetAudio */


int
  Pioneer6000SetVideo(VideoObject*	theObject,
		      int		vc)
{
  if (vc==FeatureOn)
    return(Pioneer6000SendCode(theObject, "1B", 0));
  else
    return(Pioneer6000SendCode(theObject, "1C", 0));
}								    /* end function Pioneer6000SetVideo */



int
  Pioneer6000QueryFrame(VideoObject* theObject)
{  
  return(Pioneer6000SendCode(theObject, "D3", 1));
}								    /* end function Pioneer6000QueryFrame */


int
  Pioneer6000QueryStatus(VideoObject* theObject)
{  
  return(Pioneer6000SendCode(theObject, "D4", 1));
}								    /* end function Pioneer6000QueryStatus */



int 
  Pioneer6000Ping(VideoObject* theObject)
{  
  int	nr;
  char	response[3];
  
  write(theObject->DevConfig->fd, "D5", 2);			    /* Transmit extended status command */
  sprintf(diagMsg, "%s :\tSent ping.\n",
	  theObject->DevConfig->serialPort);
  PrintDiagnostics(diagMsg);
  nr = read(theObject->DevConfig->fd, response, 3);
  if (nr)
  {
    sprintf(diagMsg, "%s :\tSuccessful ping.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
  }
  
  else
  {
    sprintf(diagMsg, "%s :\tPing unsuccessful.\n",
	    theObject->DevConfig->serialPort);
    PrintDiagnostics(diagMsg);
  }
  
  return nr;
}								    /* end function Pioneer6000Ping */
