/*******************************************************************************
+
+  LEDA  3.0
+
+
+  rb_tree.h
+
+
+  Copyright (c) 1992  by  Max-Planck-Institut fuer Informatik
+  Im Stadtwald, 6600 Saarbruecken, FRG     
+  All rights reserved.
+ 
*******************************************************************************/


#ifndef LEDA_RB_TREE_H
#define LEDA_RB_TREE_H

//------------------------------------------------------------------------------
//
// rb_tree:  red black trees (leaf oriented)
//
// Walter Zimmer      (1989)
//
// Implementation as described in
// Kurt Mehlhorn : "Data Structures and Algorithms 1", section III.5.2
//
// Modifications:
// - Virtual compare function                         (S. Naeher, Aug. 1989)
// - Delete: overwrite copies of keys in inner nodes  (S. Naeher, Okt. 1989)
// 
//------------------------------------------------------------------------------

#include <LEDA/basic.h>
#include <LEDA/b_stack.h>

 
// falls TEST def. ist => tracen der aufgerufenen fkt.
// falls TEST1 def. ist => test auf schwarztiefe und print moegl. 

#define NOTEST
#define TEST1

#ifdef TEST
#define TRACE(x)  cout << form x
#define TRACE2(x,ebene)  if ( debug >= (ebene) ) cout << form  x
#define TRACE_FUNC(x) x
// extern int debug ;  muss im testprogramm definiert werden falls
// TRACE2(x,ebene) benutzt wird
#else 
#define TRACE(x) 
#define TRACE2(x,ebene) 
#define TRACE_FUNC(x) 
#endif


//--------------------------------------------------------------------
// some declarations and type definitions
//--------------------------------------------------------------------

class rb_tree;
class rb_tree_node;

typedef rb_tree_node* rb_tree_node_ptr;

typedef rb_tree_node* rb_tree_item;

typedef rb_tree_node* (*get_node_fct)(GenPtr);


typedef void (*DRAW_RB_NODE_FCT)(double,double,void*);
typedef void (*DRAW_RB_EDGE_FCT)(double,double,double,double);

struct stack_el {
  rb_tree_node* nodeptr;

  int dir;   /* 0: left
                1: right
                2: undefined
             */

  stack_el (rb_tree_node* p,int d)
     { nodeptr = p; dir = d ; }
  stack_el ()
     { nodeptr = 0; dir = 2; }
};

typedef stack_el* stack_el_ptr;
typedef b_stack<stack_el>  rb_tree_stack;

// ----------------------------------------------------------------
// class rb_tree_node 
// ----------------------------------------------------------------

class rb_tree_node
{  GenPtr e;                  // entry
   GenPtr k;                  // searchkey

   int c;                  // color
                           // for leaves red =2 , black = 3
                           // for nodes  red =0 , black = 1
                           // 4: unknown

   rb_tree_node* son[2];   // left and right son

   friend class rb_tree;
 
   public :

   GenPtr entry()   { return e; }
   GenPtr key()     { return k; }
   int col()  { return c; }

   // fuer blatter ist red =2 , black = 3
   // fuer knoten  ist red =0 , black = 1

   int is_node()   { return  col() <  2; } 
   int is_leaf()   { return  col() <  2; }
   int is_black()  { return  col() &  1; }   
   int is_red()    { return  !is_black();}   

   void gets_red_node()   { c = 0; }
   void gets_black_node() { c = 1; }
   void gets_red_leaf()   { c = 2; }
   void gets_black_leaf() { c = 3; }

   rb_tree_node(GenPtr K = 0 , GenPtr E = 0 ,  int color = 4  ,                            rb_tree_node* leftson = 0 , rb_tree_node* rightson = 0)
   {  
      k = K ;
      e = E ; 
      c = color ;
      son[0] = leftson ;
      son[1] = rightson;
   }         

   rb_tree_node(rb_tree_node_ptr p)
   {
      k = p->key();
      e = p->entry() ;
      c = p->col();
      son[0] = 0;
      son[1] = 0;
   }         

   LEDA_MEMORY(rb_tree_node)

};

 
// ----------------------------------------------------------------
// class rb_tree
// ----------------------------------------------------------------

class rb_tree
{ 
  rb_tree_node* root;
  rb_tree_node* list_ptr;   // zeigt auf des minimum im baum
  int counter;             // auf Vax genuegt int , sonst vielleicht long noetig
  rb_tree_stack st;

  rb_tree_node* find_node(GenPtr)   const;
  rb_tree_node* search(GenPtr);
  rb_tree_node* copy_tree(rb_tree_node*,rb_tree_node_ptr&) const;
  void del_tree(rb_tree_node*);
  void pop(rb_tree_node_ptr& q,int& dir);
  void push(rb_tree_node_ptr q,int dir1) { stack_el z(q,dir1) ; st.push(z) ; }
  void rotation(rb_tree_node*,rb_tree_node*,int);
  void double_rotation(rb_tree_node*,rb_tree_node*,rb_tree_node*,int,int);
  void if_root(rb_tree_node*,rb_tree_node*);

  virtual int cmp(GenPtr x, GenPtr y) const { return compare(x,y); }

  virtual void clear_key(GenPtr& x) const { Clear(x); }
  virtual void clear_inf(GenPtr& x) const { Clear(x); }

  virtual void copy_key(GenPtr& x)  const { Copy(x);  }
  virtual void copy_inf(GenPtr& x)  const { Copy(x);  }

  virtual int  int_type() const { return 0; }

  protected:

  rb_tree_node* item(void* p) const { return (rb_tree_node*)p; }

  public:

  rb_tree_node* first_item()             const  { return list_ptr; }
  rb_tree_node* next_item(rb_tree_node*) const;

  rb_tree_node* succ(rb_tree_node* p) const
  {return ( !p  ||  p->son[1] == list_ptr ) ? 0 : p->son[1]; }

  rb_tree_node* pred(rb_tree_node* p) const
  { return ( !p  ||  p == list_ptr   )  ?  0 : p->son[0] ; }

  rb_tree_node* cyclic_succ(rb_tree_node* p) const { return p ? p->son[1] : 0; }
  rb_tree_node* cyclic_pred(rb_tree_node* p) const { return p ? p->son[0] : 0; }
  rb_tree_node* min() const { return list_ptr; }
  rb_tree_node* max() const { return ( list_ptr ? list_ptr->son[0] : 0  ) ; }
  rb_tree_node* insert(GenPtr,GenPtr);
  rb_tree_node* lookup(GenPtr) const;


  void del(GenPtr);
  void del_item(rb_tree_node* p) { if (p)     del(p->k); }
  void change_inf(rb_tree_node*,GenPtr);
  void clear();

  GenPtr&  access(GenPtr);
  GenPtr   key(rb_tree_node* p)  const { return  p->key() ;  }
  GenPtr&  info(rb_tree_node* p) const { return  p->e ;      }
  GenPtr   inf(rb_tree_node* p)  const { return  p->e ;      }

  int member(GenPtr) const;
  int size()   const { return counter; } 
  int empty()  const { return root ? false : true ; }


  // priority queue operations

  rb_tree_node* find_min() const { return list_ptr; }
  void del_min()                 { if (root)  del(min()->k); }
  void decrease_key(rb_tree_node* p, GenPtr k) { insert(k,p->e); del_item(p);}



  rb_tree()  : st(32)
  {  root  = list_ptr = 0 ;
     counter = 0; 
  }

  rb_tree(const rb_tree&);
  rb_tree& operator=(const rb_tree&);

  virtual ~rb_tree() { clear(); }


  void draw(DRAW_RB_NODE_FCT, DRAW_RB_NODE_FCT, DRAW_RB_EDGE_FCT, rb_tree_node*,
          double, double, double, double, double);

  void draw(DRAW_RB_NODE_FCT, DRAW_RB_NODE_FCT, DRAW_RB_EDGE_FCT, 
          double, double, double, double);

#ifdef TEST1
  int  black_deep_test() const;
  int  deep_test(rb_tree_node*) const;
  void print() const;
  void print_tree(rb_tree_node*,int) const;
#endif

};

#endif  #RBTREE
