 /*
  * Khoros: $Id: check.c,v 1.1 1991/05/10 15:56:17 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: check.c,v 1.1 1991/05/10 15:56:17 khoros Exp $";
#endif

 /*
  * $Log: check.c,v $
 * Revision 1.1  1991/05/10  15:56:17  khoros
 * Initial revision
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 *            Copyright 1990 University of New Mexico
 *  
 *  Permission to use, copy, modify, distribute, and sell this
 *  software and its documentation for any purpose is hereby
 *  granted without fee, provided that the above copyright
 *  notice appear in all copies and that both that copyright
 *  notice and this permission notice appear in supporting docu-
 *  mentation, and that the name of UNM not be used in advertis-
 *  ing or publicity pertaining to distribution of the software
 *  without specific, written prior permission.  UNM makes no
 *  representations about the suitability of this software for
 *  any purpose.  It is provided "as is" without express or
 *  implied warranty.
 *  
 *  UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 *  INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 *  NESS, IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 *  INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 *  RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 *  ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,
 *  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 *  OF THIS SOFTWARE.
 *  
 *----------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "xvdisplay.h"


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>            Image Display Utility Routines             <<<<
   >>>>                                                       <<<<
   >>>>		    xvd_check_image()			      <<<<
   >>>>		    xvd_check_shape()			      <<<<
   >>>>		    xvd_check_clip()			      <<<<
   >>>>		    xvd_check_overlay()			      <<<<
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */



/************************************************************
*
*  MODULE NAME:  xvd_check_image
*
*      PURPOSE:  This routine is used to make sure that the
*		 input image is in proper format for viewing.
*
*		 If the image is not in a viewable format then
*		 we try to convert it to one.  xvd_check_image
*		 returns False if we were not able to use the
*		 image for viewing.
*
*		 In order for an image to be "displayable" it
*		 must have the correct format (bit or byte).
*		 The image must also not have a depth greater than
*		 the display (otherwise we must color compress to
*		 that depth).  Or if the image is to be viewed on
*		 a monchrome screen, then it may be necessary to
*		 dither it first.
*
*		 note:  xvd_check_image computes the number of colors
*			(ncolors) which represents the number of
*			elements in xcolors, etc.
*
*        INPUT:  xvdisplay - the display structure
*
*       OUTPUT:  returns the "checked" image
*
*    CALLED BY:  internal routine
*
*   WRITTEN BY:  Mark Young
*
*
*************************************************************/

int xvd_check_image(xvdisplay)

DisplayStructure *xvdisplay;
{
	Display	*display = xvdisplay->display;
	struct xvimage *image = xvdisplay->image;


	/*
	 *  If the new image is not either byte or bit, then try
	 *  to convert it.  Note, all images are converted to byte
	 *  format, even if we are on a monochrome display. This is
	 *  done since all the dithering routines take byte images.
	 */
	xvdisplay->disp_image = image;
	if ((image->data_storage_type != VFF_TYP_BIT) &&
	    (image->data_storage_type != VFF_TYP_1_BYTE))
	{
	   if (!xvd_convert_image(xvdisplay))
	      return(FALSE);
	}

	/*
	 *  If we are on a monochrome display dither the image
	 *  or if the image is a bitmap then just return the
	 *  dithered image.
	 */
	if (XDefaultDepth(display, XDefaultScreen(display)) == 1)
	{
	   if (!xvd_dither_image(xvdisplay))
	      return(FALSE);
	}
	return(TRUE);

} /* end xvd_check_image */



/************************************************************
*
*  MODULE NAME:  xvd_check_shape
*
*      PURPOSE:  This routine is used to make sure that the
*		 input shape is in proper format for being
*		 used as a shape mask.
*
*		 If the shape is not in bit form then we try
*		 convert it to bit, otherwise the bit image
*		 is left untouched and we return True.
*
*        INPUT:  xvdisplay - the display structure
*
*       OUTPUT:  returns the True 
*
*    CALLED BY:  internal routine
*
*   WRITTEN BY:  Mark Young
*
*
*************************************************************/


int xvd_check_shape(xvdisplay)

DisplayStructure *xvdisplay;
{
	
	if (xvdisplay->shape == NULL)
	{
	   xvf_error_wait("No shape image found.  Unable to complete shape \
operation on the image display window.", "xvd_check_shape", NULL);
	   return(False);
	}

	/*
	 *  Check the image to make sure that it is still a proper image
	 */
	if (xvdisplay->shape->data_storage_type != VFF_TYP_BIT)
	{
	   if (!lvconvert(xvdisplay->shape, VFF_TYP_BIT, 0, 1, 0.0, 1.0, 0))
	   {
	      xvf_error_wait("Failed to convert shape image.  Shape images \
must be of type bit (VFF_TYP_BIT).  Please try again with a different image.",
				"xvd_check_shape", NULL);
	      return(False);
	   }
	}
	return(True);

} /* end xvd_check_shape */



/************************************************************
*
*  MODULE NAME:  xvd_check_clip
*
*      PURPOSE:  This routine is used to make sure that the
*		 input clip is in proper format for being
*		 used as a clip mask.
*
*		 If the clip is not in bit form then we try
*		 convert it to bit, otherwise the bit image
*		 is left untouched and we return True.
*
*        INPUT:  xvdisplay - the display structure
*
*       OUTPUT:  returns the True 
*
*    CALLED BY:  internal routine
*
*   WRITTEN BY:  Mark Young
*
*
*************************************************************/


int xvd_check_clip(xvdisplay)

DisplayStructure *xvdisplay;
{
	if (xvdisplay->clip == NULL)
	{
	   xvf_error_wait("No clip image found.  Unable to complete clip \
operation on the image display window.", "xvd_check_clip", NULL);
	   return(False);
	}

	/*
	 *  Check the image to make sure that it is still a proper image
	 */
	if (xvdisplay->clip->data_storage_type != VFF_TYP_BIT)
	{
	   if (!lvconvert(xvdisplay->clip, VFF_TYP_BIT, 0, 1, 0.0, 1.0, 0))
	   {
	      xvf_error_wait("Failed to convert clip image.  Clip images \
must be of type bit (VFF_TYP_BIT).  Please try again with a different image.",
			"xvd_check_clip", NULL);
	      return(False);
	   }
	}
	return(True);

} /* end xvd_check_clip */



/************************************************************
*
*  MODULE NAME:  xvd_check_overlay
*
*      PURPOSE:  This routine is used to make sure that the
*		 input overlay is in proper format for being
*		 used as a overaly image.
*
*		 If the overlay image is not in bit or byte form
*		 then we try convert it to byte, otherwise the bit
*		 image is left untouched.  With the overlay image is an
*		 mask, which is used to indicate which pixels should
*		 be used.  If the image is in bit then we use the
*		 overlay as the overlay mask image.  Otherwise we
*		 create a bit overlay mask image from the overlay
*		 image.
*
*        INPUT:  xvdisplay - the display structure
*
*       OUTPUT:  returns the True 
*
*    CALLED BY:  internal routine
*
*   WRITTEN BY:  Mark Young
*
*
*************************************************************/


int xvd_check_overlay(xvdisplay)

DisplayStructure *xvdisplay;
{
	struct xvimage *copyimage();


	if (xvdisplay->overlay == NULL)
	{
	   xvf_error_wait("No overlay image found.  Unable to complete overlay \
operation on the image display window.", "xvd_check_overlay", NULL);
	   return(False);
	}

	/*
	 *  Check the image to make sure that it is still a proper image
	 */
	if (xvdisplay->overlay->data_storage_type != VFF_TYP_BIT)
	{
	   if (xvdisplay->overlay->data_storage_type != VFF_TYP_1_BYTE)
	   {
	      if (!lvconvert(xvdisplay->overlay, VFF_TYP_1_BYTE,
				0, 1, 0.0, 255.0, False))
	      {
	         xvf_error_wait("Failed to convert the overlay image.  \
Overlay images that are not of type byte or bit are automatically converted \
to byte.  Please try again with a different image.", "xvd_check_overlay",
NULL);
	         return(False);
	      }
	   }

	   if ((xvdisplay->ovmask = copyimage(xvdisplay->overlay)) == NULL)
	   {
	      xvf_error_wait("Failed to copy the overlay image in order to \
create an overlay mask.  Byte overlays must have a mask for overlaying.  \
Please try again with a different overlay image.", "xvd_check_overlay",
NULL);
	      return(False);
	   }

	   if (!lvthresh(xvdisplay->ovmask, True, 0.0, False, 0.0, 255.0))
	   {
	      xvf_error_wait("Failed to create a threshold image of the \
overlay image.  A byte overlay image must be thresholded into a type bit \
(VFF_TYP_BIT) which is used as the mask for overlaying the overlay image.  \
Please try again with a different overlay image.", "xvd_check_overlay", NULL);
	      return(False);
	   }

	   if (xvdisplay->ovmask->map_scheme != VFF_MS_NONE &&
	       xvdisplay->ovmask->maps != NULL )
	   {
	      if (!lvremmap(xvdisplay->ovmask))
	      {
	         xvf_error_wait("Failed to remove the thresholded image's map. \
The byte overlay thresholded image must be converted into a type bit \
(VFF_TYP_BIT) which is used as the mask for overlaying the overlay image.  \
Please try again with a different overlay image.", "xvd_check_overlay", NULL);
	         return(False);
	      }
	   }

	   if (!lvconvert(xvdisplay->ovmask, VFF_TYP_BIT, 0, 1, 0.0, 1.0,False))
	   {
	      xvf_error_wait("Failed to convert the overlay's image mask.  \
Overlay images that are not of type bit (VFF_TYP_BIT) must have a mask for \
overlaying.  Please try again with a different overlay image.",
"xvd_check_overlay", NULL);
	      return(False);
	   }
	}
	else
	   xvdisplay->ovmask = xvdisplay->overlay;

	return(True);

} /* end xvd_check_overlay */
