/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvdrape.c
 >>>>
 >>>>      Program Name: vdrape
 >>>>
 >>>> Date Last Updated: Mon Apr 15 22:19:24 1991 
 >>>>
 >>>>          Routines: lvdrape - the library call for vdrape
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
#define VCALL(func) if (! func ) { fprintf(stderr,"lvdrape: %s failed\n","func"); return(0); }
#include "remote_gis/lvrast.h"
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvdrape - library call for vdrape
*
* Purpose:
*    
*    Project or drape an image over a surface
*    
*    
* Input:
*    
*    img            Pointer to FLOAT image to be draped over the  sur-
*                   face.
*    
*    surface        Pointer to FLOAT image used to  represent  a  sur-
*                   face.
*    
*    xi             X coordinate  of  image  pixel  to  serve  as  the
*                   imagery reference location (integer)
*    
*    yi             Y coordinate  of  image  pixel  to  serve  as  the
*                   imagery reference location (integer)
*    
*    xs             X coordinate of surface pixel to serve as the sur-
*                   face reference location (integer)
*    
*    ys             Y coordinate of surface pixel to serve as the sur-
*                   face reference location (integer)
*    
*    
* Output:
*    
*    out            Points to the output image.
*    
*    
*
* Written By: Scott Wilson
*    
*    
****************************************************************/


/* -library_def */
int
lvdrape(image,surface,xi,yi,xs,ys,out)
struct xvimage *image,*surface,**out;
int xi,yi,xs,ys;
/* -library_def_end */

/* -library_code */
  {
    int i,j,drape_x,drape_y,ix1,ix2,ix3,ix4,iy1,iy2,iy3,iy4;
    int irows,icols,npixelsx,npixelsy,srows,scols;
    float *fi,*fc,*fd;
    float interpf(),xtmp,ytmp,xmin,xmax,ymin,ymax,fr;
    float sx1,sx2,sx3,sx4,sy1,sy2,sy3,sy4;
    double r1,r2,r3,r4,theta,cang(),h1,h2,h3,h4,rint();
    struct xvimage *drape;
    struct line right,left,top,bot;

    if (image->num_of_images != 1 ||
        image->num_data_bands != 1 ||
        image->location_type != VFF_LOC_IMPLICIT)
      {
        fprintf(stderr,"lvdrape: Input image should be a single band,");
        fprintf(stderr,"        single image, implicit location, FLOAT");
        fprintf(stderr,"        image.");
        return(0);
      }

    if (surface->num_of_images != 1 ||
        surface->num_data_bands != 1 ||
        surface->location_type != VFF_LOC_IMPLICIT) 
      {
        fprintf(stderr,"lvdrape: Surface image should be a single band,");
        fprintf(stderr,"        single image, implicit location, FLOAT");
        fprintf(stderr,"        image.");
        return(0);
      }

    VCALL(lvcost(surface,NULL,xs,ys))

    srows = surface->col_size;
    scols = surface->row_size;
    irows = image->col_size;
    icols = image->row_size;

    /* Figure out how big the output image should be by looking at
       the extent of the surface */
    fc = (float *)(surface->imagedata);
    xmin = XV_MAXFLOAT;
    xmax = -XV_MAXFLOAT;
    ymin = XV_MAXFLOAT;
    ymax = -XV_MAXFLOAT;
    for (i=0; i<srows; i++)
      {
        /* Run down the left edge */
        fr = fc[PIXEL(0,i,srows,scols)];
        theta = cang((float)(0.0-xs),(float)(i-ys));
        xtmp = fr*cos(theta);
        ytmp = fr*sin(theta);
        if (xtmp < xmin) xmin = xtmp;
        if (xtmp > xmax) xmax = xtmp;
        if (ytmp < ymin) ymin = ytmp;
        if (ytmp > ymax) ymax = ytmp;
        /* Run down the right edge */
        fr = fc[PIXEL(scols-1,i,srows,scols)];
        theta = cang((float)(scols-1-xs),(float)(i-ys));
        xtmp = fr*cos(theta);
        ytmp = fr*sin(theta);
        if (xtmp < xmin) xmin = xtmp;
        if (xtmp > xmax) xmax = xtmp;
        if (ytmp < ymin) ymin = ytmp;
        if (ytmp > ymax) ymax = ytmp;
      }
    for (i=0; i<scols; i++)
      {
        /* Run down the top edge */
        fr = fc[PIXEL(i,0,srows,scols)];
        theta = cang((float)(i-xs),(float)(0.0-ys));
        xtmp = fr*cos(theta);
        ytmp = fr*sin(theta);
        if (xtmp < xmin) xmin = xtmp;
        if (xtmp > xmax) xmax = xtmp;
        if (ytmp < ymin) ymin = ytmp;
        if (ytmp > ymax) ymax = ytmp;
        /* Run down the bottom edge */
        fr = fc[PIXEL(i,srows-1,srows,scols)];
        theta = cang((float)(i-xs),(float)(srows-1-ys));
        xtmp = fr*cos(theta);
        ytmp = fr*sin(theta);
        if (xtmp < xmin) xmin = xtmp;
        if (xtmp > xmax) xmax = xtmp;
        if (ytmp < ymin) ymin = ytmp;
        if (ytmp > ymax) ymax = ytmp;
      }

    npixelsx = (xmax-xmin)/(surface->pixsizx) + 1;
    npixelsy = (ymax-ymin)/(surface->pixsizy) + 1;

    /* Allocate the output image and get it set up */
    drape = createimage(npixelsy,npixelsx,VFF_TYP_FLOAT,1,1,
                        "vdrape image",0,0,
                        VFF_MS_NONE, VFF_MAPTYP_NONE,
                        VFF_LOC_IMPLICIT,0);
    drape->pixsizx = surface->pixsizx;
    drape->pixsizy = surface->pixsizy;
    fd = (float *)(drape->imagedata);
    bzero(fd,npixelsx*npixelsy*sizeof(float));
   
    fi = (float *)(image->imagedata);

    /* Compute location of reference pixel in output image */
    drape_x = -xmin/surface->pixsizx;
    drape_y = -ymin/surface->pixsizy;

    /* Begin warping the input image onto the surface */
    for (i=0; i<irows-1; i++)
      {
        for (j=0; j<icols-1; j++)
          {
            /* Obtain pixel indices */
            ix1 = j;    iy1 = i;
            ix2 = ix1+1; iy2 = iy1;
            ix3 = ix1+1; iy3 = iy1+1;
            ix4 = ix1;   iy4 = iy1+1;
            /* Obtain pixel angles */
            h1 = hypot((double)(ix1-xi),(double)(iy1-yi));
            if (h1 == 0.0) h1 = 1.0;
            h2 = hypot((double)(ix2-xi),(double)(iy2-yi));
            if (h2 == 0.0) h2 = 1.0;
            h3 = hypot((double)(ix3-xi),(double)(iy3-yi));
            if (h3 == 0.0) h3 = 1.0;
            h4 = hypot((double)(ix4-xi),(double)(iy4-yi));
            if (h4 == 0.0) h4 = 1.0;
            /* Interpolate surface arc length for each pixel */
            xtmp = (ix1-xi)*image->pixsizx; ytmp = (iy1-yi)*image->pixsizy;
            r1 = interpf(surface,xtmp,ytmp,xs,ys);
            xtmp = (ix2-xi)*image->pixsizx; ytmp = (iy2-yi)*image->pixsizy;
            r2 = interpf(surface,xtmp,ytmp,xs,ys);
            xtmp = (ix3-xi)*image->pixsizx; ytmp = (iy3-yi)*image->pixsizy;
            r3 = interpf(surface,xtmp,ytmp,xs,ys);
            xtmp = (ix4-xi)*image->pixsizx; ytmp = (iy4-yi)*image->pixsizy;
            r4 = interpf(surface,xtmp,ytmp,xs,ys);
            /* Compute new pixel location */
            /* Note that (ix-xi)/h1 is really cos(theta) ! */
            sx1 = drape_x + r1*((double)(ix1-xi)/h1)/surface->pixsizx;
            sx2 = drape_x + r2*((double)(ix2-xi)/h2)/surface->pixsizx;
            sx3 = drape_x + r3*((double)(ix3-xi)/h3)/surface->pixsizx;
            sx4 = drape_x + r4*((double)(ix4-xi)/h4)/surface->pixsizx;
            sy1 = drape_y + r1*((double)(iy1-yi)/h1)/surface->pixsizy;
            sy2 = drape_y + r2*((double)(iy2-yi)/h2)/surface->pixsizy;
            sy3 = drape_y + r3*((double)(iy3-yi)/h3)/surface->pixsizy;
            sy4 = drape_y + r4*((double)(iy4-yi)/h4)/surface->pixsizy;
            /* Attempt to "clean up" the values if they are on an
               edge of the image area */
            if (fabs(sx1-rint((double)sx1)) < 1e-5) sx1 = rint((double)sx1);
            if (fabs(sy1-rint((double)sy1)) < 1e-5) sy1 = rint((double)sy1);
            if (fabs(sx2-rint((double)sx2)) < 1e-5) sx2 = rint((double)sx2);
            if (fabs(sy2-rint((double)sy2)) < 1e-5) sy2 = rint((double)sy2);
            if (fabs(sx3-rint((double)sx3)) < 1e-5) sx3 = rint((double)sx3);
            if (fabs(sy3-rint((double)sy3)) < 1e-5) sy3 = rint((double)sy3);
            if (fabs(sx4-rint((double)sx4)) < 1e-5) sx4 = rint((double)sx4);
            if (fabs(sy4-rint((double)sy4)) < 1e-5) sy4 = rint((double)sy4);
            /* Perform raster fill */
            make_line(&left, ix1,iy1,ix4,iy4,sx1,sy1,sx4,sy4,
                      fi[PIXEL(ix1,iy1,irows,icols)],
                      fi[PIXEL(ix4,iy4,irows,icols)]);
            make_line(&right,ix2,iy2,ix3,iy3,sx2,sy2,sx3,sy3,
                      fi[PIXEL(ix2,iy2,irows,icols)],
                      fi[PIXEL(ix3,iy3,irows,icols)]);
            make_line(&top  ,ix1,iy1,ix2,iy2,sx1,sy1,sx2,sy2,
                      fi[PIXEL(ix1,iy1,irows,icols)],
                      fi[PIXEL(ix2,iy2,irows,icols)]);
            make_line(&bot  ,ix4,iy4,ix3,iy3,sx4,sy4,sx3,sy3,
                      fi[PIXEL(ix4,iy4,irows,icols)],
                      fi[PIXEL(ix3,iy3,irows,icols)]);
            bilin_fill_f(&left,&right,&top,&bot,fd,npixelsy,npixelsx);
          }
      }
    *out = drape;
    return(1);
  }

static float
interpf(image,dx,dy,x,y)
struct xvimage *image;
float dx,dy;
int x,y;
  {
    /* Compute the value of the image at point (x*dx,y+dy) where
       x and y specify the reference pixel location and dx and dy are float
       offsets in meters from that point. */
    float *f,f1,f2,f3,f4,flo,fhi,fl,fr,xp,yp,r;
    int nr,nc;
    int ylo,yhi,xlo,xhi;
    int x1,x2,x3,x4,y1,y2,y3,y4;

    xp = x + dx/image->pixsizx;
    yp = y + dy/image->pixsizy;

    nr = image->number_of_rows;
    nc = image->number_of_cols;

    /* Check to see if we are on the image */
    if ((xp < 0) || (xp > nc-1) ||
        (yp < 0) || (yp > nr-1))
      {
        fprintf(stderr,"lvdrape: interpf: Pixel is off-image!\n");
        return(0.0);
      }

    f = (float *)(image->imagedata);

    /* Interpolate the answer */
    if (xp == nc-1 && yp == nr-1)   /* Bottom right corner */
      {
        return(f[PIXEL(nc-1,nr-1,nr,nc)]);
      }
    else if (xp == nc-1)              /* Right edge */
      {
        ylo = floor((double)yp);
        yhi = ylo+1;
        flo = f[PIXEL(nc-1,ylo,nr,nc)];
        fhi = f[PIXEL(nc-1,yhi,nr,nc)];
        r = flo + (fhi-flo)*(yp-ylo);
        return(r);
      }
    else if (yp == nr-1)              /* Bottom edge */
      {
        xlo = floor((double)(xp));
        xhi = xlo+1;
        flo = f[PIXEL(xlo,nr-1,nr,nc)];
        fhi = f[PIXEL(xhi,nr-1,nr,nc)];
        r = flo + (fhi-flo)*(xp-xlo);
        return(r);
      }
    else                                /* In the middle (general case) */
      {
        x1 = floor((double)xp); y1 = floor((double)yp);
        x2 = x1+1; y2 = y1;
        x3 = x1+1; y3 = y1+1;
        x4 = x1  ; y4 = y1+1;
        f1 = f[PIXEL(x1,y1,nr,nc)];
        f2 = f[PIXEL(x2,y2,nr,nc)];
        f3 = f[PIXEL(x3,y3,nr,nc)];
        f4 = f[PIXEL(x4,y4,nr,nc)];
        fl = f1 + (f4-f1)*(yp-y1);
        fr = f2 + (f3-f2)*(yp-y2);
        r = fl + (fr-fl)*(xp-x1);
        return(r);
      }
  }
/* -library_code_end */
