/****
   init.c:  The initialization routines for "interpolate".
   Written by Toby Orloff
   orloff@poincare.geom.umn.edu
   July 10, 1990
****/

/*
 * Copyright (c) 1990, Geometry Supercomputer Project
 *                     University of Minnesota
 *                     1200 Washington Ave. S
 *                     Minneapolis, MN  55415
 *
 * email address: software@geom.umn.edu
 *
 * This software is copyrighted as noted above.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the authors, who may or may not act on them as they desire.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 */

#include <stdio.h>
#include "types.h"
#include "globals.h"

double atof();

PrintUsageMessage()

{
     fprintf(stderr,"Usage: interpolate <options> file1 ... filen outfile\n");
     fprintf(stderr,"  options:\n");
     fprintf(stderr,"    -LINEAR      (Linear interpolation)\n");
     fprintf(stderr,"    -CARDINAL    (Cardinal spline interpolation\n");
     fprintf(stderr,"                 (default = LINEAR)\n");
     fprintf(stderr,"    -ASCII       (ASCII data files)\n");
     fprintf(stderr,"    -BINARY      (BINARY data files)\n");
     fprintf(stderr,"                 (default = ASCII)\n");
     fprintf(stderr,"    -number nnn  (nnn = number output files.)\n");
     fprintf(stderr,"                 (default = 10)\n");
     fprintf(stderr,"    -skip nnn  (nnn = number lines to skip (ASCII))\n");
     fprintf(stderr,"               (nnn = number bytes to skip (BINARY))\n");
     fprintf(stderr,"               (default = 0)\n");
     fprintf(stderr,"    -in_time n time1 time2 ... timen\n");
     fprintf(stderr,"               (n = number of times, timei = times\n");
     fprintf(stderr,"               (default = uniform by number of input files)\n");
     fprintf(stderr,"    -out_time n time1 time2 ... timen\n");
     fprintf(stderr,"               (n = number of times, timei = times\n");
     fprintf(stderr,"               (default = uniform by number of output files)\n");
     fprintf(stderr,"    -suffix strng  (strng appended to output names)\n");
     fprintf(stderr,"    -chunk nnn  (process data nnn items at a time)\n");
     fprintf(stderr,"                (default = 1000)\n");

}

InitializeInterp()


{

  type = ASCII;
  bytes = 1;
  style = LINEAR;
  skip = 0;
  chunk = 1000;
  integer = 0;
  suffix[0] = '\0';
  number = 10;
  num_in = -1;    /* To be set later */
  num_out = -1;
}


ProcessInterpCommandLine(argc,argv)

int *argc;
char *argv[];

{
  int i,j;

  /* loop through argument list */

  for(i=0;i<*argc;++i) {

     if(strcmp("-LINEAR",argv[i]) == 0) {
        style = LINEAR;
        fprintf(stderr,"style changed to LINEAR.\n");

        /* Update i, argc and argv */

        for(j=i+1;j<*argc;++j) {
           argv[j-1] = argv[j];
          }

        *argc -= 1;
        i--;

        continue;
       }

     if(strcmp("-CARDINAL",argv[i]) == 0) {
        style = CARDINAL;
        fprintf(stderr,"style changed to CARDINAL.\n");

        /* Update i, argc and argv */

        for(j=i+1;j<*argc;++j) {
           argv[j-1] = argv[j];
          }

        *argc -= 1;
        i--;

        continue;
       }

     if(strcmp("-ASCII",argv[i]) == 0) {
        type = ASCII;
        fprintf(stderr,"file type changed to type ASCII.\n"); 

        /* Update i, argc and argv */

        for(j=i+1;j<*argc;++j) {
           argv[j-1] = argv[j];
          }

        *argc -= 1;
        i--;

        continue;
       }

     if(strcmp("-BINARY",argv[i]) == 0) {
        type = BINARY;
        fprintf(stderr,"file type changed to type BINARY.\n"); 

        /* Update i, argc and argv */

        for(j=i+1;j<*argc;++j) {
           argv[j-1] = argv[j];
          }

        *argc -= 1;
        i--;

        continue;
       }

        
     if(strcmp("-skip",argv[i]) == 0) {

        skip = atoi(argv[i+1]);
        fprintf(stderr,"skip changed to %d.\n",skip);

        /* Update i, argc and argv */

        for(j=i+2;j<*argc;++j) {
           argv[j-2] = argv[j];
          }

        *argc -= 2;
        i--;


        continue;
       }

     if(strcmp("-chunk",argv[i]) == 0) {

        chunk = atoi(argv[i+1]);
        fprintf(stderr,"chunk changed to %d.\n",chunk);

        /* Update i, argc and argv */

        for(j=i+2;j<*argc;++j) {
           argv[j-2] = argv[j];
          }

        *argc -= 2;
        i--;

        continue;
       }


     if(strcmp("-number",argv[i]) == 0) {

        if(num_out == -1) {  /* Otherwise ignore this setting */
           number = atoi(argv[i+1]);
           fprintf(stderr,"number changed to %d.\n",number);
          }

        /* Update i, argc and argv */

        for(j=i+2;j<*argc;++j) {
           argv[j-2] = argv[j];
          }

        *argc -= 2;
        i--;

        continue;
       }


     if(strcmp("-suffix",argv[i]) == 0) {

        strcpy(suffix,argv[i+1]);
        fprintf(stderr,"suffix changed to %s.\n",suffix);

        /* Update i, argc and argv */

        for(j=i+2;j<*argc;++j) {
           argv[j-2] = argv[j];
          }

        *argc -= 2;
        i--;

        continue;
       }

     if(strcmp("-in_time",argv[i]) == 0) {
 
        num_in = atoi(argv[i+1]);
        if(num_in < 2) {
           fprintf(stderr,"Impossible number of input files.\n");
           PrintUsageMessage();
           exit(0);
          }

        in_time = (double *) malloc(num_in * sizeof(double));

        for(j=0;j<num_in;++j) {
           in_time[j] = atof(argv[i+2+j]);
          }


        /* Update argv,argc,and i */

        for(j=i+2+num_in;j<*argc;++j) {
           argv[j-2-num_in] = argv[j];
          }

        *argc -= num_in + 2;
        i--;

        continue;
       }

     if(strcmp("-out_time",argv[i]) == 0) {

        num_out = atoi(argv[i+1]);
        number = num_out;
        if(num_out < 1) {
           fprintf(stderr,"Impossible number of output files.\n");
           PrintUsageMessage();
           exit(0);
          }

        out_time = (double *) malloc(num_out * sizeof(double));

        for(j=0;j<num_out;++j) {
           out_time[j] = atof(argv[i+2+j]);
          }


        /* Update argv,argc,and i */

        for(j=i+2+num_out;j<*argc;++j) {
           argv[j-2-num_out] = argv[j];
          }

        *argc -= num_out + 2;
        i--;

        continue;
       }

    }


  /* Initialize in and out times if necessary */

  if(*argc < 4) return;  /* Not enough arguments */

  if(num_in == -1)  { /* Set it now */
      num_in = (*argc) - 2;

      in_time = (double *) malloc(num_in * sizeof(double));

      /* Distribute evenly for now */

     for(i=0;i<num_in;++i)  {
        in_time[i] = (double) i;
       }
    }

  if(num_in != *argc - 2) {
       fprintf(stderr,"Incorrect number of input files.\n");
       PrintUsageMessage();
       exit(0);
      }

  if(num_out == -1) {  /* Out times need to be set */

     num_out = number;
     out_time = (double *) malloc(number * sizeof(double));

     /* Distribute evenly between in_times  */

     for(i=0;i<number;++i) {
        out_time[i] = (double) i * (double) (num_in-1) / (double) (number-1);
       }
    }
}

