/* $Id$
 *
 * gprocs.c: drawing procedures
 *
 * This file is to be #included by graphics.c with the following macros
 * set:
 *
 *   CLASS:		storage class of procedure names (static or extern)
 *   *PW:		pixwin of canvas
 *   WIDTH:		width of canvas
 *   HEIGHT:		height of canvas
 *   X_COORD:		macro to convert to canvas x coord
 *   Y_COORD:		macro to convert to canvas y coord
 *   DIMEN:		macro to convert to canvas dimension
 *   ARCRES:		arc resolution
 *   AINC_MIN:		minimum angular increment
 *   AINC_MAX:		max angular increment
 *   POINT_LIST:	list of points relative to (x,y) to plot
 *   POINT_LIST_COUNT:	number of points in POINT_LIST
 *   BATCHING:		flag indicating whether we're in batch mode
 *   ARCPROC:		name of arc drawing procedure
 *   POINTPROC:		name of point drawing procedure
 *   SEGMENTPROC:	name of segment drawing procedure
 *   ERASEPROC:		name of canvas erasing procedure
 *   BATCHPROC:		name of batching procedure
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

/*-----------------------------------------------------------------------
 * Function:	ARCPROC
 * Description:	draw an arc
 * Args  IN:	x,y: center of arc
 *		r: radius of arc
 *		a1: initial angle of arc (radians)
 *		a2: final angle of arc (radians)
 * Returns:	success
 * Author:	mbp
 * Date:	Fri Apr 13 17:00:41 1990
 * Notes:	The angle is drawn counterclockwise from a1 to a2.
 */
CLASS int
  ARCPROC(x, y, r, a1, a2)
double x, y, r, a1, a2;
{
  double a, ainc;
  int ix, iy, ir, px1, py1, px2, py2;

  /*
   * Convert to canvas coords.  Since the canvas is inverted, we
   * invert the angles and draw counterclockwise in the canvas's
   * pixel coordinate system.
   */
  ix = X_COORD(x);
  iy = Y_COORD(y);
  ir = DIMEN(r);
  a1 = -a1;
  a2 = -a2;
  while (a2 > a1) a2-=2*M_PI;

  /*
   * Compute the angular increment for this arc.
   */  
  ainc = ((double)ARCRES) / ((double)ir);
  if (ainc < AINC_MIN)
    ainc = AINC_MIN;
  else if (ainc > AINC_MAX)
    ainc = AINC_MAX;

  /*
   * Draw the arc
   */
  px1 = ix + ROUND(ir * cos(a1));
  py1 = iy + ROUND(ir * sin(a1));

  /* Always batch around arc drawing */
  if (!BATCHING) pw_batch_on(PW);

  for (a=a1-ainc; a>a2; a-=ainc) {
    px2 = ix + ROUND(ir * cos(a));
    py2 = iy + ROUND(ir * sin(a));
    pw_vector(PW, px1, py1, px2, py2, PIX_SRC, 1);
    px1 = px2;
    py1 = py2;
  }

  px2 = ix + ROUND(ir * cos(a2));
  py2 = iy + ROUND(ir * sin(a2));
  pw_vector(PW, px1, py1, px2, py2, PIX_SRC, 1);

  /* Turn batching off if it wasn't already on */
  if (!BATCHING) pw_batch_off(PW);

  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	POINTPROC
 * Description:	plot a point
 * Args  IN:	x,y: coords of point to plot
 * Returns:	success code
 * Author:	mbp
 * Date:	Fri Apr 13 16:40:16 1990
 * Notes:	POINT_LIST actually contains 2*POINT_LIST_COUNT ints
 */
CLASS int
  POINTPROC(x, y)
double x,y;
{
  int i, ix,iy;

  ix = X_COORD(x);
  iy = Y_COORD(y);

  for (i=0; i<POINT_LIST_COUNT; ++i)
    pw_put(PW,
	   ix + POINT_LIST[2*i],
	   iy + POINT_LIST[2*i+1],
	   1);
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	SEGMENTPROC
 * Description:	Plot a line segment
 * Args  IN:	x1,y1: first endpoint
 *		x2,y2: second endpoint
 * Returns:	success
 * Author:	mbp
 * Date:	Fri Apr 13 17:00:33 1990
 */
CLASS int
  SEGMENTPROC(x1, y1, x2, y2)
double x1,y1,x2,y2;
{
  pw_vector(PW,
	    X_COORD(x1), Y_COORD(y1),
	    X_COORD(x2), Y_COORD(y2),
	    PIX_SRC, 1);
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	ERASEPROC
 * Description:	erase the canvas
 * Args:	(none)
 * Returns:	success
 * Author:	mbp
 * Date:	Fri Apr 13 17:00:37 1990
 */
CLASS int
  ERASEPROC()
{
  pw_writebackground(PW, 0,0, WIDTH, HEIGHT, PIX_SRC);
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	BATCHPROC
 * Description:	turn batch mode on or off
 * Args  IN:	onoff: 0 to turn off, 1 to turn on
 * Returns:	nothing
 * Author:	mbp
 * Date:	Sat Apr 14 15:46:04 1990
 * Notes:	
 */
CLASS int
  BATCHPROC(onoff)
{
  if (BATCHING=onoff)
    pw_batch_on(PW);
  else
    pw_batch_off(PW);
}
