/* $Id: gdb.c,v 1.2 90/07/11 13:10:23 mbp Exp Locker: mbp $
 *
 * gdb.c: Module for maintaining a database of geometric objects
 */

/************************************************************************
 *		Copyright (C) 1989 by Mark B. Phillips                  *
 * 									*
 * Permission to use, copy, modify, and distribute this software and    *
 * its documentation for any purpose and without fee is hereby granted, *
 * provided that the above copyright notice appear in all copies and    *
 * that both that copyright notice and this permission notice appear in *
 * supporting documentation, and that the name of Mark B. Phillips or   *
 * the University of Maryland not be used in advertising or publicity   *
 * pertaining to distribution of the software without specific, written *
 * prior permission.  This software is provided "as is" without express *
 * or implied warranty.                                                 *
 ************************************************************************/

/*
   Changes made since $Revision: 1.2 $:
 */ 

#include <stdio.h>
#include <malloc.h>
#include "gdb.h"		/* Declarations/definitions which must be
				 * included by any program using this module */

double gdb_entity_data_dist();


/* Type for node in linked list of entities: */
typedef struct Enode_s {
  gdb_Entity_type	type;	/* type of entity    */
  char                  *name;	/* name of entity */
  char	                *data;	/* pointer to entity's data */
  struct Enode_s        *next;	/* link to next entity in list */
}               Enode;

/* Type for a node in a linked list used by blocks to store their
 * entities.  This list is called an "E-list". */
typedef struct Elnode_s {
  Enode * e;
  struct Elnode_s *next;
}               Elnode;

/* Type for node in linked list of blocks: */
typedef struct Bnode_s {
  char           *name;		/* name of block */
  Elnode         *el_head, 	/* head of E-list */
                 *el_tail;	/* tail of E-list */
  struct Bnode_s *next;		/* link to next Bnode in list */
}               Bnode;

#define YES	1
#define NO	0

#define MAXNAMELEN	80

/* First and last nodes of entity list */
static Enode   *en_head = NULL;
static Enode   *en_tail = NULL;
/* 1st and last nodes of block list. */
static Bnode   *bn_head = NULL;
static Bnode   *bn_tail = NULL;	
/* pointer to currently open block */
static Bnode   *bn_open = NULL;

Enode *last_entity_retrieved=NULL;
Bnode *last_block_retrieved=NULL;

/* Format of save file; this number is used to check compatibility
 *  format=1: GDB version 1.0 (original)
 *  format=2: GDB version 2.0
 */
static short    current_file_format = 2;	

/*
 * Macros for manipulating linked lists:
 */
/* Advance a pointer: */
#define LL_ADVANCE(ptr) (ptr)=(ptr)->next
/* Insert node (at end) of list with given head and tail: */
#define LL_INSERT(head,node,tail) {		\
  if ((tail)==NULL) (head)=(tail)=(node);	\
  else {(tail)->next=(node); (tail)=(node);}	\
}
/* Find predecessor of node in list with given head: */
#define LL_PREDECESSOR(head,pred,node,found) {			\
  found=NO;							\
  if ( (head)!=NULL ) {						\
    if ( (head)==(node) ) { (pred)=NULL; found=YES; }		\
    else for ((pred)=(head);(pred)->next!=NULL;LL_ADVANCE(pred))\
           if ( (pred)->next==(node) ) { found=YES; break; }	\
  }								\
}
/* Delete successor of pred in a list with given head and tail: */
#define LL_DELETE_SUCCESSOR(head,pred,tail) {			\
  if ( (pred)==NULL ) {						\
    if ( (tail)==(head) ) (head)=(tail)=(head)->next;		\
    else LL_ADVANCE(head); }					\
  else { if ( (pred)->next==(tail) )  (tail)=(pred);		\
	 LL_ADVANCE((pred)->next); }				\
}

/*
 * The following variables and macros are to make life easier while writing
 * code to read/write files.
 */
static int      nitems;		/* number of items to read/write */
static int      nrw;		/* number of items actually read/written */
#define READIT(a,b,c,d) {nrw=fread(a,b,nitems=(c),d);\
			 if (nrw!=nitems) goto io_abort;}
#define READSTRING(a,b,c) {read_string(a,b,c); if(!*(c)) goto io_abort;}
#define WRITEIT(a,b,c,d) {nrw=fwrite(a,b,nitems=(c),d);\
			 if (nrw!=nitems) goto io_abort;}
#define WRITESTRING(a,b,c) {write_string(a,b,c); if(!*(c)) goto io_abort;}
#define END_OF_IO_PROC(f) {*f=YES;return;io_abort:*f=NO;return;}
#define IOABORT goto io_abort

/*
 ***********************************************************************
 *		  P U B L I C    P R O C E D U R E S                   *
 ***********************************************************************
 */

/*-----------------------------------------------------------------------
 * Function:     gdb_add_entity
 * Description:  adds an entity to the database
 * Arguments IN: type: type number of new entity
 *		 data: pointer to data of new entity
 *		 name: name of new entity
 * 	    OUT: *entity: gets set to handle of new entity
 * Returns:      nothing
 * Notes:	 *entity=NULL indicates failure
 */
gdb_add_entity(type, data, name, entity )
     gdb_Entity_type    type;
     char           *data, *name;
     gdb_Entity  *entity;
{
  Enode          *enew;
  /* Create the new node */
  enew = (Enode *) malloc(sizeof(Enode));
  if (enew == NULL) {*entity=NULL;return;}
  enew->type = type;
  enew->next = NULL;
  /* Install name */
  enew->name = malloc((unsigned)((strlen(name)+1)*sizeof(char)));
  if (enew->name == NULL) {*entity=NULL;return;}
  strcpy(enew->name, name);
  /* Install data */
  enew->data = malloc((unsigned)gdb_entity_data_size(type));
  if (enew->data == NULL) {*entity=NULL;return;}
  gdb_entity_data_copy(enew->data, data, type);
  /* Insert new node in list */
  LL_INSERT(en_head, enew, en_tail);
  *entity = (gdb_Entity) enew;
  /* If a block is currently open, add this entity to it */
  if (bn_open!=NULL) gdb_add_to_block( (gdb_Block)bn_open,
				       (gdb_Entity)enew );
}

/*-----------------------------------------------------------------------
 * Function:     gdb_delete_entity
 * Description:  deletes an entity from the database
 * Arguments IN: entity: handle of entity to be deleted
 * Returns:      nothing
 * Notes:	 Automatically removes the entity from any blocks
 *		 which reference it.
 */
gdb_delete_entity( entity )
     gdb_Entity   entity;
{
  Enode *pred;
  Elnode *elpred, *eln;
  Bnode *bn;
  int found;
  
  /* First remove any references to this entity by blocks */
  do {
    find_block_reference((Enode*)entity, &bn, &elpred);
    if (bn!=NULL) {		/* we have a reference; delete it */
      eln = (elpred==NULL) ? bn->el_head : elpred->next;
      LL_DELETE_SUCCESSOR((bn->el_head),elpred,(bn->el_tail));
      free( (char*)eln );
    }
  } while (bn!=NULL);      
  /* Now delete the entity itself; search for it... */
  LL_PREDECESSOR(en_head,pred,((Enode*)entity),found);
  if (found) {					/* ..if we found it,  */
    LL_DELETE_SUCCESSOR(en_head,pred,en_tail);	/* delete from list,  */
    free_enode((Enode*)entity);			/* and free up space. */
  }
}

/*-----------------------------------------------------------------------
 * Function:     gdb_entity_name
 * Description:  returns the name of an entity
 * Arguments IN: entity: the entity's handle
 * Returns:      a pointer to the entity's name string
 * Notes:        this pointer points to the actual storage location
 *		   used in the database; the contents should NOT be
 *		   changed.
 */
char     *  gdb_entity_name(entity)
     gdb_Entity   entity;
{
  if (entity == NULL)
    return (NULL);
  else
    return(((Enode*)entity)->name);
}

/*-
 * Function:     gdb_entity_data
 * Description:  returns a pointer to an entity's data
 * Arguments IN: entity: the entity's handle
 * Returns:      pointer to entity's data
 * Notes:        The pointer is to the actual storage in the
 *		   database.
 */
char * gdb_entity_data(entity)
     gdb_Entity entity;
{
  return( ((Enode*)entity)->data );
}

/*-----------------------------------------------------------------------
 * Function:     gdb_entity_type
 * Description:  returns the type of an entity
 * Arguments IN: entity: the entity's handle
 * Returns:      the type number
 */
gdb_Entity_type gdb_entity_type(entity)
     gdb_Entity entity;
{
  return( ((Enode*)entity)->type );
}

/*-----------------------------------------------------------------------
 * Function:     gdb_retrieve_entity
 * Description:  retrieve an entity's handle from the database
 * Arguments IN: tmask: mask telling what types of entities to look for
 * 		 search_type: type of search to do
 *		 *search_spec: what to look for
 *          OUT: *entity: handle of retrieved entity
 * Returns:      nothing
 */
gdb_retrieve_entity(tmask, search_type, search_spec, entity )
     gdb_Entity_type      tmask;
     gdb_Search_type search_type;
     char            *search_spec;
     gdb_Entity   *entity;
{
  switch (search_type) {
  case GDB_SEQUENTIAL:
    retrieve_enode_sequential( (Enode**)entity,
			      (gdb_Sequence_spec*)search_spec, tmask);
    break;
  case GDB_NAME:
    retrieve_enode_name( (Enode**)entity, search_spec, tmask );
    break;
  case GDB_DISTANCE:
    retrieve_enode_distance((Enode **) entity,
		((gdb_Distance_spec *) search_spec)->seek_data,
		((gdb_Distance_spec *) search_spec)->seek_type,
		((gdb_Distance_spec *) search_spec)->tolerance,
	       &(((gdb_Distance_spec *) search_spec)->sequence_spec),
		tmask	);
    break;
  }
  if (*entity!=NULL) last_entity_retrieved = (Enode*)*entity;
}

/*-----------------------------------------------------------------------
 * Function:     gdb_create_block
 * Description:  Creates a new block
 * Arguments IN: *name: the name to use for the block
 *		 leave_open: flag telling wether to leave block open
 *          OUT: *block: the new block's handle
 * Returns:      nothing
 * Notes:        The new block contains no entities.  
 *	         *block=NULL indicates failure.
 */
gdb_create_block( name, block, leave_open )
     char	   *name;
     gdb_Block *block;
     int	   leave_open;
{
  Bnode *bnew;
  bnew = (Bnode*) malloc(sizeof(Bnode)); /* Create the new node */
  if (bnew==NULL) {*block=NULL;return;}
  bnew->el_head = bnew->el_tail = NULL; /* Initialize its data */
  bnew->name = malloc((unsigned)((strlen(name)+1)*sizeof(char)));
  strcpy(bnew->name, name);
  bnew->next = NULL;
  LL_INSERT(bn_head, bnew, bn_tail); /* Insert node in list */
  *block = (gdb_Block)bnew;
  if (leave_open) gdb_open_block((gdb_Block)bnew);
}

/*-----------------------------------------------------------------------
 * Function:     gdb_delete_block
 * Description:  Deletes a block
 * Arguments IN: block: the handle of the block to be deleted
 * Returns:      nothing
 * Notes:        Does not delete the block's entities
 */
gdb_delete_block( block )
     gdb_Block block;
{
  Bnode *pred;
  int found;
  if (block==NULL) return;
  free_elist( ((Bnode*)block)->el_head );
  LL_PREDECESSOR(bn_head,pred,(Bnode*)block,found);
  if (!found) return;
  if ( (Bnode*)block == bn_open ) bn_open=NULL;
  LL_DELETE_SUCCESSOR(bn_head,pred,bn_tail);
  free( (char*)block );
}

/*-----------------------------------------------------------------------
 * Function:     gdb_open_block
 * Description:  Opens a block
 * Arguments IN: block: the handle of the block to be opened
 * Returns:      nothing
 * Notes:        For a block to be "open" means that all newly added
 *		 entities are automatically appended to it.  A block
 *		 must already exist in order too be opened.
 */
gdb_open_block( block )
     gdb_Block block;
{
  bn_open = (Bnode*)block;
}

/*-
 * Function:     gdb_close_block
 * Description:  Closes any currently open block
 * Arguments:    none
 * Returns:      nothing
 * Notes:        Has no effect if there is no block currently open
 */
gdb_close_block()
{
  bn_open = NULL;
}

/*-----------------------------------------------------------------------
 * Function:     gdb_add_to_block
 * Description:  Adds an entity to a block
 * Arguments IN: block: the handle of the block to add to
 *		 entity: the handle of the entity to be added
 * Returns:      nothing
 * Notes:        The block need not be open.  If the entity is already
 *		 in the block, it is not added.
 */
gdb_add_to_block( block, entity )
     gdb_Block block;
     gdb_Entity entity;
{
  Elnode *elpred, *elnew;
  int found;
  
  if ( (block==NULL) || (entity==NULL) ) return;
  /* First check to see if the entity is already in the block */
  elist_predecessor( ((Bnode*)block)->el_head, &elpred,
		     ((Enode*)entity), &found );
  if (found) return;
  /* Then create the new node for the E-list, set its data, and insert it */
  elnew = (Elnode*) malloc(sizeof(Elnode));
  elnew->e = (Enode*)entity;
  elnew->next = NULL;
  LL_INSERT( (((Bnode*)block)->el_head), elnew, (((Bnode*)block)->el_tail) );
}		       

/*-----------------------------------------------------------------------
 * Function:     gdb_subtract_from_block
 * Description:  Removes and entity from a block
 * Arguments IN: block: block to remove from
 *		 entity: handle of entity to be removed
 * Returns:      nothing
 * Notes:        The entity is not deleted from the database.
 */
gdb_subtract_from_block( block, entity )
     gdb_Entity entity;
     gdb_Block block;
{
  Elnode *elpred, *eln;
  int found;
  
  if ( (entity==NULL) || (block==NULL) ) return;
  elist_predecessor( ((Bnode*)block)->el_head, &elpred,
		     ((Enode*)entity), &found );
  if (found) {
    eln = (elpred==NULL) ? ((Bnode*)block)->el_head : elpred->next; 
    LL_DELETE_SUCCESSOR( ((Bnode*)block)->el_head, elpred,
			((Bnode*)block)->el_tail );
    free((char*)eln);
  }
}

/*-----------------------------------------------------------------------
 * Function:     gdb_empty_block
 * Description:  Removes all entities from a block
 * Arguments IN: block: the handle of the block to be emptied
 * Returns:      nothing
 * Notes:        The block, which is not deleted from the database,
 *		 is left empty.
 */
gdb_empty_block( block )
     gdb_Block block;
{
  /* First, we free up all data in the E-list */
  free_elist( ((Bnode*)block)->el_head );
  /* Then we reset the block's E-list pointers to NULL */
  ((Bnode*)block)->el_head = ((Bnode*)block)->el_tail = NULL;
}

/*-----------------------------------------------------------------------
 * Function:     gdb_block_entity_count
 * Description:  count the number of entities in a block
 * Arguments IN: block: the block's handle
 * Returns:      the number of entities
 * Notes:        returns 0 if there are no entities, or if handle
 *		 is bad.
 */
gdb_block_entity_count( block )
     gdb_Block block;
{
  Elnode *eln;
  Bnode *bn;
  int count;
  
  bn = (Bnode*)block;
  if (bn==NULL) return(0);
  for ( count=0, eln=bn->el_head; eln!=NULL; LL_ADVANCE(eln) )
    ++count;
  return(count);
}

/*-----------------------------------------------------------------------
 * Function:     gdb_block_entity_list
 * Description:  stores the handles of a block's entities in an array
 * Arguments IN: block: handle of block
 *		 harray: ptr to array of entity handles in which to
 *		   store handles
 * Returns:      nothing
 * Notes:        The calling procedure is responsible for making sure
 *		 the harray is large enough; use gdb_block_entity_count
 *		 to find out how large is large enough.
 */
gdb_block_entity_list( block, harray )
     gdb_Block block;
     gdb_Entity harray[];
{
  Elnode *eln;
  Bnode *bn;
  int i;
  
  bn = (Bnode*)block;
  if (bn==NULL) return;
  for ( i=0,eln=bn->el_head; eln!=NULL; LL_ADVANCE(eln),++i )
    harray[i] = (gdb_Entity)(eln->e);
}

/*-----------------------------------------------------------------------
 * Function:     gdb_delete_block_entities
 * Description:  Deletes a block's entities
 * Arguments IN: block: the handle of the block
 * Returns:      nothing
 * Notes:        This deletes the block's entities from the database;
 * 		 It does not delete the block itself, which is left
 *		 empty.
 */
gdb_delete_block_entities( block )
     gdb_Block block;
{
  Elnode *eln;
  if (block==NULL) return;
  for (eln = ((Bnode*)block)->el_head;  eln!=NULL;  LL_ADVANCE(eln) )
    gdb_delete_entity( (gdb_Block)(eln->e) );
}

/*-----------------------------------------------------------------------
 * Function:     gdb_block_name
 * Description:  returns the name of a block
 * Arguments IN: block: the block's handle
 * Returns:      a pointer to the block's name string
 * Notes:        this points to the actual storage; don't change it
 */
char * gdb_block_name( block )
     gdb_Block block;
{
  return( ((Bnode*)block)->name );
}

/*-----------------------------------------------------------------------
 * Function:     gdb_retrieve_block
 * Description:  retrieve's a block's handle
 * Arguments IN: search_type: type of search to do
 *		 *search_spec: what to look for
 *          OUT: *block: handle of retrieved block
 * Returns:      nothing
 */
gdb_retrieve_block( search_type, search_spec, block )
     gdb_Search_type search_type;
     char *search_spec;
     gdb_Block *block;
{
  switch (search_type) {
  case GDB_SEQUENTIAL:
    retrieve_bnode_sequential( (Bnode**)block,
			      (gdb_Sequence_spec *) search_spec );
    break;
  case GDB_NAME:
    retrieve_bnode_name( (Bnode**)block, search_spec );
    break;
  case GDB_DISTANCE:
    retrieve_bnode_distance( (Bnode**)block,
		((gdb_Distance_spec *) search_spec)->seek_data,
		((gdb_Distance_spec *) search_spec)->seek_type,
		((gdb_Distance_spec *) search_spec)->tolerance,
	       &(((gdb_Distance_spec *) search_spec)->sequence_spec),
	       &(((gdb_Distance_spec *) search_spec)->found_entity) );
      break;
  }
  if (*block!=NULL) last_block_retrieved = (Bnode*)*block;
}

/*-----------------------------------------------------------------------
 * Function:     gdb_delete_all
 * Description:  completely clears the database
 * Arguments:    none
 * Returns:      nothing
 */
gdb_delete_all()
{
  /* First reset all global params to original state */
  last_entity_retrieved = NULL;
  last_block_retrieved = NULL;
  bn_open=NULL;
  /* Delete blocks first */
  while (bn_head!=NULL) gdb_delete_block((gdb_Block)bn_head);
  /* Then delete the entities */
  while (en_head!=NULL) gdb_delete_entity((gdb_Entity)en_head);
}

/*-----------------------------------------------------------------------
 * Function:     gdb_save
 * Description:  saves the database in a file
 * Arguments IN: *fname: name of file to write to
 *          OUT: *success: success indicator
 * Returns:      nothing
 * Notes:        If the file already exists, it is overwritten.
 *		 The contents of the database are unchanged.
 */
gdb_save( fname, success )
     char *fname;
     int *success;
{
  FILE *fp;
  int n_entities, n_blocks;
  Enode *en;
  Bnode *bn;

  fp = fopen(fname, "w");
  if (fp == NULL) {*success = NO;return;}
  WRITEIT(&current_file_format, sizeof(short), 1, fp);
  n_entities = entity_count();
  WRITEIT(&n_entities, sizeof(int), 1, fp);
  for ( en=en_head; en!=NULL; LL_ADVANCE(en) )
    { write_entity( en, fp, success ); if (!*success) IOABORT; }
  n_blocks = block_count();
  WRITEIT(&n_blocks, sizeof(int), 1, fp);
  for ( bn=bn_head; bn!=NULL; LL_ADVANCE(bn) )
    { write_block( bn, fp, success ); if (!*success) IOABORT; }

  *success=YES;
  fclose(fp);
  return;
 io_abort:
  *success=NO;
  fclose(fp);
  return;
}

/*-----------------------------------------------------------------------
 * Function:     gdb_load
 * Description:  loads a database from a file written with gdb_save
 * Arguments IN: *fname: name of file to read
 *          OUT: *success: success indicator
 * Returns:      nothing
 * Notes:        The current database will be cleared before reading
 *		 the file; this clearing is not done, however, until
 *		 after the file has been successfully opened and the
 *		 version number checked. This prevents loss of the
 *		 database if an incorrect or non-existent file name
 *		 is given.
 */
gdb_load( fname, success )
char *fname;
int *success;
{
  FILE *fp;
  short file_format;
  int n_entities, n_blocks;

  fp = fopen(fname, "r");
  if (fp == NULL) {*success=NO; return;}
  READIT(&file_format, sizeof(short), 1, fp);
  if (file_format != current_file_format) IOABORT;
  gdb_delete_all();
  READIT(&n_entities, sizeof(int), 1, fp);
  while (n_entities--) {
    read_entity( fp, success ); if (!*success) IOABORT;
  }
  READIT(&n_blocks, sizeof(int), 1, fp);
  while (n_blocks--) {
    read_block( fp, success ); if (!*success) IOABORT;
  }

  *success=YES;
  fclose(fp);
  return;
 io_abort:
  *success=NO;
  fclose(fp);
  return;
}

/*-----------------------------------------------------------------------
 * Function:     gdb_generate_unique_name
 * Description:  generates a name different from all current names
 *		 in database
 * Arguments IN: prefix: prefix to use in name
 *          OUT: name: generated name (including prefix)
 * Returns:      name
 * Notes:        If prefix is NULL or the null string (""), no
 *		 prefix is used and the resulting name will
 *		 consist only of digits.
 */
char * gdb_generate_unique_name( prefix, name )
     char *prefix, *name;
{
  int n, max, prefix_len;
  Enode *en;
  Bnode *bn;

  if (prefix==NULL) {
    strcpy(name,"");
    prefix_len = 0;
  }
  else {
    strcpy(name,prefix);
    prefix_len = strlen(prefix);
  }

  /* Find largest n for which there is either an entity or a
   * block named "<prefix>n"  */

  /* First check the entities: */
  for ( max=0,en=en_head; en!=NULL; LL_ADVANCE(en) )
    if (strncmp(en->name,name,prefix_len)==0)
      if (is_integer_string((en->name)+prefix_len)) {
	n=atoi((en->name)+prefix_len);
	max = n > max ? n : max;
      }
  
  /* Then check the blocks: */
  for ( bn=bn_head; bn!=NULL; LL_ADVANCE(bn) )
    if (strncmp(bn->name,name,prefix_len)==0)
      if (is_integer_string((bn->name)+prefix_len)) {
	n=atoi((bn->name)+prefix_len);
	max = n > max ? n : max;
      }

  /* Set the name to be "<prefix>m", where m = max+1 */

  if (prefix==NULL) sprintf( name, "%1d", max+1 );
  else sprintf( name, "%s%1d", prefix, max+1 );

  return(name);
}

/*
 ***********************************************************************
 *		 P R I V A T E    P R O C E D U R E S                  *
 ***********************************************************************
 */

/*-----------------------------------------------------------------------
 * pFunction:    free_enode
 * Description:  free up space used by an Enode
 * Arguments IN: e: pointer to node to be freed
 * Returns:      nothing
 */
static free_enode(e)
     Enode *e;
{
  if (e==NULL) return;
  free(e->name);
  free(e->data);
  free((char*)e);
}

/*-----------------------------------------------------------------------
 * pFunction:    elist_predecessor
 * Description:  find a reference to an entity in an E-list
 * Arguments IN: head: ptr to head of E-list to search
 *		 entity: the handle of the entity to search for
 *          OUT: *pred: pointer to node in E-list preceeding the
 *		   one which references entity
 *		 *found: flag indicating whether a reference was found
 * Returns:      nothing
 * Notes:        If *found=YES and *pred=NULL, entity is referenced by
 *		 the first node (head) in E-list.
 */
static elist_predecessor( head, pred, entity, found )
     Elnode *head, **pred;
     Enode *entity;
     int *found;
{
  *found=NO;
  if ( (head==NULL) || (entity==NULL) ) return;
  if (head->e==entity) {
    *found=YES;
    *pred=NULL;
    return;
  }
  *pred=head;
  while ( (*pred)->next != NULL )
    if ( (*pred)->next->e==entity ) {*found=YES;return;}
    else LL_ADVANCE(*pred);
  /* If we got this far, it isn't there; set *pred=NULL to avoid */
  /* confusion. */
  *pred=NULL;
}

/*-
 * pFunction:    find_block_reference
 * Description:  locates references to an enity by a block
 * Arguments IN: entity: handle of entity to look for
 *          OUT: **block: handle of block found to reference enitity
 *		 **elpred: predecessor node of enitity in block's E-list
 * Returns:      nothing
 * Notes:        *block=NULL means entity is not in any block
 *		 *pred=NULL but *block nonNULL means entity is first
 *		 in block's E-list
 */
static find_block_reference(entity, block, elpred)
     Enode *entity;
     Bnode **block;
     Elnode **elpred;
{
  int found;
  
  if (entity==NULL) {*block=NULL;return;}
  found = NO;
  *block=bn_head;
  while ( (*block!=NULL) && (!found) ) {
    /* Check this block for entity */
    elist_predecessor( (*block)->el_head, elpred, entity, &found );
    if (!found) LL_ADVANCE(*block);
  }
}

/*-----------------------------------------------------------------------
 * pFunction:    free_elist
 * Description:  free up space in an E-list
 * Arguments IN: eln: pointer to first node in list
 * Returns:      nothing
 */
static free_elist( eln )
     Elnode *eln;
{
  Elnode *eln_next;
  
  while (eln!=NULL) {
    eln_next = eln->next;
    free((char*)eln);
    eln = eln_next;
  }
}

/*-----------------------------------------------------------------------
 * pFunction:    retrieve_enode_sequential
 * Description:  search sequentially for an entity's handle
 * Arguments IN: *spec: specifies where to begin search
 *		 tmask: tells which entity types to look for
 *          OUT: *entity: handle of retrieved entity
 *		 *spec: always set to GDB_NEXT
 * Returns:      nothing
 * Notes:        *spec=GDB_FIRST==> get FIRST entity of matching type in
 *		 database; *spec=GDB_NEXT==> get NEXT entity of matching
 *		 type after the last entity retrieved.
 */
static retrieve_enode_sequential( entity, spec, tmask )
     Enode **entity;
     gdb_Sequence_spec *spec;
     gdb_Entity_type tmask;
{
  if ( (*spec==GDB_NEXT) && (last_entity_retrieved!=NULL) )
    *entity = last_entity_retrieved->next;
  else
    *entity = en_head;
  while (*entity!=NULL)
    if ( ((*entity)->type) & tmask ) break;
    else LL_ADVANCE(*entity);
  *spec = GDB_NEXT;
}

/*-----------------------------------------------------------------------
 * pFunction:    retrieve_bnode_sequential
 * Description:  search sequentially for a block's handle
 * Arguments IN: *spec: specifies where to begin search
 *          OUT: *block: handle of retrieved block
 *		 *spec: always set to GDB_NEXT
 * Returns:      nothing
 * Notes:        *spec=GDB_FIRST==> get FIRST block in database;
 *		 *spec=GDB_NEXT==> get NEXT block after the last
 *		 block retrieved.
 */
static retrieve_bnode_sequential( block, spec )
     Bnode **block;
     gdb_Sequence_spec *spec;
{
  if ( (*spec==GDB_NEXT) && (last_block_retrieved!=NULL) )
    *block = last_block_retrieved->next;
  else
    *block = bn_head;
  *spec = GDB_NEXT;
}

/*-----------------------------------------------------------------------
 * pFunction:    gdb_retrieve_enode_name
 * Description:  retrieve an entity's handle from its name
 * Arguments IN: *name: the name of an entity to search for
 *          OUT: *entity: the entity's handle
 * Returns:      nothing
 * Notes:	 Does not check entity's type
 */
static retrieve_enode_name( entity, name, tmask )
     Enode **entity;
     char *name;
     gdb_Entity_type tmask;
{
  *entity=en_head;
  while (*entity!=NULL)
    if (   (strcmp((*entity)->name,name)==0)
	&& (((*entity)->type)&tmask        ) ) break;
    else LL_ADVANCE(*entity);
}

/*-----------------------------------------------------------------------
 * pFunction:    retrieve_bnode_name
 * Description:  retrieve a block's handle from its name
 * Arguments IN: *name: the name of a block to search for
 *          OUT: *block: the block's handle
 * Returns:      nothing
 */
static retrieve_bnode_name( block, name )
     Bnode **block;
     char *name;
{
  *block=bn_head;
  while (*block!=NULL)
    if (strcmp((*block)->name,name)==0) break;
    else LL_ADVANCE(*block);
}

/*-----------------------------------------------------------------------
 * pFunction:    gdb_retrieve_enode_distance
 * Description:  retrieve an entity based on its distance from something
 * Arguments IN: seek_data: data of entity to compute distance to
 *		 seek_type: type of data given in seek_data
 *		 tolerance: maximum distance to accept
 *		 *spec: tells where to start searching
 *		 tmask: tells what type of entities to look for
 *          OUT: *entity: handle of found entity
 *		 *spec: set to GDB_NEXT on all calls
 * Returns:      nothing
 * Notes:        
 */
static retrieve_enode_distance( entity, seek_data, seek_type,
				   tolerance, spec, tmask )
     Enode **entity;
     char * seek_data;
     gdb_Entity_type seek_type;
     double tolerance;
     gdb_Sequence_spec *spec;
     gdb_Entity_type tmask;
{
  if ( (*spec==GDB_NEXT) && (last_entity_retrieved!=NULL) )
    *entity = last_entity_retrieved->next;
  else
    *entity = en_head;
  while (*entity!=NULL) {
    if ( ((*entity)->type) & tmask ) {
      if ( dist_acceptable( seek_data, seek_type, (char*)((*entity)->data),
			    (*entity)->type, tolerance ) )
	break;
    }
    LL_ADVANCE(*entity);
  }
  *spec = GDB_NEXT;
}

/*-----------------------------------------------------------------------
 * pFunction:    retrieve_bnode_distance
 * Description:  retrieve a block based on its distance from something
 * Arguments IN: seek_data: data of entity to compute distance to
 *		 seek_type: type of data given in seek_data
 *		 tolerance: maximum distance to accept
 *		 *spec: tells where to start searching
 *		 tmask: tells what type of entities to look for
 *          OUT: *block: handle of found block
 *		 *found_entity: handle of entity in block which caused
 *		    the match; this entity is within tolerance of seek_data
 *		 *spec: set to GDB_NEXT on all calls
 * Returns:      nothing
 * Notes:        
 */
static retrieve_bnode_distance( block, seek_data, seek_type,
				   tolerance, spec, found_entity )
     Bnode **block;
     char *seek_data;
     gdb_Entity_type seek_type;
     double tolerance;
     gdb_Sequence_spec *spec;
     Enode **found_entity;
{
  Elnode *eln;
  int found;
  
  if ( (*spec==GDB_NEXT) && (last_block_retrieved!=NULL) )
    *block = last_block_retrieved->next;
  else
    *block = bn_head;
  found=NO;
  while ( (*block!=NULL) && (!found) ) {
    /* search this block's E-list */
    eln=(*block)->el_head;
    while ( (eln!=NULL) && (!found) ) {
      found = dist_acceptable( seek_data, seek_type,
			      (char*)(eln->e->data), eln->e->type, tolerance );
      if (!found) LL_ADVANCE(eln);
    }
    if (!found) LL_ADVANCE(*block);
  }
  if (found) *found_entity=eln->e;
  *spec = GDB_NEXT;
}

/*-----------------------------------------------------------------------
 * pFunction:    entity_count
 * Description:  returns number of entities in database
 * Arguments:	 none
 * Returns:      number of entities
 */
static entity_count()
{
  int count;
  Enode *en;

  for ( count=0,en=en_head; en!=NULL; LL_ADVANCE(en),++count );
  return(count);
}

/*-----------------------------------------------------------------------
 * pFunction:    block_count
 * Description:  returns number of blocks in database
 * Arguments:	 none
 * Returns:      number of blocks
 */
static block_count()
{
  int count;
  Bnode *bn;

  for ( count=0,bn=bn_head; bn!=NULL; LL_ADVANCE(bn),++count );
  return(count);
}

/*-----------------------------------------------------------------------
 * pFunction:    write_entity
 * Description:  writes an entity to a file
 * Arguments IN: entity: handle of entity to be written
 *		 fp: FILE pointer
 *          OUT: *success: success indicator
 * Returns:      nothing
 */
static write_entity( entity, fp, success )
Enode *entity;
FILE *fp;
int *success;
{
  WRITESTRING( fp, entity->name, success );
  WRITEIT( &(entity->type), sizeof(gdb_Entity_type), 1, fp );
  WRITEIT( entity->data, gdb_entity_data_size(entity->type), 1, fp );
  END_OF_IO_PROC(success);
}

/*-----------------------------------------------------------------------
 * pFunction:    read_entity
 * Description:  reads an entity from a disk file and adds it to database
 * Arguments IN: fp: file pointer
 *          OUT: *success: success indicator
 * Returns:      nothing
 */
static read_entity( fp, success )
FILE *fp;
int *success;
{
  char name[MAXNAMELEN+1];
  gdb_Entity_type type;
  char *data;
  gdb_Entity entity;

  data=NULL;
  READSTRING( fp, name, success);
  READIT( &type, sizeof(gdb_Entity_type), 1, fp );
  data = malloc( (unsigned)gdb_entity_data_size(type) );
  if (data==NULL) IOABORT;
  READIT( data, gdb_entity_data_size(type), 1, fp );
  gdb_add_entity( type, data, name, &entity );
  if (entity==NULL) IOABORT;

  *success=YES;
  if (data!=NULL) free(data);
  return;
 io_abort:
  *success=NO;
  if (data!=NULL) free(data);
  return;
}

/*-----------------------------------------------------------------------
 * pFunction:    write_block
 * Description:  write a block to a file
 * Arguments IN: block: handle of block to be written
 *		 fp: pointer of file to write to
 *          OUT: *success: success indicator
 * Returns:      nothing
 */
static write_block( block, fp, success )
     Bnode *block;
     FILE *fp;
     int *success;
{
  int n_elnodes;
  Elnode *eln;

  WRITESTRING( fp, block->name, success );
  n_elnodes = gdb_block_entity_count( (gdb_Block)block );
  WRITEIT( &n_elnodes, sizeof(int), 1, fp );  
  for ( eln=block->el_head; eln!=NULL; LL_ADVANCE(eln) )
    WRITESTRING( fp, eln->e->name, success );
  END_OF_IO_PROC(success);
}

/*-----------------------------------------------------------------------
 * pFunction:    read_block
 * Description:  read a block from a file, and add it to database
 * Arguments IN: fp: pointer of file to read from
 *          OUT: *success: success indicator
 * Returns:      nothing
 */
static read_block( fp, success )
FILE *fp;
int *success;
{
  char bname[MAXNAMELEN+1], ename[MAXNAMELEN+1];
  int n_elnodes;
  gdb_Block block;
  gdb_Entity entity;

  READSTRING( fp, bname, success );
  gdb_create_block( bname, &block, NO );
  READIT( &n_elnodes, sizeof(int), 1, fp );
  while (n_elnodes--) {
    READSTRING( fp, ename, success );
    gdb_retrieve_entity( GDB_ALL_ENTITIES, GDB_NAME, ename, &entity );
    if (entity==NULL) IOABORT;
    else gdb_add_to_block( block, entity );
  }
  END_OF_IO_PROC(success);
}

/*-----------------------------------------------------------------------
 * pFunction:    write_string
 * Description:  writes a string to a file
 * Arguments IN: fp: pointer of file to write to
 *		 *string: string to be written
 *          OUT: *success: success indicator
 * Returns:	 nothing
 * Notes:	 *string must be null-terminated; the terminating
 *		 null character is the last char written to the file
 */
static write_string(fp, string, success)
     FILE           *fp;
     char           *string;
     int            *success;
{
  if (string == NULL) IOABORT;
  WRITEIT(string, sizeof(char), strlen(string)+1, fp);
  END_OF_IO_PROC(success);
}

/*-----------------------------------------------------------------------
 * pFunction:    read_string
 * Description:  reads a string from a file
 * Arguments IN: fp: pointer of file to read from
 *		 
 *          OUT: *string: string read in
 *		 *success: success indicator
 * Returns:      nothing
 * Notes:        *string is null-terminated
 */
static read_string(fp, string, success)
     FILE           *fp;
     char           *string;
     int            *success;
{
  if (string == NULL) IOABORT;
  do {
    READIT(string++, sizeof(char), 1, fp);
  } while (*(string - 1) != '\0');
  END_OF_IO_PROC(success);
}

/*-----------------------------------------------------------------------
 * pFunction:    is_integer_string
 * Description:  tells whether a string consists only of digits
 * Arguments IN: *string: string to test
 * Returns:      YES or NO
 *		 *string must be null-terminated
 */
static is_integer_string( string )
char *string;
{
  while (*string!='\0') {
    if ( (*string<'0') || (*string>'9') ) return(NO);
    ++string;
  }
  return(YES);
}
  
/*-----------------------------------------------------------------------
 * pFunction:    dist_acceptable
 * Description:  checks whether a distance is <= a tolerance
 * Arguments IN: data1,data2: the two data objects to measure distance
 *		    between
 *		 type1,type2: the entity types of these objects
 *		 tolerance: the tolerance to test against
 * Returns:      YES if distance is <= tolerance; NO otherwise
 * Notes:        If the distance computation procedure 
 *		 gdb_entity_data_dist() returns the
 *		 special value GDB_INFINITY, we automatically return
 *		 NO regardless of the tolerance level.
 */
static dist_acceptable( data1, type1, data2, type2, tolerance )
char *data1, *data2;
gdb_Entity_type type1, type2;
double tolerance;
{
  double d;

  d = gdb_entity_data_dist( data1, type1, data2, type2 );
  if (d==GDB_INFINITY)
    return(NO);
  else
    return( d <= tolerance );
}


#ifdef DEBUG
/*
 ***********************************************************************
 *	       D E B U G G I N G    P R O C E D U R E S                *
 *                                                                     *
 *  (These procedures should be removed when debugging is complete.)   *
 *                                                                     *
 ***********************************************************************
 */

print_entity_list()
{
  Enode *en;
  int count;

  printf("\n");
  for (count=0,en=en_head; en!=NULL; LL_ADVANCE(en),++count )
    print_entity(en);
  if (count==0)
    printf("Entity list is empty\n");
  else
    printf("%1d entit%s in list\n", count, count>1?"ies":"y");
}

print_entity( entity )
Enode *entity;
{
  printf("name: '%s'\n", entity->name );
  printf("type: %o\n", entity->type );
  printf("data: "); gdb_entity_data_print( entity->data, entity->type );
  printf("\n");
  printf("next: X%x\n", entity->next);
  printf("\n");
}

print_block_list()
{
  Bnode *bn;
  int count;

  printf("\n");
  for (count=0,bn=bn_head; bn!=NULL; LL_ADVANCE(bn),++count )
    print_block(bn);
  if (count==0)
    printf("Block list is empty\n");
  else
    printf("%1d block%s in list\n", count, count>1?"s":"" );
}

print_block( block )
Bnode *block;
{
  Elnode *eln;
  int count;

  printf("name: '%s'\n", block->name );
  printf("E-list:");
  for (count=0,eln=block->el_head; eln!=NULL; LL_ADVANCE(eln),++count)
    printf(" %s", eln->e->name);
  printf("\n");
  printf("(%1d entit%s in E-list)\n", count, count==1?"y":"ies");
  printf("next: X%x\n", block->next );
  printf("\n");
}
#endif 
