(* $Id: RH2.m,v 1.17 90/07/11 13:08:27 mbp Exp Locker: mbp $
 *
 * RH2.m: computations in real hyperbolic plane
 *)

(**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************)

(*
 *    Mark B. Phillips			    Robert R. Miner
 *    Department of Mathematics		    Department of Mathematics
 *    University of Maryland		    University of Maryland
 *    College Park, MD  20742		    College Park, MD  20742
 *
 *    mbp@lakisis.umd.edu		    lena@sofya.umd.edu
 *    (301) 454-6550			    (301) 454-2991
 *)

BeginPackage["RH2`", "Complex`"]

kPoint::usage = "kPoint[x,y] represents a point with coordinates (x,y)
in the Klein disk.  If p is a kPoint, pPoint, uPoint, or r21Vector,
kPoint[p] yields the corresponding point in the Klein disk."

pPoint::usage = "pPoint[x,y] represents a point with coordinates (x,y)
in the Poincare disk.  If p is a kPoint, pPoint, uPoint, or r21Vector,
pPoint[p] yields the corresponding point in the Poincare disk."

uPoint::usage = "uPoint[x,y] represents a point with coordinates (x,y)
in the upper half plane.  uPoint[Infinity] represents the point at
infinity in the upper half plane.  If p is a kPoint, pPoint, uPoint,
or r21Vector, uPoint[p] yields the corresponding point in the upper
half plane."

hPointQ::usage = "hPointQ[x] tests whether x is a kPoint, hPoint, or
uPoint."

hTangentVectorQ::usage = "hTangentVectorQ[v] tests whether v is a
kTangentVector, pTangentVector, or uTangentVector."

hSegmentQ::usage = "hSegmentQ[v] tests whether v is a kSegment,
pSegment, or uSegment."

kSegment::usage = "kSegment[p1,p2], where p1 and p2 are kPoints,
represents a geodesic segment in the Klein disk with endpoints p1 and
p2.  If either p1 or p2 is a uPoint or a pPoint, it is converted to
kPoint form.  If s is a kSegment, pSegment, or uSegment, then
kSegment[s] yields the kSegment corresponding to s."

pSegment::usage = "pSegment[p1,p2], where p1 and p2 are pPoints,
represents a geodesic segment in the Poincare disk with endpoints p1
and p2.  If either p1 or p2 is a kPoint or a uPoint, it is converted
to pPoint form.  If s is a kSegment, pSegment, or uSegment, then
pSegment[s] yields the pSegment corresponding to s."

uSegment::usage = "uSegment[p1,p2], where p1 and p2 are uPoints,
represents a geodesic segment in the upper half plane with endpoints
p1 and p2.  If either p1 or p2 is a kPoint or a pPoint, it is
converted to uPoint form.  If s is a kSegment, pSegment, or uSegment,
then uSegment[s] yields the uSegment corresponding to s."

kLine::usage = "kLine[x,y,z] represents the (infinite) line in the
Klein model whose polar point has projective coordinates [x,y,z].
kLine[{x,y,z}] is converted to kLine[x,y,z].  kLine[a, b] gives the
kLine through the kPoints a and b.  kLine will eventually be a
graphics primitive, but at the moment this has not been implemented."

r21Vector::usage = "r21Vector[x,y,z] is the vector (x,y,z) in R(2,1).
If x is something which can meaningfully be converted to a vector in
R(2,1), r21Vector[x] does the conversion."

hAffine::usage = "hAffine[p] converts p to a point in the affine plane
(R2).  Specifically, if p is a kPoint, pPoint, or uPoint, then
hAffine[p] returns {x,y} where x and y are the coordinates of p.  If p
is the r21Vector {x,y,z}, hAffine[p] returns {x/z, y/z}."

hAffineQ::usage = "hAffineQ[p] tests whether p is a point in the
affine plane (R2), i.e. a vector of length 2."

kMetric::usage = "kMetric[p] gives the 2x2 matrix of the Riemannian
metric of the Klein model of RH2 at the kPoint p.  If p is
a pPoint or uPoint it is first converted to a kPoint."

pMetric::usage = "pMetric[p] gives the 2x2 matrix of the Riemannian
metric of the Poincare model of RH2 at the pPoint p.  If p is
a kPoint or uPoint it is first converted to a pPoint."

uMetric::usage = "uMetric[p] gives the 2x2 matrix of the Riemannian
metric of the upper half-plane model of RH2 at the finite uPoint p.
If p is a kPoint or pPoint it is first converted to uPoint."

kTangentVector::usage = "kTangentVector[p, dir] is the tangent vector
in the Klein model with footpoint at the kPoint p and direction dir.
If p is a pPoint or a uPoint, it is converted to a kPoint. dir is a
vector in R2, i.e.  a list of length 2.  If v is a pTangentVector or
uTangentVector, kTangentVector[v] is the corresponding
kTangentVector."

pTangentVector::usage = "pTangentVector[p, dir] is the tangent vector
in the Poincare model with footpoint at the pPoint p and direction
dir.  If p is a kPoint or a uPoint, it is converted to a pPoint. dir
is a vector in R2, i.e.  a list of length 2.  If v is a kTangentVector
or uTangentVector, pTangentVector[v] is the corresponding
pTangentVector."

uTangentVector::usage = "uTangentVector[p, dir] is the tangent vector
in the upper half-plane model with footpoint at the uPoint p and
direction dir.  If p is a kPoint or a pPoint, it is converted to a
uPoint. dir is a vector in R2, i.e.  a list of length 2.  If v is a
kTangentVector or pTangentVector, uTangentVector[v] is the
corresponding uTangentVector."

hExp::usage = "hExp is the exponential map from the tangent bundle of
RH2 to RH2.  hExp[v], where v is a tangent vector, is the image of v
in RH2."

hQForm::usage = "hQForm[v] gives the value of the quadratic form of RH2
on the tangent vector v."

hMetric::usage = "hMetric[u,v] gives the value <u,v> of the Riemannian
metric of RH2 for the tangent vectors u and v.  u and v must be
tangent vectors with the same basepoint."

hAngle::usage = "hAngle[u,v] gives the hyperbolic angle between u and
v.  If u and v are tangent vectors, they must have the same basepoint,
and hAngle[u,v] returns the counterclockwise angle from u to v.  If u
and v are segments or lines, hAngle[u,v] returns the acute angle
between them."

hLength::usage = "hLength[x] gives the hyperbolic length of x.  x
can be a tangent vector or a segment."

hDistance::usage = "hDistance[p1, p2] gives the hyperbolic distance
between the points p1 and p2."

hBisector::usage = "hBisector[a, b] returns the kLine which bisects
the line segment from a to b; a and b should be kPoints, pPoint, or
uPoints.  If u and v are negative r21Vectors, hBisector[u, v] returns
the r21Vector which is polar to the line bisecting the segment
connected the points which u and v represent.  The coordinates of the
result are signed so its h21Dot product with u is positive and with v
is negative."

hUnitVector::usage = "hUnitVector[v] gives a unit hyperbolic-length
tangent vector with the same base point and direction as v.  v can be
a kTangentVector, pTangentVector, or uTangentVector."

hLog::usage = "hLog[b, a] is the vector in the tangent space at b
whose hExp is a.  b and a can be kPoints, pPoints, or uPoints; the
vector is returned in the model of b.  If the point a is at infinity,
hLog[b, a] returns the (hyperbolic) unit vector at b which points
towards a."

h21Dot::usage = "h21Dot[u, v] is the indefinite scalar product of the
vectors u and v in R21.  h21Dot[v] is the same as h21Dot[v, v]."

h21Matrix::usage = "h21Matrix is the matrix of the indefinite scalar product
ofR21."

h21Cross::usage = "h21Cross[u, v] is the indefinite cross product of the
vectors u and v in R21."

hPerp::usage = "hPerp[l,x] returns the kLine passing through the point
x perpendicular to the line l.  Here l is an r21Vector or a kLine,
while x is a kPoint, pPoint, uPoint or r21Vector.  The point x need
not lie on l."

hMidPoint::usage = "hMidPoint[x,y] returns the midpoint of the the
segment spanned by x and y.  The points x and y may be given as
kPoints, pPoints, uPoints, or r21Vectors.  hMidPoint[s], where s is a
segment, returns the midpoint of s.  The result will be in the same
model as the input."

hIntersection::usage = "hIntersection[l1, l2] gives the point at
which the kLines l1 and l2 intersect."

hReflection::usage = "hReflection[l] returns the matrix of the
reflection in l.  The line l must be given as a kline or an
r21Vector."

hTranslation::usage = "hTranslation[x, y] returns the matrix for the 
translation along the line spanned by x and y which takes x to y.
The points x and y may be given as kPoints, pPoints, uPoints, or
r21Vectors."

hRotation::usage = "hRotation[x, theta] returns the matrix for the rotation
which fixes x, and acts like a Euclidean rotation by angle theta on the
tangent space at x.  x may be given as a kPoint, pPoint, uPoint or
an r21Vector."

hParabolic::usage = "hParabolic[x, theta] returns the matrix for the
parabolic transformation fixing x, given by the composition of reflections
in lines intersecting at x at an angle theta.  Thus, -Pi / 2 < theta < Pi /2.
x may be given as a kPoint, pPoint, uPoint or an r21Vector."

hApply::usage = "hApply[ m, o ] returns the image of the object o under the
transformation m.  The type of the image object will agree with the type of
the input object.  The object o may be a list of objects.  If the input object 
contains multiple objects, their types must agree.  The transformation m
must be specified as a matrix."

hSegmentEndPoints::usage = "hSegmentEndPoints[s] gives a list of the two
endpoints of the segment s."

Begin["`private`"]

Unprotect[Complex]
Complex[p_pPoint] := p[[1]] + I p[[2]]
Complex[p_kPoint] := p[[1]] + I p[[2]]
Complex[p_uPoint] := p[[1]] + I p[[2]]
Protect[Complex]

kPoint[p_kPoint] := p
pPoint[p_pPoint] := p
uPoint[p_uPoint] := p

kPoint[z_?NumberQ] := kPoint[Re[z], Im[z]]
pPoint[z_?NumberQ] := pPoint[Re[z], Im[z]]
uPoint[z_?NumberQ] := uPoint[Re[z], Im[z]]

pPoint[{x_,y_}] := pPoint[x,y]
uPoint[{x_,y_}] := uPoint[x,y]
kPoint[{x_,y_}] := kPoint[x,y]

kPoint[v_r21Vector] := kPoint[hAffine[v]]
pPoint[v_r21Vector] := pPoint[kPoint[v]]
uPoint[v_r21Vector] := uPoint[kPoint[v]]

(*
 * pPoint --> kPoint:   
 *
 *     [ 2 z / (1 + |z|^2)  if |z|^2 <  1 ]
 * w = [                                  ]
 *     [ z                  if |z|^2 >= 1 ]
 *)

kPoint[ p_pPoint ] :=
  kPoint[ pPointAsComplex[Complex[p]] ]

kPoint[ pPointAsComplex[z_]  ] :=
  kPoint[ pPointAsComplex[z], pPointNormSquared[ComplexNormSquared[z]] ]

kPoint[ pPointAsComplex[z_], pPointNormSquared[nz2_]  ] :=
  kPoint[ 2 z / (1 + nz2 ) ] /; (nz2 < 1.0)

kPoint[ pPointAsComplex[z_], pPointNormSquared[nz2_]  ] :=
  kPoint[ z ] /; (nz2  >= 1.0)

(*
 * kPoint --> pPoint:   
 *
 * 1. If 0 <= |z|^2 <= 0.5 let
 *		      
 *		nw2 = [ |z|^2 / (1 + Sqrt[1-|z|^2])^2  if |z|^2 <= 0.5 ]
 *    and
 *		  w = z ( 1 + nw2 ) / 2
 *
 * 2. If 0.5 < |z|^2 < 1.0 let
 *
 *		nw2 = [ (1 - Sqrt[1-|z|^2])^2 / |z|^2  if |z|^2 >  0.5 ]
 *    and
 *		  w = z ( 1 + nw2 ) / 2
 *
 * 3. if |z|^2 >= 1, let
 *
 *		  w = z
 *)

pPoint[ p_kPoint ] :=
  pPoint[ kPointAsComplex[Complex[p]] ]

pPoint[ kPointAsComplex[z_] ] :=
  pPoint[ kPointAsComplex[z], kPointNormSquared[ComplexNormSquared[z]] ]

pPoint[ kPointAsComplex[z_], kPointNormSquared[nz2_] ] :=
  pPoint[ z ] /; (nz2 >= 1.0)

pPoint[ kPointAsComplex[z_], kPointNormSquared[nz2_] ] :=
  pPoint[ kPointAsComplex[z],
	  pPointNormSquared[ nz2 / (1 + Sqrt[ 1 - nz2 ])^2 ] ] /; (nz2 <= 0.5)

pPoint[ kPointAsComplex[z_], kPointNormSquared[nz2_] ] :=
  pPoint[ kPointAsComplex[z],
	  pPointNormSquared[ (1 - Sqrt[ 1 - nz2 ])^2 / nz2 ] ] /; (nz2 >  0.5)

pPoint[ kPointAsComplex[z_], pPointNormSquared[nw2_] ] :=
  pPoint[ z ( 1 + nw2 ) / 2 ]


(*
 * pPoint --> uPoint
 *
 *  w = I (1 + z) / (1 - z) if z != 1
 *  w = uPoint[Infinity]  if z = 1
 *)

uPoint[ p_pPoint ] :=
  uPoint[ pPointAsComplex[Complex[p]] ]

uPoint[ pPointAsComplex[z_] ] :=
  uPoint[Infinity] /; (N[z - 1] == 0)

uPoint[ pPointAsComplex[z_] ] :=
  uPoint[ I (1 + z) / (1 - z) ]

uPoint[ ComplexInfinity ] :=
  uPoint[ Infinity ]

(*
 * uPoint --> pPoint:
 *
 * w = (z - I) / (z + I) if z != infinity
 * w = 1 if z = infinity
 *)

pPoint[ uPoint[Infinity] ] :=
  pPoint[1,0]

pPoint[ p_uPoint ] :=
  pPoint[ uPointAsComplex[Complex[p]] ]

pPoint[ uPointAsComplex[z_] ] :=
  pPoint[ (z - I) / (z + I) ]

(*
 *  uPoint <--> kPoint:
 *)

kPoint[ p_uPoint ] :=
  kPoint[ pPoint[p] ]
uPoint[ p_kPoint ] :=
  uPoint[ pPoint[p] ]

hPointQ[ x_ ] := 
  TrueQ[ Head[x]==kPoint || Head[x]==pPoint || Head[x]==uPoint ]

hSegmentQ[ x_ ] := 
  TrueQ[ Head[x]==kSegment || Head[x]==pSegment || Head[x]==uSegment ]

kSegment[ e1_, e2_ ] :=
  kSegment[ kPoint[e1], e2 ] /; (Head[e1] != kPoint)
kSegment[ e1_, e2_ ] :=
  kSegment[ e1, kPoint[e2] ] /; (Head[e2] != kPoint)

pSegment[ e1_, e2_ ] :=
  pSegment[ pPoint[e1], e2 ] /; (Head[e1] != pPoint)
pSegment[ e1_, e2_ ] :=
  pSegment[ e1, pPoint[e2] ] /; (Head[e2] != pPoint)

uSegment[ e1_, e2_ ] :=
  uSegment[ uPoint[e1], e2 ] /; (Head[e1] != uPoint)
uSegment[ e1_, e2_ ] :=
  uSegment[ e1, uPoint[e2] ] /; (Head[e2] != uPoint)

kSegment[ s_kSegment ] := s
pSegment[ s_pSegment ] := s
uSegment[ s_uSegment ] := s

kSegment[ s_pSegment ] :=
  kSegment[ kPoint[s[[1]]], kPoint[s[[2]]] ]
kSegment[ s_uSegment ] :=
  kSegment[ kPoint[s[[1]]], kPoint[s[[2]]] ]

pSegment[ s_kSegment ] :=
  pSegment[ pPoint[s[[1]]], pPoint[s[[2]]] ]
pSegment[ s_uSegment ] :=
  pSegment[ pPoint[s[[1]]], pPoint[s[[2]]] ]

uSegment[ s_kSegment ] :=
  uSegment[ uPoint[s[[1]]], uPoint[s[[2]]] ]
uSegment[ s_pSegment ] :=
  uSegment[ uPoint[s[[1]]], uPoint[s[[2]]] ]


kSegment[ l_kLine ] :=
  Apply[ kSegment, LineEndPoints[l] ]
pSegment[ l_kLine ] :=
  pSegment[ kSegment[l] ]
uSegment[ l_kLine ] :=
  uSegment[ kSegment[l] ]

hSegmentEndPoints[s_kSegment] :=
  s /. kSegment->List
hSegmentEndPoints[s_pSegment] :=
  s /. pSegment->List
hSegmentEndPoints[s_uSegment] :=
  s /. uSegment->List

kLine[ r21Vector[x_,y_,z_] ] :=
  kLine[x,y,z]
kLine[ a_?hPointQ, b_?hPointQ ] :=
  kLine[ r21Vector[a], r21Vector[b] ]
kLine[ a_r21Vector, b_r21Vector ] :=
  kLine[ h21Cross[ a, b] ]

hDistance[p1_pPoint, p2_pPoint] :=
  hDistance[ pPointAsComplex[Complex[p1]], pPointAsComplex[Complex[p2]] ]
hDistance[ pPointAsComplex[z_], pPointAsComplex[w_] ] :=
  hDistance[ AbsOneMinusZWBar[ Abs[ 1 - z Conjugate[w] ] ],
    	     AbsZMinusW[ Abs[ z - w ] ] ]
hDistance[ AbsOneMinusZWBar[ a_ ], AbsZMinusW[ b_ ] ] :=
  Log[ (a + b) / (a - b) ]

hDistance[ p1_, p2_ ] :=
  hDistance[ pPoint[p1], pPoint[p2] ]

pMetric[ p_pPoint ] :=
  4 IdentityMatrix[2] / (1 - ComplexNormSquared[Complex[p]])^2
pMetric[ p_kPoint ] :=
  pMetric[ pPoint[p] ]
pMetric[ p_uPoint ] :=
  pMetric[ pPoint[p] ]

uMetric[ p_uPoint ] :=
  IdentityMatrix[2] / Im[Complex[p]]^2
uMetric[ p_kPoint ] :=
  uMetric[ uPoint[p] ]
uMetric[ p_pPoint ] :=
  uMetric[ uPoint[p] ]

kMetric[  p_kPoint  ] :=
  kMetric[ kPoint1Squared[ p[[1]]^2 ],
	   kPoint2Squared[ p[[2]]^2 ],
	   kPoint12[ p[[1]] p[[2]] ] ]
kMetric[ kPoint1Squared[ x1sq_ ],
  	 kPoint2Squared[ x2sq_ ],
	 kPoint12[ x1x2_ ] ]:=
  { { 1 - x2sq, x1x2}, {x1x2, 1 - x1sq} } / (1 - x1sq - x2sq)^2
kMetric[ p_pPoint ] :=
  kMetric[ kPoint[p] ]
kMetric[ p_uPoint ] :=
  kMetric[ kPoint[p] ]

dPoincareToUHP::usage = "dPoincareToUHP[p] gives the 2x2 matrix of the
tangent map at p of the function which converts Poincare --> UHP."

dPoincareToUHP[ pPoint[x_,y_] ] :=
  Release[{{4(-1 + x)y,                -2(-1 + x - y)(-1 + x + y)}, 
	   {2(-1 + x - y)(-1 + x + y), 4(-1 + x)y                }}
	  / ((x-1)^2 + y^2)^2 ]

dUHPToPoincare::usage = "dUHPToPoincare[p] gives the 2x2 matrix of the
tangent map at p of the function which converts UHP --> Poincare."

dUHPToPoincare[ uPoint[x_,y_] ] :=
  Release[{{4x(1 + y),                -2(-1 + x - y)(1 + x + y)}, 
	   {2(-1 + x - y)(1 + x + y), 4x(1 + y)                }}
	  / (x^2 + (y+1)^2)^2 ]

dPoincareToKlein::usage = "dPoincareToKlein[p] gives the 2x2 matrix of
the tangent map at p of the function which converts Poincare --> Klein."

dPoincareToKlein[ pPoint[x_,y_] ] :=
  Release[{{2(1 - x^2 + y^2), -4 x y          },
	   {-4 x y,           2(1 + x^2 - y^2)}}
	  / (1 + x^2 + y^2)^2 ]

dKleinToPoincare::usage = "dKleinToPoincare[p] gives the 2x2 matrix of
the tangent map at p of the function which converts Klein --> Poincare."

dKleinToPoincare[ p_kPoint ] :=
  Inverse[ dPoincareToKlein[ pPoint[p] ] ]

dKleinToUHP::usage = "dKleinToUHP[p] gives the 2x2 matrix of the tangent
map at p of the function which converts Klein --> UHP."

dKleinToUHP[ p_kPoint ] :=
  dPoincareToUHP[ pPoint[p] ] . dKleinToPoincare[p]

dUHPToKlein::usage = "dUHPToKlein[p] gives the 2x2 matrix of the tangent
map at p of the function which converts UHP --> Klein."

dUHPToKlein[ p_uPoint ] :=
  dPoincareToKlein[ pPoint[p] ] . dUHPToPoincare[p]

hTangentVectorQ[ x_ ] := 
  TrueQ[ Head[x]==kTangentVector ||
	 Head[x]==pTangentVector ||
	 Head[x]==uTangentVector  ]

kTangentVector[ p_pPoint, dir_ ] :=
  kTangentVector[ kPoint[p], dir ]
kTangentVector[ p_uPoint, dir_ ] :=
  kTangentVector[ kPoint[p], dir ]

pTangentVector[ p_kPoint, dir_ ] :=
  pTangentVector[ pPoint[p], dir ]
pTangentVector[ p_uPoint, dir_ ] :=
  pTangentVector[ pPoint[p], dir ]

uTangentVector[ p_pPoint, dir_ ] :=
  uTangentVector[ uPoint[p], dir ]
uTangentVector[ p_kPoint, dir_ ] :=
  uTangentVector[ uPoint[p], dir ]

pTangentVector[ kTangentVector[ p_kPoint, dir_ ] ] :=
  pTangentVector[ pPoint[p], dKleinToPoincare[p] . dir ]
pTangentVector[ uTangentVector[ p_uPoint, dir_ ] ] :=
  pTangentVector[ pPoint[p], dUHPToPoincare[p] . dir ]
  
kTangentVector[ pTangentVector[ p_pPoint, dir_ ] ] :=
  kTangentVector[ kPoint[p], dPoincareToKlein[p] . dir ]
kTangentVector[ uTangentVector[ p_uPoint, dir_ ] ] :=
  kTangentVector[ kPoint[p], dUHPToKlein[p] . dir ]
  
uTangentVector[ kTangentVector[ p_kPoint, dir_ ] ] :=
  uTangentVector[ uPoint[p], dKleinToUHP[p] . dir ]
uTangentVector[ pTangentVector[ p_pPoint, dir_ ] ] :=
  uTangentVector[ uPoint[p], dPoincareToUHP[p] . dir ]

kTangentVector /:
  s_ kTangentVector[ p_kPoint, dir_ ] := kTangentVector[ p, s dir ]
pTangentVector /:
  s_ pTangentVector[ p_pPoint, dir_ ] := pTangentVector[ p, s dir ]
uTangentVector /:
  s_ uTangentVector[ p_uPoint, dir_ ] := uTangentVector[ p, s dir ]

(*
 * We define hExp in the Klein model.  We first define it on
 * the tangent space at the origin. We do this via the formula
 *                   v Tanh[t |v|] / |v|
 * for a constant speed geodesic through the origin having tangent
 * vector v at t=0.  Here |v| is the hyperbolic norm of v, which
 * equals the Euclidean norm of v since v is at the origin.  Thus
 *		hExp[v] = v Tanh[ |v| ] / |v|
 *)
  
hExp[ kTangentVector[ kPoint[0,0], v_ ] ] :=
   hExp[ kTangentVector[ kPoint[0,0], v ],
         DirNorm[ Sqrt[v[[1]]^2 + v[[2]]^2] ] ]
hExp[ kTangentVector[ kPoint[0,0], v_ ], DirNorm[ norm_ ] ] :=
  kPoint[ v Tanh[norm] / norm ] /; (norm > 0)
hExp[ kTangentVector[ kPoint[0,0], v_ ], DirNorm[ norm_ ] ] :=
  kPoint[ 0, 0 ] /; (Chop[norm]===0)

(*
 * Next we define hExp for a point (r,0) on the positive x-axis.  We do
 * this via the hyperbolic transformation T which takes (r,0) to (0,0)
 * along the x-axis.  T is given by
 *
 *               x c - s     y
 *    T(x,y) = ( -------, ------- )
 *               c - x s  c - x s
 *
 * where c = cosh[t], s = sinh[t], and t = hyperbolic distance from
 * (0,0) to (r,0).  (Solving for c and s in terms of r gives
 * c = 1 / Sqrt[1 - r^2] and s = r / Sqrt[1 - r^2].)  hExp is then
 * given by
 *               -1
 *    hExp[v] = T  ( hExp[ dT     (v) ] )
 *			     (r,0)
 *
 * where the inner hExp acts on the tangent space at the origin, since
 * T(r,0) = (0,0).
 *)

hExp[ kTangentVector[ kPoint[r_, 0], v_ ] ] :=
  Block[ {omr2, sqrtomr2, c, s, Tinv, dT},
    omr2 = 1 - r^2;
    sqrtomr2 = Sqrt[omr2];
    c = 1 / sqrtomr2;
    s = r / sqrtomr2;
    Tinv[ kPoint[xp_,yp_] ] := kPoint[ {xp c + s, yp} / (c + xp s) ];
    dT = { {1, 0}, {0, sqrtomr2} } / omr2;

    Return[ Tinv[ hExp[ kTangentVector[ kPoint[0, 0], dT . v ] ] ] ];
    ]

(*
 * Finally we define the general hExp via the elliptic transformation
 * R that moves the point (x,y) to the positive x-axis.  R is given by
 *
 *                   1        [ x    y ] [ x' ]
 * R(x',y') = --------------- [        ] [    ]
 *            Sqrt[x^2 + y^2] [-y    x ] [ y' ]
 *
 * hExp is then given by
 *               -1
 *    hExp[v] = R  ( hExp[ dR     (v) ] )
 *			     (x,y)
 *
 * where the inner hExp acts on the tangent space at (|(x,y)|, 0),
 * since R(x,y) = (|(x,y)|, 0).
 *)

hExp[ kTangentVector[ kPoint[x_, y_], v_ ] ] :=
  Block[ {norm, RinvMatrix, Rinv, dR},
    norm = Sqrt[ x^2 + y^2 ];
    RinvMatrix = { {x, -y}, {y, x} } / norm;
    Rinv[ kPoint[xp_,yp_] ] := kPoint[ RinvMatrix . {xp,yp} ];
    dR = { {x, y}, {-y, x} } / norm;

    Return[ Rinv[ hExp[ kTangentVector[ kPoint[norm, 0], dR . v ] ] ] ];
    ]

hExp[ v_pTangentVector ] :=
  pPoint[ hExp[ kTangentVector[v] ] ]

hExp[ v_uTangentVector ] :=
  uPoint[ hExp[ kTangentVector[v] ] ]

hQForm[ kTangentVector[p_kPoint, dir_] ] :=
  dir . kMetric[p] . dir

hQForm[ pTangentVector[p_pPoint, dir_] ] :=
  dir . pMetric[p] . dir

hQForm[ uTangentVector[p_uPoint, dir_] ] :=
  dir . uMetric[p] . dir

hLength[ v_kTangentVector ] :=
  Sqrt[ hQForm[v] ]

hLength[ v_pTangentVector ] :=
  Sqrt[ hQForm[v] ]

hLength[ v_uTangentVector ] :=
  Sqrt[ hQForm[v] ]

hLength[ kSegment[p1_,p2_] ] :=
  hDistance[ p1, p2 ]

hLength[ pSegment[p1_,p2_] ] :=
  hDistance[ p1, p2 ]

hLength[ uSegment[p1_,p2_] ] :=
  hDistance[ p1, p2 ]

hMetric::badbase = "Basepoints of `1` and `2` are not equal."

hMetric[ kTangentVector[x1_, v1_], kTangentVector[x2_, v2_] ] :=
  v1 . kMetric[x1] . v2 /; N[ hAffine[x1] - hAffine[x2] ] === {0.0, 0.0}
hMetric[ u_kTangentVector, v_kTangentVector ] :=
  ( Message[hMetric::badbase, u, v] ; Infinity )
hMetric[ u_kTangentVector, v_pTangentVector ] :=
  hMetric[ u, kTangentVector[v] ]
hMetric[ u_kTangentVector, v_uTangentVector ] :=
  hMetric[ u, kTangentVector[v] ]

hMetric[ pTangentVector[x1_, v1_], pTangentVector[x2_, v2_] ] :=
  v1 . pMetric[x1] . v2 /; N[ hAffine[x1] - hAffine[x2] ] === {0.0, 0.0}
hMetric[ u_pTangentVector, v_pTangentVector ] :=
  ( Message[hMetric::badbase, u, v] ; Infinity )
hMetric[ u_pTangentVector, v_kTangentVector ] :=
  hMetric[ u, pTangentVector[v] ]
hMetric[ u_pTangentVector, v_uTangentVector ] :=
  hMetric[ u, pTangentVector[v] ]

hMetric[ uTangentVector[x1_, v1_], uTangentVector[x2_, v2_] ] :=
  v1 . uMetric[x1] . v2 /; N[ hAffine[x1] - hAffine[x2] ] === {0.0, 0.0}
hMetric[ u_uTangentVector, v_uTangentVector ] :=
  ( Message[hMetric::badbase, u, v] ; Infinity )
hMetric[ u_uTangentVector, v_pTangentVector ] :=
  hMetric[ u, uTangentVector[v] ]
hMetric[ u_uTangentVector, v_kTangentVector ] :=
  hMetric[ u, uTangentVector[v] ]

hAngle[u_?hTangentVectorQ, v_?hTangentVectorQ] :=
  0 /; (h21Dot[r21Vector[u[[1]]]] >= 0)

hAngle[u_?hTangentVectorQ, v_?hTangentVectorQ] :=
  ArcCos[ hMetric[u,v] / ( hLength[u] hLength[v] ) ]

hAngle[ a_?hSegmentQ, b_ ] :=
  hAngle[ kLine[a], b ]
hAngle[ a_, b_?hSegmentQ ] :=
  hAngle[ a, kLine[b] ]

hAngle[ a_kLine, b_kLine ] :=
  ArcCos[ Abs[ h21Dot[a,b] ] / Sqrt[ h21Dot[a] h21Dot[b] ] ] /;
    (h21Dot[h21Cross[a,b]] < 0)

(* for asymptotic or ultraparallel lines: *)
hAngle[ a_kLine, b_kLine ] := 0 

r21Vector[ p_kPoint ] := r21Vector[p[[1]], p[[2]], 1]
r21Vector[ p_pPoint ] := r21Vector[ kPoint[p] ]
r21Vector[ p_uPoint ] := r21Vector[ kPoint[p] ]
r21Vector[ kLine[x_,y_,z_] ] := r21Vector[x, y, z]
r21Vector[ v_?hAffineQ ] := r21Vector[v[[1]], v[[2]], 1]
r21Vector[ v_r21Vector ] := v
r21Vector[ {x_,y_,z_} ]:= r21Vector[x,y,z]

r21Vector/: r21Vector[x1_,y1_,z1_] + r21Vector[x2_,y2_,z2_] :=
  r21Vector[x1+x2, y1+y2, z1+z2]
r21Vector/: r21Vector[x1_,y1_,z1_] - r21Vector[x2_,y2_,z2_] :=
  r21Vector[x1-x2, y1-y2, z1-z2]
r21Vector/: s_ * r21Vector[x_,y_,z_] :=
  r21Vector[s x, s y, s z]
r21Vector/: r21Vector[x_,y_,z_] / s_ :=
  r21Vector[x/s, y/s, z/s]
r21Vector/: Dot[m_?MatrixQ, r21Vector[x_,y_,z_]] :=
  r21Vector[ m . {x,y,z} ]
r21Vector/: Dot[r21Vector[x_,y_,z_], m_?MatrixQ] :=
  r21Vector[ {x,y,z} . m ]

hAffine[ x_kPoint ] := {x[[1]], x[[2]]}
hAffine[ x_pPoint ] := {x[[1]], x[[2]]}
hAffine[ x_uPoint ] := {x[[1]], x[[2]]}
hAffine[ x_r21Vector ] := {x[[1]], x[[2]]} / x[[3]]
hAffine[ x_?hAffineQ ] := x

hAffineQ[ x_ ] := TrueQ[ VectorQ[x] && Length[x]==2 ]

hUnitVector[ kTangentVector[p_, v_] ] :=
  kTangentVector[p, v] / hLength[ kTangentVector[p, v] ]

hUnitVector[ pTangentVector[p_, v_] ] :=
  pTangentVector[p, v] / hLength[ pTangentVector[p, v] ]

hUnitVector[ uTangentVector[p_, v_] ] :=
  uTangentVector[p, v] / hLength[ uTangentVector[p, v] ]

hLog[ b_kPoint, a_kPoint ] :=
  (hDistance[b, a]  * 
    hUnitVector[ kTangentVector[ b, hAffine[a] - hAffine[b] ] ]) /;
      (h21Dot[r21Vector[a]] < 0)
hLog[ b_kPoint, a_kPoint ] :=
    hUnitVector[ kTangentVector[ b, hAffine[a] - hAffine[b] ] ]
hLog[ b_kPoint, a_pPoint ] :=
  hLog[ b, kPoint[a] ]
hLog[ b_kPoint, a_uPoint ] :=
  hLog[ b, kPoint[a] ]

hLog[ b_pPoint, a_pPoint ] :=
  pTangentVector[ hLog[ kPoint[b], kPoint[a] ] ]
hLog[ b_pPoint, a_kPoint ] :=
  pTangentVector[ hLog[ kPoint[b], a ] ]
hLog[ b_pPoint, a_uPoint ] :=
  pTangentVector[ hLog[ kPoint[b], kPoint[a] ] ]

hLog[ b_uPoint, a_uPoint ] :=
  uTangentVector[ hLog[ kPoint[b], kPoint[a] ] ]
hLog[ b_uPoint, a_pPoint ] :=
  uTangentVector[ hLog[ kPoint[b], kPoint[a] ] ]
hLog[ b_uPoint, a_kPoint ] :=
  uTangentVector[ hLog[ kPoint[b], a ] ]

hLog::notincident = "Base point does not lie on line"

hLog[ b_kPoint, l_kLine ] :=
  hUnitVector[ kTangentVector[b, {l[[2]],-l[[1]]}] ] /;
    (Chop[h21Dot[b, l]]===0)
hLog[ b_kPoint, kLine[x_,y_,z_] ] :=
  (Message[hLog::notincident]; Null)
hLog[ b_pPoint, l_kLine ] :=
  pTangentVector[ hLog[ kPoint[b], l ] ]
hLog[ b_uPoint, l_kLine ] :=
  uTangentVector[ hLog[ kPoint[b], l ] ]

h21Dot[ r21Vector[x1_,y1_,z1_], r21Vector[x2_,y2_,z2_] ] :=
  x1 x2 + y1 y2 - z1 z2
h21Dot[ u_?hPointQ, v_ ] :=  h21Dot[ r21Vector[u], v ]
h21Dot[ u_, v_?hPointQ ] :=  h21Dot[ u, r21Vector[v] ]
h21Dot[ u_kLine, v_ ] :=     h21Dot[ r21Vector[u], v ]
h21Dot[ u_, v_kLine ] :=     h21Dot[ u, r21Vector[v] ]
h21Dot[ u_?hAffineQ, v_ ] := h21Dot[ r21Vector[u], v ]
h21Dot[ u_, v_?hAffineQ ] := h21Dot[ u, r21Vector[v] ]

h21Dot[ u_ ] := h21Dot[ u, u]

h21Matrix = DiagonalMatrix[ {1, 1, -1} ]

h21Cross[ r21Vector[x1_, y1_, z1_], r21Vector[x2_, y2_, z2_] ] :=
  r21Vector[Release[ Det[ { h21Matrix, {x1,y1,z1}, {x2,y2,z2} } ] ]]
h21Cross[ u_?hPointQ, v_ ] := h21Cross[ r21Vector[u], v ]
h21Cross[ u_, v_?hPointQ ] := h21Cross[ u, r21Vector[v] ]
h21Cross[ u_kLine, v_ ] :=    h21Cross[ r21Vector[u], v ]
h21Cross[ u_, v_kLine ] :=    h21Cross[ u, r21Vector[v] ]
h21Cross[ u_?AffineQ, v_ ] := h21Cross[ r21Vector[u], v ]
h21Cross[ u_, v_?AffineQ ] := h21Cross[ u, r21Vector[v] ]

hBisector[ a_r21Vector, b_r21Vector ] :=
  a / Sqrt[-h21Dot[a]] - b / Sqrt[-h21Dot[b]]
hBisector[ a_?hPointQ, b_?hPointQ ] :=
  kLine[ hBisector[ r21Vector[a], r21Vector[b] ] ]

hPerp[ l_r21Vector, p_r21Vector ] := h21Cross[ l, p ]
hPerp[ l_kLine, p_ ] := kLine[ hPerp[ r21Vector[l], p ] ]
hPerp[ l_, p_?hPointQ ] := hPerp[ l, r21Vector[p] ]

hMidPoint[ a_r21Vector, b_r21Vector ] :=
  h21Cross[h21Cross[a,b], hBisector[a, b]]
hMidPoint[ a_kPoint, b_kPoint ] := 
  kPoint[h21Cross[h21Cross[a,b], hBisector[a, b]]]
hMidPoint[ a_pPoint, b_pPoint ] := 
  pPoint[h21Cross[h21Cross[a,b], hBisector[a, b]]]
hMidPoint[ a_uPoint, b_uPoint ] := 
  uPoint[h21Cross[h21Cross[a,b], hBisector[a, b]]]
hMidPoint[ s_?hSegmentQ ] :=
  hMidPoint[ s[[1]], s[[2]] ]

hIntersection::noexist = "Lines are ultraparallel."

hIntersection[ l1_kLine, l2_kLine ] :=
  hIntersection[ ProposedIntersection[ h21Cross[l1, l2] ] ]
hIntersection[ ProposedIntersection[ v_r21Vector ] ] :=
  kPoint[ v ] /; h21Dot[ v ] <= 0
hIntersection[ ProposedIntersection[ v_r21Vector ] ] :=
  Message[hIntersection::noexist]

hReflection[ l_r21Vector ] := IdentityMatrix[3] - (2 / h21Dot[l,l]) {
         {l[[1]] l[[1]], l[[1]] l[[2]], -l[[1]] l[[3]]},
         {l[[1]] l[[2]], l[[2]] l[[2]], -l[[2]] l[[3]]},
         {l[[1]] l[[3]], l[[2]] l[[3]], -l[[3]] l[[3]]}}
hReflection[ l_kLine ] := hReflection[ r21Vector[l] ]

hTranslation[ a_r21Vector, a_ ] := {{1,0,0},{0,1,0},{0,0,1}}
hTranslation[ a_r21Vector, b_r21Vector ] := 
	 hReflection[hBisector[b,hReflection[hPerp[kLine[a,b],b]] . a]] .
	 hReflection[hPerp[kLine[a,b],b]]
hTranslation[ a_?hPointQ, b_ ] :=	hTranslation[ r21Vector[a], b ] 
hTranslation[ a_, b_?hPointQ ] :=	hTranslation[ a, r21Vector[b] ] 

hRotation[ p_r21Vector, theta_?NumberQ] := 
	hTranslation[ r21Vector[0, 0, 1], p ] .
	{{Cos[theta] , -Sin[theta], 0}, {Sin[theta], Cos[theta], 0},
	 {0, 0, 1}} .
	hTranslation[ p, r21Vector[0, 0, 1] ]
hRotation[ p_?hPointQ, theta_?NumberQ] := hRotation[ r21Vector[p], theta ] 

hParabolic[ a_r21Vector, theta_?NumberQ ] :=
        hReflection[kLine[a, 
	{(a[[1]] (1 - Cos[theta / 2]) + a[[2]] Sin[theta / 2]) / a[[3]],
	 (a[[2]] (1 - Cos[theta / 2]) - a[[1]] Sin[theta / 2]) / a[[3]], 1}]] . 
	hReflection[hLine[a, kPoint[0,0]]] /; hDot[a,a] == 0 
hParabolic[ a_?hPointQ, theta_?NumberQ ] :=hParabolic[ r21Vector[a], theta ]

hApply[m_List, p_r21Vector ] := m . p
hApply[m_List, p_pPoint] := pPoint[m . r21Vector[p]]
hApply[m_List, p_kPoint] := kPoint[m . r21Vector[p]]
hApply[m_List, p_uPoint] := uPoint[m . r21Vector[p]]
hApply[m_List, s_pSegment] := 
	pSegment[hApply[m, s[[1]]], hApply[m, s[[2]]]]
hApply[m_List, s_kSegment] := 
	kSegment[hApply[m, s[[1]]], hApply[m, s[[2]]]]
hApply[m_List, s_uSegment] := 
	uSegment[hApply[m, s[[1]]], hApply[m, s[[2]]]]
hApply[m_List, l_kLine ] := 
	kLine[hApply[m, r21Vector[l]]]

hApply[t_, list_List] := Map[ hApply[t,#]&, list ]

LineEndPoints::badline = "illegal line --- polar point is negative"

LineEndPoints[kLine[x0_, y0_, z0_]] :=
  Block[ {b, n2, n, en2, x0z0, y0z0},
    n2 = h21Dot[r21Vector[x0,y0,z0]];
    If[ Chop[n2]<0, Message[LineEndPoints::badline]; Return[Null] ];
    n = Sqrt[n2];
    x0z0 = x0 z0;
    y0z0 = y0 z0;
    (* Note: these points already lie on the unit circle; we FORCE them
       to do so even more to account for round-off errors: *)
    Return[ Map[ kPoint[UnitAffineVector[#]]&,
		 { {x0z0 + y0 n, y0z0 - x0 n},
		   {x0z0 - y0 n, y0z0 + x0 n} } / (x0^2 + y0^2) ] ]
    ]

UnitAffineVector[{x_, y_}] :=
  {x, y} / Sqrt[x^2 + y^2]

End[]

EndPackage[]
