/* $Id: print.c,v 2.3 89/09/20 17:01:46 mbp Exp $
 *
 * print.c: procedures for creating and using printer window
 */

/***************************************************************************
 *                Copyright (C) 1990 by Mark B. Phillips                   *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both      *
 *      that copyright notice and this permission notice appear in         *
 *      supporting documentation, and that the names of Mark B.            *
 *      Phillips, or the University of Maryland not be used in             *
 *      advertising or publicity pertaining to distribution of the         *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the author Mark B. Phillips    *
 *      in any publication which uses part or all of any image produced    *
 *      by this software.                                                  *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************/

#include <stdio.h>
#include "gr.h"
#include "internal.h"

/* This max caption length assumes a point size of 16: */
#define MAX_CAPTION_LENGTH 129
#define FILENAME_DISPLAY_LENGTH 30
#define FILENAME_STORED_LENGTH 256
#define LABEL_DISPLAY_LENGTH 35
/* Max label length is 26 less than max caption length, because time */
/* string is 26 chars: */
#define LABEL_STORED_LENGTH (MAX_CAPTION_LENGTH-26)
#define PS_MESSAGE_LENGTH 37

Frame		GR_ps_frame = NULL;
Panel		GR_ps_panel = NULL;
static Panel_item	ps_printer_button, ps_file_button, ps_done_button;
static Panel_item	ps_filename_text, ps_label_text, ps_message;
static Panel_item	ps_time_cycle;

static int	ps_file_button_proc(), ps_printer_button_proc(), ps_done();

char *GR_logo=NULL;

/*-----------------------------------------------------------------------
 * Function:	GR_show_ps_frame
 * Description:	Display the PostScript frame
 * Args:	(none)
 */
int
GR_show_ps_frame()
{
  /* If we haven't created the ps frame yet, do so now */
  if (GR_ps_frame == NULL)
    initialize_ps_frame();

  /* Clear any message currently in the message area */
  show_ps_message("");

  /* Now make it visible */
  window_set(GR_ps_frame, WIN_SHOW, TRUE, 0);
}

/*-----------------------------------------------------------------------
 * Function:	initialize_ps_frame
 * Description:	Initialize the PostScript frame
 * Args:	(none)
 */
static int
initialize_ps_frame()
{
  GR_ps_frame =
    window_create(GR_base_frame, FRAME,
		  WIN_X, 457,
		  WIN_Y, 0,
		  WIN_FONT, GR_regular_font,
		  FRAME_LABEL, "PostScript",
		  FRAME_SHOW_LABEL, TRUE,
		  0);
  if (GR_ps_frame == NULL) {
    GR_error("I can't create any more windows !");
    return(1);
  }
  
  GR_ps_panel =
    window_create(GR_ps_frame, PANEL,
		  WIN_X, 0,
		  WIN_Y, 0,
		  WIN_FONT, GR_regular_font,
		  WIN_WIDTH, 352,
		  WIN_HEIGHT, 142,
		  0);
  if (GR_ps_panel == NULL) {
    GR_error("I can't create any more windows !");
    return(1);
  }
  
  ps_printer_button =
    panel_create_item(GR_ps_panel, PANEL_BUTTON, 
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 12,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(GR_ps_panel, "Printer", 8, 0),
		      PANEL_NOTIFY_PROC, ps_printer_button_proc,
		      0);
  
  ps_file_button =
    panel_create_item(GR_ps_panel, PANEL_BUTTON, 
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 44,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(GR_ps_panel, "File", 5, 0),
		      PANEL_NOTIFY_PROC, ps_file_button_proc,
		      0);

  ps_label_text =
    panel_create_item(GR_ps_panel, PANEL_TEXT, 
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 72,
		      PANEL_VALUE_DISPLAY_LENGTH, LABEL_DISPLAY_LENGTH,
		      PANEL_VALUE_STORED_LENGTH, LABEL_STORED_LENGTH,
		      PANEL_LABEL_STRING, "Label:",
		      0);
  
  ps_filename_text =
    panel_create_item(GR_ps_panel, PANEL_TEXT, 
		      PANEL_ITEM_X, 70,
		      PANEL_ITEM_Y, 47,
		      PANEL_VALUE_DISPLAY_LENGTH, FILENAME_DISPLAY_LENGTH,
		      PANEL_VALUE_STORED_LENGTH, FILENAME_STORED_LENGTH,
		      PANEL_LABEL_STRING, "Filename:",
		      0);

  ps_time_cycle = 
    panel_create_item(GR_ps_panel, PANEL_CYCLE,
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 95,
		      PANEL_LABEL_STRING, "Time on Picture",
		      PANEL_CHOICE_STRINGS,
		        "No",  /* panel_value = 0 */
		        "Yes", /* panel_value = 1 */
		        0,
		      PANEL_VALUE, 1,
		      0);
  
  ps_done_button =
    panel_create_item(GR_ps_panel, PANEL_BUTTON, 
		      PANEL_ITEM_X, 258,
		      PANEL_ITEM_Y, 9,
		      PANEL_LABEL_IMAGE,
		      panel_button_image(GR_ps_panel, "Done", 7, 0),
		      PANEL_NOTIFY_PROC, ps_done,
		      0);

  ps_message =
    panel_create_item(GR_ps_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X, 10,
		      PANEL_ITEM_Y, 118,
		      PANEL_LABEL_STRING, "Printer Error Message Here",
		      PANEL_LABEL_FONT, GR_bold_font,
		      0);
  
  window_fit(GR_ps_frame);
  GR_register_interposer(GR_ps_frame);
}

/*-----------------------------------------------------------------------
 * Function:	ps_done
 * Description:	close the PostScript window
 */
static int
ps_done()
{
  window_set(GR_ps_frame, WIN_SHOW, FALSE, 0);
}

/*-----------------------------------------------------------------------
 * Function:	ps_file_button_proc
 * Description:	notify proc for "File" button
 */
static int
ps_file_button_proc()
{
  char *filename;
  char msg[256];

  /* Get the file name */
  filename = (char *)panel_get_value(ps_filename_text);

  /* Confirm overwrite, if necessary */
  if (GR_file_openable(filename, "r")) {
    if (!gr_user_confirm("Overwite existing file?")) {
      show_ps_message("File not written");
      return(0);
    }
  }

  /* Check for write permission */
  if (!GR_file_openable(filename, "w")) {
    sprintf(msg,"I can't write to file \"%s\" !", filename);
    show_ps_error(msg);
    return(0);
  }

  /* Write the file and signal success or failure */
  if (write_postscript_file(filename) == 0) {
    sprintf(msg, "Wrote \"%s\".", filename);
    show_ps_message(msg);
  }
  else {
    sprintf(msg, "Couldn't write \"%s\".", filename);
    show_ps_error(msg);
  }

  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	ps_printer_button_proc
 * Description:	send PostScript to printer
 * Args:	(none)
 */
static int
ps_printer_button_proc()
{
  int r;
  char *mktemp(), *getenv();
  static char template[] = "/tmp/grXXXXXX";
  char filename[sizeof(template)+1];
  char command[256], *spool_command;

  /* Make sure we are able to print PostScript files */
  /*   First check for presence of GR_SPOOL environment variable */
  spool_command = getenv("GR_SPOOL");
  if (spool_command == NULL) {
    /* Then check for default spool command given at compile time */
    if (strlen(DEFAULT_SPOOL_COMMAND) > 0) {
      spool_command = DEFAULT_SPOOL_COMMAND;
    }
    else {
      /* We can't print! */
      show_ps_error("Don't know how to print PostScript !");
      return(1);
    }
  }

  /* Get name for temporary file for PostScript output */
  strcpy(filename, template);
  mktemp(filename);
  if (!GR_file_openable(filename, "w")) {
    show_ps_error("Can't write temporary file !");
  }
  else {

    /* Write the file */
    if (write_postscript_file(filename) != 0) {
      show_ps_error("Error in writing temporary file !");
    }
    
    else {

      /* Send the file to the printer */
      sprintf(command, "%s %s", spool_command, filename);
      if (system(command) == 0)
	show_ps_message("PostScript image sent to printer");
      else {
	show_ps_error("Error in sending PostScript to printer !");
    }
    
    /* Now delete the temporary file */
    sprintf(command,"/bin/rm -f %s", filename);
    system(command);

    }
  }
}

/*-----------------------------------------------------------------------
 * Function:	write_postscript_file
 * Description:	write PostScript to a file
 * Args  IN:	filename: name of file to write to
 * Notes:	This calls the value of GR_redraw_proc to cause the
 *		canvas to be redrawn in "postscript" mode.  This procedure
 *		should have been specified in the initial call to
 *		gr_print_button.
 */
static int
write_postscript_file(filename)
     char *filename;
{
  int r;
  char *label, *timeptr, timestr[30];
  long clock;
  extern char *ctime();

  /* Get label */
  label = (char*)panel_get_value(ps_label_text);
  if (strlen(label) == 0) label=NULL;

  /* Get time string, maybe */
  if (panel_get_value(ps_time_cycle)) {
    /* Time cycle is "Yes", so do use time string */
    clock = time(0);
    strcpy(timestr, ctime(&clock));    /* (26 chars with final newline) */
    timestr[strlen(timestr)-1] = '\0'; /* (this takes out final newline) */
    timeptr = timestr;
  }
  else
    timeptr = NULL;
    
  if (gr_postscript_begin(filename, GR_logo, label, timeptr) != 0) return(1);
  if (GR_redraw_proc == NULL) return(1);
  r = (*GR_redraw_proc)();
  r |= gr_postscript_end();
  return(r);
}

/*-----------------------------------------------------------------------
 * Function:	show_ps_message
 * Description:	display a string in the message area of the ps panel
 * Args  IN:	s: the string to be displayed
 * Notes:	Only the first PS_MESSAGE_LENGTH chars are displayed
 */
static int
show_ps_message(s)
char *s;
{
  return(GR_show_message(ps_message, s, PS_MESSAGE_LENGTH));
}

/*-----------------------------------------------------------------------
 * Function:	show_ps_error
 * Description:	display a string in the message area of the ps panel
 *		  and beep the bell
 * Args  IN:	s: the string to be displayed
 * Notes:	Only the first PS_MESSAGE_LENGTH chars are displayed
 */
static int
show_ps_error(s)
char *s;
{
  window_bell(GR_ps_panel);
  window_bell(GR_ps_panel);
  return(GR_show_message(ps_message, s, PS_MESSAGE_LENGTH));
}
