(* $Id: GraphicsNames.m,v 2.0 89/10/03 21:38:34 mbp Exp $
 *    
 * GraphicsNames.m: functions for keeping track of names of graphical
 *   objects
 *)

(***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************)

(*
 * This package is for use with an external graphics program "with
 * memory".  This means that the program assigns a name to each object
 * it draws, and this name can be used later to perform various
 * graphical operations (such as erase, hide, unhide) on the object.
 *
 * The names generated by the external graphics program are usually
 * meaningless to the user (typically they are just numbers), so it is
 * desirable to keep them at a level that the user never sees.
 * Furthermore, a good interactive graphics program will let the user
 * specify her own names for objects.  This means that internally, the
 * program must perform a mapping between the user-specified names and
 * program-generated names.  The purpose of this package is to perform
 * this mapping.  The mapping functions are:
 * 
 *                           GraphicsName
 *          user-     -------------------------->  graphics
 *        specified                                 object
 *          name     <--------------------------     name
 * 			    UserName
 * 
 * The function SetGraphicsName sets up the correspondence between a
 * particular user-specified name and the corresponding graphics
 * object name.
 * 
 * The user-specified names are simply Mathematica symbols, such as
 * 'x'.  These symbols generally have values, such as x = Point[1,2].
 * The association of a graphics object name with the 'user-specified'
 * name 'x' needs to be done in a way that does not interfere with the
 * value of x.  This is not a trivial task, which is why we have a
 * package for doing this.
 *)

BeginPackage["GraphicsNames`"]

(*----------------------------------------------------------------------*)

SetGraphicsName::usage = "SetGraphicsName[x, s] associates the graphics
object name s with the user-specified name x.  x can be any lvalue
(anything which could legally appear on the left-hand side of an
assignment).  s can be any expression, but is usually a string or an
integer."

GraphicsName::usage = "GraphicsName[x] gives the name of the graphics
object associated with x."

UserName::usage = "UserName[s] gives the HoldForm of the
user-specified name corresponding to the object with graphics name s.
The HoldForm is returned to prevent the expression from being
evaluated."

GraphicsNamedQ::usage = "GraphicsNamedQ[x] returns True if x has been
given a graphics name with SetGraphicsName, and False otherwise."

UnsetGraphicsName::usage = "UnsetGraphicsName[x] removes the graphics
name for x which was defined with SetGraphicsName."

ClearGraphicsNames::usage = "ClearGraphicsNames[] removes all currently
defined graphics names."

(*----------------------------------------------------------------------*)

Begin["`Private`"]

HoldLevel0::usage = "HoldLevel0[expr] returns HoldForm of expr with
all levels below level 0 evaluated.  Warning: this currently works
only when expr is either an atom or an expression whose head is an
atom.  (It won't work, for example, if expr = x[1][2]."

SetAttributes[HoldLevel0, HoldAll]

HoldLevel0[x_Symbol[args___]] :=
  Block[{evargs},
    evargs = args;
      If[Length[{evargs}]>0,
	HoldForm[x[##]]&[evargs],
	HoldForm[x[]]]]

HoldLevel0[x_Symbol] :=
  HoldForm[x]

Initialize::usage = "Initialize[] sets up the initial definitions for
the function GraphicsName function."

Initialize[] := (
  SetAttributes[GraphicsName, HoldFirst];
  GraphicsName[x_] :=
    GraphicsName[Release[HoldLevel0[x]]] /; Head[x] =!= HoldForm;
  )

Initialize[]

SetAttributes[SetGraphicsName, HoldFirst]

SetGraphicsName[x_, name_] := (
  UnsetGraphicsName[x];
  UserName[name] = HoldLevel0[x];
  GraphicsName[Release[UserName[name]]] = name
  )

SetAttributes[UnsetGraphicsName, HoldAll]

UnsetGraphicsName[x_] :=
  If[ GraphicsNamedQ[x],
      UserName[GraphicsName[x]] =. ;
      GraphicsName[Release[HoldLevel0[x]]] =.; ]

SetAttributes[GraphicsNamedQ, HoldAll]

GraphicsNamedQ[x_] :=
  Head[GraphicsName[x]] =!= GraphicsName

ClearGraphicsNames[] := (
  Clear[GraphicsName];
  Clear[UserName];
  Initialize[];
  Null )

End[]

EndPackage[]

Null
