
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/************************************************************************
*
*  File:  metric.c
*
*  Contents: Functions to calculate area, energy and their gradients
*            according to the LINEAR STRING model.
*            With background metric.
*/

#include "include.h"

double euclidean_area;   /* euclidean area for conformal metrics */

/************************************************************************
*
*  Calculates all forces on control points due to edge and
*  accumulates them at each control point.
*/

void edge_force_l_metric(e_id)
edge_id e_id;
{
  REAL v[MAXCOORD],len,f,fp,*tforce,*hforce;
  int i,j,k;
  vertex_id tv = get_edge_tailv(e_id);
  vertex_id hv = get_edge_headv(e_id);
  REAL density = get_edge_density(e_id);
  REAL g[MAXCOORD][MAXCOORD];
  REAL g_partial[MAXCOORD][MAXCOORD][MAXCOORD];
  REAL *xt=get_coord(tv);
  REAL *xh=get_coord(hv);
  REAL midx[MAXCOORD];
  REAL gg,gg_partial[MAXCOORD];

  /* force due to linear tension, metric evaluated at midpoint */
  for ( i = 0 ; i < web.sdim ; i++ )
    { midx[i] = (xt[i] + xh[i])/2;
      v[i] = xh[i] - xt[i];
    }
  if ( web.conformal_flag )
   { eval_all(&web.metric[0][0],midx,web.sdim,&gg,gg_partial);
     len = gg*dot(v,v,web.sdim);
   }
  else
   { for ( i = 0 ; i < web.sdim ; i++ )
      for ( j = 0 ; j < web.sdim ; j++ )
        eval_all(&web.metric[i][j],midx,web.sdim,&g[i][j],
	       g_partial[i][j]);
     for ( len = 0.0, i = 0 ; i < web.sdim ; i++ )
      for ( j = 0 ; j < web.sdim ; j++ )
        len += v[i]*g[i][j]*v[j];
    }
  len = sqrt(len);
  tforce = get_force(tv);
  hforce = get_force(hv);
  for ( k = 0 ; k < web.sdim ; k++ )
    { if ( web.conformal_flag )
	{ fp = gg_partial[k]*dot(v,v,web.sdim);
	  f  = gg*v[k];
	}
      else
       for ( f = fp = 0.0, i = 0 ; i < web.sdim ; i++ )
        { for ( j = 0 ; j < web.sdim ; j++ )
	    fp += g_partial[i][j][k]*v[i]*v[j]/4;
	  f += g[k][i]*v[i];
	}
      tforce[k] += density*(f-fp)/len;
      hforce[k] -= density*(f+fp)/len;
   }
  set_edge_length(e_id,len);

  /* following two lines for area normalization option */
  add_vertex_star(tv,len);
  add_vertex_star(hv,len);

}


/************************************************************************
*
*  Returns energy due to one edge.
*
*/

void edge_energy_l_metric(e_id)
edge_id e_id;
{
  REAL energy;
  REAL midx[MAXCOORD];
  int i,j;
  vertex_id tv = get_edge_tailv(e_id);
  vertex_id hv = get_edge_headv(e_id);
  REAL *xt=get_coord(tv);
  REAL *xh=get_coord(hv);
  REAL v[MAXCOORD];
  double euclidean;

  /* energy due to linear tension, metric evaluated at midpoint */
  for ( i = 0 ; i < web.sdim ; i++ )
    { midx[i] = (xt[i] + xh[i])/2;
      v[i] = xh[i] - xt[i];
    }
  if ( web.conformal_flag )
    { double gg = eval(&web.metric[0][0],midx);
      euclidean = dot(v,v,web.sdim);
      energy = gg*euclidean;
    }
  else
   for ( energy = 0.0, i = 0 ; i < web.sdim ; i++ )
    for ( j = 0 ; j < web.sdim ; j++ )
      energy += v[i]*v[j]*eval(&web.metric[i][j],midx);
  energy = sqrt(energy);

  if ( web.dimension == STRING )
     web.total_area   += energy;   /* don't count triple junction as area */
  if ( web.conformal_flag )
    { euclidean_area += sqrt(euclidean);
    }
  /* accumulate star area around each vertex to scale motion */
    { add_vertex_star(hv,energy);
      add_vertex_star(tv,energy);
    }


  energy *= get_edge_density(e_id);
  web.total_energy += energy;

}

/************************************************************************
*
*  Function: edge_energy_q_metric()
*
   Purpose:  Finds energy due to one edge in metric model.
*
*  Quadratic version.
*/

void edge_energy_q_metric(e_id)
edge_id e_id;
{
  REAL *pt[EDGE_CTRL];
  REAL tang[MAXCOORD];
  vertex_id v[EDGE_CTRL];
  int i,j,k;
  REAL gpt[MAXCOORD];
  REAL len;
  double euclidean;

  v[0] = get_edge_tailv(e_id);
  v[1] = get_edge_midv(e_id);
  v[2] = get_edge_headv(e_id);
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    {
      pt[i] = get_coord(v[i]);
    }
    
  /* calculate tangents at integration points and accumulate */
  for ( i = 0 ; i < EDGE_INTERP ; i++ )
    {
      for ( j = 0 ; j < web.sdim ; j ++ )
        {
          tang[j] = 0.0;
          gpt[j] = 0.0;
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            { tang[j] += sdip[k][i]*pt[k][j];
              gpt[j] += gcombo[k][i]*pt[k][j];
            }
        }
      if ( web.conformal_flag )
	{ euclidean = eval(&web.metric[0][0],gpt);
	  len = euclidean*dot(tang,tang,web.sdim);
	}
      else
        for ( len = 0.0, k = 0 ; k < web.sdim ; k++ )
         for ( j = 0 ; j < web.sdim ; j++ )
          len += tang[k]*tang[j]*eval(&web.metric[k][j],gpt);
      len = gauss2wt[i]*sqrt(len);
      if ( web.conformal_flag )
	euclidean_area += gauss2wt[i]*sqrt(euclidean);
      if ( web.dimension == STRING )
        { web.total_area   += len;   /* don't count triple junction as area */
          /* accumulate  area around each vertex to scale motion */
            { add_vertex_star(v[0],len);
              add_vertex_star(v[1],len);
              add_vertex_star(v[2],len);
	    }
         }

      len *= get_edge_density(e_id);
      web.total_energy += len;
    }
    
  return;
}

/************************************************************************
*
*  Function: edge_force_q_metric()
*
   Purpose:  Finds force due to one edge in metric model.
*
*  Quadratic version.
*/

void edge_force_q_metric(e_id)
edge_id e_id;
{
  REAL *pt[EDGE_CTRL];
  REAL tang[MAXCOORD];
  vertex_id v[EDGE_CTRL];
  int i,j,k,m,n;
  REAL gpt[MAXCOORD];
  REAL g[MAXCOORD][MAXCOORD];
  REAL g_partial[MAXCOORD][MAXCOORD][MAXCOORD];
  REAL *force[EDGE_CTRL];
  REAL f,fp,fudge;
  REAL density = get_edge_density(e_id);
  REAL len;
  double vv;
        
  v[0] = get_edge_tailv(e_id);
  v[1] = get_edge_midv(e_id);
  v[2] = get_edge_headv(e_id);
  for ( i = 0 ; i < EDGE_CTRL ; i++ )
    {
      pt[i] = get_coord(v[i]);
      force[i] = get_force(v[i]);
    }
    
  /* calculate tangents at integration points and accumulate */
  for ( i = 0 ; i < EDGE_INTERP ; i++ )
    {
      for ( j = 0 ; j < web.sdim ; j ++ )
        {
          tang[j] = 0.0;
          gpt[j] = 0.0;
          for ( k = 0 ; k < EDGE_CTRL ; k++ )
            { tang[j] += sdip[k][i]*pt[k][j];
              gpt[j] += gcombo[k][i]*pt[k][j];
            }
        }
      if ( web.conformal_flag )
       {
          eval_all(&web.metric[0][0],gpt,web.sdim,&g[0][0],
	       g_partial[0][0]);
	  vv = dot(tang,tang,web.sdim);
	  len = vv*g[0][0];
       }
      else
       {
         for ( k = 0 ; k < web.sdim ; k++ )
           for ( j = 0 ; j < web.sdim ; j++ )
             eval_all(&web.metric[k][j],gpt,web.sdim,&g[k][j],
	       g_partial[k][j]);
         for ( len = 0.0, k = 0 ; k < web.sdim ; k++ )
           for ( j = 0 ; j < web.sdim ; j++ )
             len += tang[k]*g[k][j]*tang[j];
	}
      len = sqrt(len);
      if ( len == 0.0 ) continue;
      fudge = density*gauss2wt[i]/len/2;
      for ( m = 0 ; m < EDGE_CTRL ; m++ )
        for ( n = 0 ; n < web.sdim ; n++ )
          { if ( web.conformal_flag )
	      { fp = g_partial[0][0][n]*vv;
	        f  = g[0][0]*v[n];
              }
	    else
	     for ( f = fp = 0.0, k = 0 ; k < web.sdim ; k++ )
              { for ( j = 0 ; j < web.sdim ; j++ )
	          fp += g_partial[k][j][n]*v[k]*v[j];
	        f += g[n][k]*v[k];
              }
            force[m][n] -= fudge*(2*sdip[m][i]*f + gcombo[m][i]*fp);
	  }
   }

  return;
}

/**********************************************************************
*
*  function: simplex_energy_metric()
*
*  purpose:  universal simplex area calculator.  Does both linear
*            and quadratic models.
*
*  return value: area of simplex. caller must multiply by facet density.
*
*  globals used: web.dimension
*                web.sdim
*                gauss_wt
*                gauss_pt
*                gpoly
*                gpolypartial
*                ctrl_num
*                gauss_num
*
**********************************************************************/

double simplex_energy_metric(v,pt)
vertex_id *v; /* list of vertices */
REAL **pt;  /* pointer to list of vertex coords for simplex */
	       /* order must agree with that used to set up gauss arrays */
{ int i,j,k;
  REAL gpt[MAXCOORD];
  REAL area = 0.0;  /* total */
  double new_area;

  /* calculate integrands at integration points and accumulate */
  for ( k = 0 ; k < gauss_num ; k++ )
    {
      /* get gauss pt */
      vec_mat_mul(gpoly[k],pt,gpt,ctrl_num,web.sdim);

      /* get tangents */
      mat_mult(gpolypartial[k],pt,tang,web.dimension,ctrl_num,
						 web.sdim);

      /* evaluate metric and fill in determinant */
      if ( web.conformal_flag )
       { metric[0][0] = eval(&web.metric[0][0],gpt);
         for (  i = 0 ; i < web.dimension ; i++ )
           for ( j = 0 ; j <= i ; j++ )
	     det_array[i][j] = det_array[j][i] =
	       metric[0][0]*dot(tang[i],tang[j],web.sdim);
       }
      else
       { for (  i = 0 ; i < web.sdim ; i++ )
          for ( j = 0 ; j < web.sdim ; j++ )
           metric[i][j] = eval(&web.metric[i][j],gpt);
         for (  i = 0 ; i < web.dimension ; i++ )
           for ( j = 0 ; j <= i ; j++ )
	     det_array[i][j] = det_array[j][i] =
	       quadratic_form(tang[i],metric,tang[j],web.sdim);
       }
/*
printf("\ntangents\n"); print_matrix(tang,web.dimension,web.sdim);
printf("tangent products\n"); print_matrix(det_array,web.dimension,web.dimension);
*/			    
      /* evaluate determinant and add to total */
      new_area = gausswt[k]*sqrt(determinant(det_array,web.dimension));
      area += new_area;
      if ( web.conformal_flag )
	euclidean_area += new_area/pow(metric[0][0],web.dimension/2.0)
	                   /simplex_factorial;
    }

  area /= simplex_factorial;

  /* accumulate area around each vertex to scale motion */
    for ( i = 0 ; i < ctrl_num ; i++ )
      add_vertex_star(v[i],area);

  return area;
}


/**********************************************************************
*
*  function: simplex_force_metric()
*
*  purpose:  Universal simplex force calculator.  Does both linear
*            and quadratic models.  Adds forces to vertices.
*
*  globals used: web.dimension
*                web.sdim
*                gauss_wt
*                gauss_pt
*                gpoly
*                gpolypartial
*                ctrl_num
*                gauss_num
*
**********************************************************************/

void simplex_force_metric(v,pt,density,forces)
vertex_id *v;  /* pointer to list of vertex ID's for simplex */
	       /* order must agree with that used to set up gauss arrays */
REAL **pt;     /* coords to used (unwrapped ) */
double density;  /* surface tension of facet */
REAL **forces;
{ int i,j,k,mu,nu,m,n;
  REAL gpt[MAXCOORD];
  REAL area = 0.0;  /* total */
  REAL tempvec1[MAXCOORD],tempvec2[MAXCOORD];
  REAL *force[MAXCTRL];
  REAL qsum[MAXCOORD];
  double vv[MAXCOORD][MAXCOORD];

  /* calculate integrands at integration points and accumulate */
  for ( k = 0 ; k < gauss_num ; k++ )
    {
      /* get gauss pt */
      vec_mat_mul(gpoly[k],pt,gpt,ctrl_num,web.sdim);

      /* get tangents */
      mat_mult(gpolypartial[k],pt,tang,web.dimension,ctrl_num,
						 web.sdim);

      /* evaluate metric and fill in determinant */
      if ( web.conformal_flag )
       { eval_all(&web.metric[0][0],gpt,web.sdim,&metric[0][0],
	       metric_partial[0][0]);
         for (  i = 0 ; i < web.dimension ; i++ )
           for ( j = 0 ; j <= i ; j++ )
	     { vv[i][j] = vv[j][i] = dot(tang[i],tang[j],web.sdim);
	       det_array[i][j] = det_array[j][i] = metric[0][0]*vv[i][j];
	     }
       }
      else
       { for (  i = 0 ; i < web.sdim ; i++ )
          for ( j = 0 ; j < web.sdim ; j++ )
            eval_all(&web.metric[i][j],gpt,web.sdim,&metric[i][j],
	       metric_partial[i][j]);
         for (  i = 0 ; i < web.dimension ; i++ )
           for ( j = 0 ; j <= i ; j++ )
	     det_array[i][j] = det_array[j][i] =
	       quadratic_form(tang[i],metric,tang[j],web.sdim);
       }

      area = sqrt(det_adjoint(det_array,web.dimension));
      /* det_array now has adjoint transpose */

      for (  i = 0 ; i < web.dimension ; i++ )
        for ( j = 0 ; j < web.dimension ; j++ )
          { double factor = 
               gausswt[k]*density*det_array[j][i]/2/area/simplex_factorial;

	    if ( web.conformal_flag )
	      { for ( m = 0 ; m < web.sdim ; m++ )
		 { tempvec1[m] = metric[0][0]*tang[i][m];
		   tempvec2[m] = metric[0][0]*tang[j][m];
		 }
                for ( n = 0 ; n < web.sdim ; n++ )
		  qsum[n] = metric_partial[0][0][n]*vv[i][j];
              }
	    else
	      {
	        matvec_mul(metric,tang[i],tempvec1,web.sdim,
	                                              web.sdim);
                matvec_mul(metric,tang[j],tempvec2,web.sdim,
                                                      web.sdim);
                for ( n = 0 ; n < web.sdim ; n++ )
	         for ( qsum[n] = 0.0,mu = 0 ; mu < web.sdim ; mu++ )
	          for ( nu = 0 ; nu < web.sdim ; nu++ )
		   qsum[n] += tang[i][mu]*metric_partial[mu][nu][n]*tang[j][nu];
	       }

	     for ( m = 0 ; m < ctrl_num ; m++ )
	       for ( mu = 0 ; mu < web.sdim ; mu++ )
                 { forces[m][mu] -= factor* 
		       (gpolypartial[k][j][m]*tempvec1[mu]
			  + gpolypartial[k][i][m]*tempvec2[mu]
			  + gpoly[k][m]*qsum[mu]);
		 }
	   } 
    }

  return;
}

