/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/**********************************************************************
*
*  File: kusner.c
*
*  Purpose: Does calculations needed for including square curvature
*           in energy. Linear model only.
*           Rob Kusner version - curvature energy on edges
*           Also square gaussian curvature energy only. 
*/

#include "include.h"

/********************************************************************
*
*  Function: kusner_energy()
*
*  Purpose:  Does square curvature energy calculation for an edge.
*            Edge curvature version.
*
*/

void kusner_energy()
{ 
  REAL s1[MAXCOORD],s2[MAXCOORD],t2[MAXCOORD];
  double s1s1,s1s2,s1t2,s2s2,t2t2,s2t2;
  double a1,a2;
  double det;
  facetedge_id fe_s1,fe_s2,fe_t2;
  edge_id e_id;
  double cos_th;   /* cosine of angle between facets */
  double modulus = 3*web.params[square_curvature_param].value;

  FOR_ALL_EDGES(e_id)
   {
     if ( get_attr(e_id) & FIXED ) continue;

     /* get edge vectors away from tail vertex */
     fe_s1 = get_edge_fe(e_id);
     fe_s2 = get_prev_edge(get_next_facet(fe_s1));
     fe_s2 = inverse_id(fe_s2);
     fe_t2 = get_prev_edge(fe_s1);
     fe_t2 = inverse_id(fe_t2);
     get_fe_side(fe_s1,s1);
     get_fe_side(fe_s2,s2);
     get_fe_side(fe_t2,t2);

     s1s1 = dot(s1,s1,web.sdim);
     s1s2 = dot(s1,s2,web.sdim);
     s1t2 = dot(s1,t2,web.sdim);
     t2t2 = dot(t2,t2,web.sdim);
     s2s2 = dot(s2,s2,web.sdim);
     s2t2 = dot(s2,t2,web.sdim);

     det = s1s1*t2t2 - s1t2*s1t2;
     a1 = sqrt(det);
     det = s1s1*s2s2 - s1s2*s1s2;
     a2 = sqrt(det);

     cos_th = (s1s2*s1t2 - s2t2*s1s1)/a1/a2;

     web.total_energy += modulus*s1s1*(1 - cos_th)/(a1 + a2);
   }
}


/************************************************************************
*
*  Function: kusner_force()
*
*  Purpose:  Does square curvature force calculation.
*
*/

void kusner_force()
{ 
  REAL s1[MAXCOORD],s2[MAXCOORD],t2[MAXCOORD];
  double s1s1,s1s2,s1t2,s2s2,t2t2,s2t2;
  double a1,a2;
  double det;
  facetedge_id fe_s1,fe_s2,fe_t2;
  edge_id e_id;
  double cos_th;   /* cosine of angle between facets */
  REAL *of,*s1f,*s2f,*t2f;  /* vertex force pointers */
  double dcosds1[MAXCOORD],dcosds2[MAXCOORD],dcosdt2[MAXCOORD];
  double da1ds1[MAXCOORD], da1dt2[MAXCOORD];
  double da2ds1[MAXCOORD], da2ds2[MAXCOORD];
  int i;
  double modulus = 3*web.params[square_curvature_param].value;

  FOR_ALL_EDGES(e_id)
   {
     if ( get_attr(e_id) & FIXED ) continue;

     /* get edge vectors away from tail vertex */
     fe_s1 = get_edge_fe(e_id);
     fe_s2 = get_prev_edge(get_next_facet(fe_s1));
     fe_s2 = inverse_id(fe_s2);
     fe_t2 = get_prev_edge(fe_s1);
     fe_t2 = inverse_id(fe_t2);
     get_fe_side(fe_s1,s1);
     get_fe_side(fe_s2,s2);
     get_fe_side(fe_t2,t2);
     of = get_force(get_edge_tailv(e_id));
     s1f = get_force(get_edge_headv(e_id));
     s2f = get_force(get_fe_headv(fe_s2));
     t2f = get_force(get_fe_headv(fe_t2));


     s1s1 = dot(s1,s1,web.sdim);
     s1s2 = dot(s1,s2,web.sdim);
     s1t2 = dot(s1,t2,web.sdim);
     t2t2 = dot(t2,t2,web.sdim);
     s2s2 = dot(s2,s2,web.sdim);
     s2t2 = dot(s2,t2,web.sdim);

     det = s1s1*t2t2 - s1t2*s1t2;
     a1 = sqrt(det);
     det = s1s1*s2s2 - s1s2*s1s2;
     a2 = sqrt(det);

     cos_th = (s1s2*s1t2 - s2t2*s1s1)/a1/a2;

     /* gradients of various terms */
     for ( i = 0 ; i < web.sdim ; i++ )
       { da1ds1[i] = 0.5/a1*(2*s1[i]*t2t2 - 2*s1t2*t2[i]);
         da1dt2[i] = 0.5/a1*(2*s1s1*t2[i] - 2*s1t2*s1[i]);
         da2ds1[i] = 0.5/a2*(2*s1[i]*s2s2 - 2*s1s2*s2[i]);
         da2ds2[i] = 0.5/a2*(2*s1s1*s2[i] - 2*s1s2*s1[i]);
         dcosds1[i] = (s2[i]*s1t2 + s1s2*t2[i] - 2*s2t2*s1[i])/a1/a2
                     - cos_th/a1*da1ds1[i] - cos_th/a2*da2ds1[i];
         dcosdt2[i] = (s1s2*s1[i] - s2[i]*s1s1)/a1/a2
                     - cos_th/a1*da1dt2[i];
         dcosds2[i] = (s1[i]*s1t2 - t2[i]*s1s1)/a1/a2
                     - cos_th/a2*da2ds2[i];
       }

     for ( i = 0 ; i < web.sdim ; i++ )
       { double f;  /* part of force */

         f = modulus*2*s1[i]*(1 - cos_th)/(a1 + a2);
         s1f[i] -= f;
         of[i]  += f;

         f = -modulus*s1s1*dcosds1[i]/(a1 + a2);
         s1f[i] -= f;
         of[i]  += f;

         f = -modulus*s1s1*(1 - cos_th)/(a1 + a2)/(a1 + a2)*da1ds1[i];
         s1f[i] -= f;
         of[i]  += f;

         f = -modulus*s1s1*(1 - cos_th)/(a1 + a2)/(a1 + a2)*da2ds1[i];
         s1f[i] -= f;
         of[i]  += f;

         f = -modulus*s1s1*dcosds2[i]/(a1 + a2);
         s2f[i] -= f;
         of[i]  += f;

         f = -modulus*s1s1*(1 - cos_th)/(a1 + a2)/(a1 + a2)*da2ds2[i];
         s2f[i] -= f;
         of[i]  += f;

         f = -modulus*s1s1*dcosdt2[i]/(a1 + a2);
         t2f[i] -= f;
         of[i]  += f;

         f = -modulus*s1s1*(1 - cos_th)/(a1 + a2)/(a1 + a2)*da1dt2[i];
         t2f[i] -= f;
         of[i]  += f;
       }
   }
}


/********************************************************************
*
*  Function: sqgauss_energy()
*
*  Purpose:  Does square gauss curvature energy calculation for a vertex.
*
*/

void sqgauss_energy()
{ 
  REAL s1[MAXCOORD],s2[MAXCOORD],s3[MAXCOORD],s4[MAXCOORD];
  REAL n1[MAXCOORD],n2[MAXCOORD],n3[MAXCOORD];
  REAL m1[MAXCOORD],m2[MAXCOORD];
  double a1,a2,a3,area=0.0;
  facetedge_id fe_s1,fe_s2,fe_s3;
  vertex_id v_id;
  double co,si;  /* cosine, sine of angle between cone normals */
  int j;
  double gc;  /* gaussian curvarture */
  double sumth; /* total turning angle */
  double ca,cb,cc,sa,sb,sc;  

  FOR_ALL_VERTICES(v_id)
   {
     /* go in order around vertex, assuming planar topology */
     fe_s1 = get_vertex_fe(v_id);
     get_fe_side(fe_s1,s1);
     fe_s2 = get_prev_edge(fe_s1);
     fe_s2 = get_next_facet(fe_s2);
     fe_s2 = inverse_id(fe_s2);
     get_fe_side(fe_s2,s2);
     cross_prod(s1,s2,n1);
     fe_s2 = get_prev_edge(fe_s2);
     fe_s2 = get_next_facet(fe_s2);
     fe_s2 = inverse_id(fe_s2);
     get_fe_side(fe_s2,s3);
     cross_prod(s2,s3,n2);
     gc = 0.0;
     a1 = sqrt(dot(n1,n1,web.sdim));
     a2 = sqrt(dot(n2,n2,web.sdim));
     area = a1/2 + a2/2;
     do
       { 
         fe_s3 = get_prev_edge(fe_s2);
         fe_s3 = get_next_facet(fe_s3);
         fe_s3 = inverse_id(fe_s3);
         get_fe_side(fe_s3,s4);
         cross_prod(s3,s4,n3);
         a3 = sqrt(dot(n3,n3,web.sdim));
         area += a3/2;
	 ca = dot(n1,n2,web.sdim)/a1/a2;
	 cb = dot(n1,n3,web.sdim)/a1/a3;
	 cc = dot(n2,n3,web.sdim)/a2/a3;
	 sa = sqrt(1-ca*ca);
	 sb = sqrt(1-cb*cb);
	 sc = sqrt(1-cc*cc);
	 gc += acos((ca-cb*cc)/sb/sc) + acos((cb-ca*cc)/sa/sc)
	       + acos((cc-ca*cb)/sa/sb) - M_PI;
         for ( j = 0 ; j < web.sdim ; j++ ) s3[j] = s4[j];
         for ( j = 0 ; j < web.sdim ; j++ ) n2[j] = n3[j];
         a2 = a3;
         fe_s2 = fe_s3;
       }
     while ( !equal_id(fe_s2,fe_s1));

     web.total_energy += (gc/(area/3))*(gc/(area/3))*(area/3);;
   }
}

/**************************************************************************
*
* function:  approx_curvature()
*
* purpose: calculate approximate curvature according to Dziuk and Schmidt.
*          Their definition is that linear interpolation of approx curvature
*          integrated with linear interpolation of velocity gives rate of 
*          area change.  So first need regular vertex force, and then
*          can solve symmetric system for approximate curvatures.
*          Coefficient matrix has star areas times 1/6 along diagonal and difacet
*          areas off diagonal times 1/12
*/


/*  C declarations of the YSMP routines  */
#include "f2c.h"
#ifdef NOPROTO
int odrv_();
int sdrvmd_();
void sdrv_flag_check();
void odrv_flag_check();
#else
int odrv_( integer *, integer *,integer *,double *, integer *,integer *,
integer *,integer *, integer *, integer *);

int sdrvmd_( integer *, integer *,integer *, integer *,integer *,double *,
        double *,double *, integer *,integer *,double *,integer *,
        integer *, integer *, double *);
                                        
void sdrv_flag_check(integer , integer , integer );
void odrv_flag_check(integer , integer );
#endif
#ifdef FULLPROTO
void odrv_( int *N, int *IA,int *JA,double *A, int *P,int *IP,
int *NSP,int *ISP, int *PATH, int *FLAG );

void sdrvmd_( int *N, int *P,int *IP, int *IA,int *JA,double *A,
        double *B,double *Z, int *NSP,int *ISP,double *RSP,int *ESP,
        int *PATH, int *FLAG, double *EMAX );
                                        
void sdrv_flag_check(integer ESP, integer FLAG, integer N);
void odrv_flag_check(integer FLAG, integer N);
#endif

void approx_curvature()
{
  integer *IA=0, *JA=0, *P=0, *IP=0, NSP, *ISP=0;
  doublereal *A=0;
  integer i,j, PATH, FLAG;
  integer total,count;
  doublereal *B;
  double *RSP;
  integer ESP;
  double EMAX;
  integer N = web.skel[VERTEX].max_ord;
  int total_entries = web.skel[VERTEX].count + web.skel[EDGE].count;
  edge_id e_id;

  /* allocate storage for arrays to be passed to ysmp */
  IA = (integer *)temp_calloc(N+1,sizeof(integer));
  JA = (integer *)temp_calloc(total_entries,sizeof(integer));
   A = (double *)temp_calloc(total_entries,sizeof(double));
   P = (integer *)temp_calloc(N,sizeof(integer));
  IP = (integer *)temp_calloc(N,sizeof(integer));
  NSP = 8*N + 16*total_entries;
  ISP = (integer *)temp_calloc(NSP,sizeof(integer));
  B  = (double *)temp_calloc(N,sizeof(double));  

  /* count entries needed for each row */
  FOR_ALL_EDGES(e_id)
   {
     int tail = ordinal(get_edge_tailv(e_id));
     int head = ordinal(get_edge_headv(e_id));
     if ( tail > head ) IA[head]++;
     else IA[tail]++;
   }
  /* set up IA pointers */
  count = 0;
  for ( i = 0 ; i < N ; i++ )
   { int temp = IA[i] + 1;  /* include diagonal element */
     IA[i] = count + 1;  /* FORTRAN indexing */
     count += temp;
     JA[IA[i]-1] = i+1;  /* diagonal */
   }
  IA[N] = count + 1;

  /* set up JA column index list for off diagonal */
  /* and fill in facet star areas */
  FOR_ALL_EDGES(e_id)
   {
     int tail = ordinal(get_edge_tailv(e_id));
     int head = ordinal(get_edge_headv(e_id));
     facetedge_id fe = get_edge_fe(e_id);
     double area1 = get_facet_area(get_fe_facet(fe));
     double area2;
     int base,addend;
     fe = get_next_facet(fe);
     area2 = get_facet_area(get_fe_facet(fe));
     
     /* add to vertex stars */
     A[IA[tail]-1] += (area1 + area2)/12;  /* each area will be added twice */
     A[IA[head]-1] += (area1 + area2)/12;  /* each area will be added twice */

     if ( tail > head ) { base = head; addend = tail; }
     else { base = tail; addend = head; }
     /* seek column, add if not already there */
     for ( j = IA[base]-1 ; j < IA[base+1]-1 ; j++ )
       if ( (JA[j] == addend+1) || (JA[j] == 0) )
          { JA[j] = addend + 1; /* in case first time */
            A[j] += (area1 + area2)/12; 
            break;
          }
     if ( j == IA[base+1]-1 )
       error("approx_curvature: cannot find edge in JA list.\n",RECOVERABLE);
   }
     

#ifdef GDEBUG
/* some debug printing */
{ REAL x;
printf("IA: ");
for ( i = 0 ; i <= N ; i++ ) printf(" %d",IA[i]);
printf("\nJA: ");
for ( i = 0 ; i < count ; i++ ) printf(" %d",JA[i]);
printf("\n");
for ( i = 0 ; i < N ; i++ ) 
  { int j,k,m;
    for ( m = 0 ; m < i ; m++ ) printf("         ");
    for ( m = i,  j = 0, k = IA[i]-1 ; m < N /* j < IA[i+1]-IA[i] */; m++ )
      if ( (m == JA[k]-1) && (k < IA[i+1]-1) )
        { printf(" %f",A[k]); k++; j++; }
      else printf(" %f",0.0);
    printf("\n");
  }
}
#endif

  /*  Call ODRV to perform the reordering on A */
      PATH = 2;        
      odrv_( &N, IA,JA,A, P,IP, &NSP,ISP, &PATH, &FLAG );
      odrv_flag_check(FLAG,N);

  /*  Call SDRVMD to compute the solution */
     {  RSP = (double *) ISP;
        PATH = 5; /* SSF and SNF only */
        sdrvmd_(&N,P,IP,IA,JA,A,B,B,&NSP,ISP,RSP,&ESP, &PATH,&FLAG,&EMAX);
        sdrv_flag_check(ESP,FLAG,N);
     }

  /* each coordinate gives a right side */
  for ( j = 0 ; j < web.sdim ; j++ )
    { vertex_id v_id;
      FOR_ALL_VERTICES(v_id)
        { 
          int vnum = ordinal(v_id);
          B[vnum] = get_force(v_id)[j];
        }
      RSP = (double *) ISP;
      PATH = 3; /* SNS */
      sdrvmd_(&N,P,IP,IA,JA,A,B,B,&NSP,ISP,RSP,&ESP, &PATH,&FLAG,&EMAX);
      sdrv_flag_check(ESP,FLAG,N);
      FOR_ALL_VERTICES(v_id)
        { 
          int vnum = ordinal(v_id);
          get_force(v_id)[j] = B[vnum];
        }
    }


  /* free stuff */
free_stuff:
  temp_free((char *)IA);
  temp_free((char *)A);
  temp_free((char *)JA);
  temp_free((char *)B);
  temp_free((char *)ISP);
  temp_free((char *)IP);
  temp_free((char *)P);
}

/**************************************************************/

/* general mobility routines */

#define FORT
#ifdef FORT
#define A_OFF 1
#else
#define A_OFF 0
#endif

/* structure for storing constraint and boundary vertex basis */
struct basis { vertex_id v_id;   /* which vertex */
	       REAL vvec[MAXCOORD][MAXCOORD];  /* basis vectors */
	       REAL *vec[MAXCOORD];  /* for matrix routines */
	     } *conbasis;
int cb_count; /* how many needed */
struct basis **cb_list;  /* indexed by vertex ordinal */
int *dimf;   /* degrees of freedom of vertices */

/*****************************************************************
*
* function: mobility_setup()
*
* purpose:  sets up sparse arrays for mobility matrix for 
*           approximate curvature.  Subsequent calls to 
*           mobility_mult() transform a formfield to a vectorfield.
*/

static  integer *IA, *JA, *P, *IP, NSP, *ISP;
static  double *RSP;
static  doublereal *A;
static  integer N;
static  double *F;
static  int *IW;
static  int *IA_INV;

void mobility_setup()
{
  integer i,j,k, PATH, FLAG=0;
  integer total,count;
  integer ESP;
  double EMAX;
  int total_entries;
  vertex_id v_id;
  edge_id e_id;
  facet_id f_id;
  int fillsize;
  REAL **proj;  /* normal projection matrix */
  REAL **q,**qq,**qqq; /* basis product matrices */
  struct basis *cb,*cba,*cbb,*cbh,*cbt;
  int dim_a,dim_b;

  proj = dmatrix(0,web.sdim-1,0,web.sdim-1);
  qq = dmatrix(0,web.sdim-1,0,web.sdim-1);
  qqq = dmatrix(0,web.sdim-1,0,web.sdim-1);

  /*************************/
  /* set up conbasis first */
  /*************************/

  /* count vertices on constraints and boundaries */
  cb_count = 0;
  FOR_ALL_VERTICES(v_id)
    { 
      if ( (get_vattr(v_id) & BOUNDARY) || get_v_constraint_state(v_id) )
	 cb_count++;
    }

  /* allocate space */
  cb_list = (struct basis **)temp_calloc(web.skel[VERTEX].max_ord,
					   sizeof(struct basis *));
  dimf = (int *)temp_calloc(web.skel[VERTEX].max_ord,sizeof(int));
  if ( cb_count )
  conbasis = (struct basis *)temp_calloc(cb_count,sizeof(struct basis));
  for ( j = 0 ; j < cb_count ; j++ )
     for( i = 0 ; i < web.sdim ; i++ )
	 conbasis[j].vec[i] = conbasis[j].vvec[i];

  /* fill in */
  cb_count = 0;
  N = 0;
  FOR_ALL_VERTICES(v_id)
    { int ord = ordinal(v_id);
      if ( get_vattr(v_id) & FIXED ) continue;  /* leave dimf as 0 */
      if ( get_vattr(v_id) & BOUNDARY ) 
        dimf[ord] = bdry_basis(v_id,conbasis[cb_count].vec);
      else if ( get_v_constraint_state(v_id) )
	dimf[ord] = constr_basis(v_id,conbasis[cb_count].vec);
      else { N += web.sdim; dimf[ord] = web.sdim;  continue; }
      conbasis[cb_count].v_id = v_id;
      N += dimf[ord];
      cb_list[ordinal(v_id)] = conbasis + cb_count;
      cb_count++;
    }

  total_entries = (N*(web.sdim+1))/2+web.sdim*web.sdim*web.skel[EDGE].count;
  /* allocate storage for arrays to be passed to ysmp */
  IA = (integer *)temp_calloc(N+1,sizeof(integer));
  JA = (integer *)temp_calloc(total_entries,sizeof(integer));
   A = (double *)temp_calloc(total_entries,sizeof(double));
   P = (integer *)temp_calloc(N,sizeof(integer));
  IP = (integer *)temp_calloc(N,sizeof(integer));
  IA_INV = (int *)temp_calloc(web.skel[VERTEX].max_ord,sizeof(integer));

#define EMPTY (-1)
  for ( i = 0 ; i < total_entries ; i++ ) JA[i] = EMPTY;

  /* count entries needed for each row, row size accum in IA */
  k = 0;
  FOR_ALL_VERTICES(v_id) /* diagonal elements */
    { int ord = ordinal(v_id);
      cb = cb_list[ord];
      IA_INV[ord] = k;
      dim_a = dimf[ord];
      for ( i = 0 ; i < dim_a ; i++ )
	IA[k++] = dim_a - i;
    }
  FOR_ALL_EDGES(e_id)
   { int tord = ordinal(get_edge_tailv(e_id));
     int hord = ordinal(get_edge_headv(e_id));
     int tail = IA_INV[tord];
     int head = IA_INV[hord];
     int dim_h = dimf[hord];
     int dim_t = dimf[tord];
     if ( tail > head ) for ( i = 0 ; i < dim_h ; i++ ) IA[head+i] += dim_t;
     else for ( i = 0 ; i < dim_t ; i++ ) IA[tail+i] += dim_h;
   }
  /* set up IA pointers */
  count = 0;
  FOR_ALL_VERTICES(v_id)
   { int ord = ordinal(v_id);
     int dim = dimf[ord];
     i = IA_INV[ord];
     for ( k = 0 ; k < dim ; k++ )
       {
         int temp = IA[i+k];
         IA[i+k] = count + A_OFF;  /* FORTRAN indexing */
         for ( j = 0 ; j < dim - k ; j++ )
	    JA[count + j] = i+k+A_OFF+j;  /* diagonal */
         count += temp;
       }
   }
  IA[N] = count + A_OFF;

  /* set up JA column index list for off diagonal */
  /* and fill in star areas */

  if ( web.dimension == STRING )
  FOR_ALL_EDGES(e_id)
   { int tord = ordinal(get_edge_tailv(e_id));
     int hord = ordinal(get_edge_headv(e_id));
     int tail = IA_INV[tord];
     int head = IA_INV[hord];
     int dim_h = dimf[hord];
     int dim_t = dimf[tord];
     int base,addend;
     double length;
     REAL side[MAXCOORD];
     int orda,ordb;
     
     length = get_edge_length(e_id);
     get_edge_side(e_id,side);

     /* figure normal projection matrix */
     for ( i = 0 ; i < web.sdim ; i++ )
       {
	 if ( effective_area_flag )
          for ( j = i ; j < web.sdim ; j++ )
	   { /* not pure projection, to keep pos def */
             proj[i][j] = - /* 0.99* */ side[i]*side[j]/length; 
           }
         else for ( j = i ; j < web.sdim ; j++ ) proj[i][j] = 0.0;
         proj[i][i] += length;  
       }

     /* add to vertex stars */
     cbh = cb_list[hord];
     if ( cbh )
      { for ( i = 0 ; i < dimf[hord] ; i++ )
         for ( j = i ; j < dimf[hord] ; j++ )
             A[IA[head+i]-A_OFF+(j-i)] += 
	       quadratic_form(cbh->vec[i],proj,cbh->vec[j],web.sdim)/3; 
      }
     else
      { for ( i = 0 ; i < dimf[hord] ; i++ )
         for ( j = i ; j < dimf[hord] ; j++ )
             A[IA[head+i]-A_OFF+(j-i)] += proj[i][j]/3;
      }
     cbt = cb_list[tord];
     if ( cbt )
      { for ( i = 0 ; i < dimf[tord] ; i++ )
         for ( j = i ; j < dimf[tord] ; j++ )
             A[IA[tail+i]-A_OFF+(j-i)] += 
	       quadratic_form(cbh->vec[i],proj,cbh->vec[j],web.sdim)/3; 
      }
     else
      { for ( i = 0 ; i < dimf[tord] ; i++ )
         for ( j = i ; j < dimf[tord] ; j++ )
             A[IA[tail+i]-A_OFF+(j-i)] += proj[i][j]/3;
      }

     if ( tail > head )
       { base = head; addend = tail; orda = tord; ordb = hord; }
     else
       { base = tail; addend = head; orda = hord; ordb = tord; }

     if ( (cbb = cb_list[ordb]) == NULL )
        q = proj; 
     else 
       { mat_mult(cbb->vec,proj,qq,dimf[ordb],web.sdim,web.sdim);
	 q = qq;
       }
     if ( (cba = cb_list[orda]) != NULL )
       { mat_mul_tr(q,cba->vec,qqq,dimf[ordb],web.sdim,dimf[orda]);
	 q = qqq;
       }
       
     /* seek column, add if not already there */
     for ( i = 0 ; i < dimf[ordb] ; i++ )
      {
        for ( j = IA[base+i]-A_OFF ; j < IA[base+i+1]-A_OFF ; j++ )
         if ( (JA[j] == addend+A_OFF) || (JA[j] == EMPTY) ) break;
        if ( j == IA[base+1]-A_OFF )
         error("approx_curvature: cannot find edge in JA list.\n",RECOVERABLE);

	 for ( k = 0 ; k < dimf[orda] ; k++ )
            { JA[j+k] = addend + A_OFF + k; /* in case first time */
	      A[j+k] += q[i][k]/6;
            }
      }
   }

  if ( web.dimension == SOAPFILM )
  FOR_ALL_FACETS(f_id)
   {
     double area = get_facet_area(f_id);
     double s1s1,s2s2,s1s2;  /* dot products */
     facetedge_id fe = get_facet_fe(f_id);
     edge_id e_id1 = get_fe_edge(fe);
     edge_id e_id2 = get_fe_edge(get_next_edge(fe));
     REAL side1[MAXCOORD],side2[MAXCOORD];
     int m;

     /* calc normal projection matrix */
     get_edge_side(e_id1,side1);
     get_edge_side(e_id2,side2);
     s1s1 = dot(side1,side1,web.sdim);
     s2s2 = dot(side2,side2,web.sdim);
     s1s2 = dot(side1,side2,web.sdim);
     for ( i = 0 ; i < web.sdim ; i++ )
      { 
	if ( effective_area_flag )
	 for ( j = 0 ; j < web.sdim ; j++ )
	   proj[i][j] = -(s2s2*side1[i]*side1[j] - s1s2*side1[i]*side2[j] 
	     - s1s2*side2[i]*side1[j] + s1s1*side2[i]*side2[j])/4/area;
        else
	   for ( j = 0 ; j < web.sdim ; j++ ) proj[i][j] = 0.0;
	proj[i][i] += area;
      }
     for ( m = 0 ; m < 3 ; m++, fe = get_next_edge(fe) )
      { int tord = ordinal(get_fe_tailv(fe));
        int hord = ordinal(get_fe_headv(fe));
        int tail = IA_INV[tord];
        int head = IA_INV[hord];
        int dim_h = dimf[hord];
        int dim_t = dimf[tord];
        int base,addend;
        double length;
        REAL side[MAXCOORD];
        int orda,ordb;
     
        /* add to vertex stars */
        cb = cb_list[tord];
        if ( cb )
         { for ( i = 0 ; i < dimf[tord] ; i++ )
            for ( j = i ; j < dimf[tord] ; j++ )
             A[IA[tail+i]-A_OFF+(j-i)] += 
	       quadratic_form(cb->vec[i],proj,cb->vec[j],web.sdim)/6; 
         }
        else
         { for ( i = 0 ; i < dimf[tord] ; i++ )
            for ( j = i ; j < dimf[tord] ; j++ )
             A[IA[tail+i]-A_OFF+(j-i)] += proj[i][j]/6;
         }

        if ( tail > head )
          { base = head; addend = tail; orda = tord; ordb = hord; }
        else
          { base = tail; addend = head; orda = hord; ordb = tord; }

        if ( (cbb = cb_list[ordb]) == NULL )
           q = proj; 
        else 
          { mat_mult(cbb->vec,proj,qq,dimf[ordb],web.sdim,web.sdim);
	    q = qq;
          }
        if ( (cba = cb_list[orda]) != NULL )
          { mat_mul_tr(q,cba->vec,qqq,dimf[ordb],web.sdim,dimf[orda]);
	    q = qqq;
          }

        /* seek column, add if not already there */
        for ( i = 0 ; i < dimf[ordb] ; i++ )
         {
           for ( j = IA[base+i]-A_OFF ; j < IA[base+i+1]-A_OFF ; j++ )
            if ( (JA[j] == addend+A_OFF) || (JA[j] == EMPTY) ) break;
           if ( j == IA[base+1]-A_OFF )
            error("approx_curvature: cannot find edge in JA list.\n",RECOVERABLE);

	    for ( k = 0 ; k < dimf[orda] ; k++ )
               { JA[j+k] = addend + A_OFF + k; /* in case first time */
	         A[j+k] += q[i][k]/12;
               }
         }
      }
   }

#ifdef GDEBUG
/* some debug printing */
{ REAL x;
printf("IA: ");
for ( i = 0 ; i <= N ; i++ ) printf(" %d",IA[i]);
printf("\nJA: ");
for ( i = 0 ; i < count ; i++ ) printf(" %d",JA[i]);
printf("\n");
for ( i = 0 ; i < N ; i++ ) 
  { int j,k,m;
    for ( m = 0 ; m < i ; m++ ) printf("         ");
    for ( m = i,  j = 0, k = IA[i]-A_OFF ; m < N /* j < IA[i+1]-IA[i] */; m++ )
      if ( (m == JA[k]-A_OFF) && (k < IA[i+1]-A_OFF) )
        { printf(" %8.6g",A[k]); k++; j++; }
      else printf(" %8.6g",0.0);
    printf("\n");
  }
}
#endif

#ifdef FORT
  NSP = 8*N + 16*total_entries;
  ISP = (integer *)temp_calloc(NSP,sizeof(integer));
  /*  Call ODRV to perform the reordering on A */
      PATH = 2;        
      odrv_( &N, IA,JA,A, P,IP, &NSP,ISP, &PATH, &FLAG );
      odrv_flag_check(FLAG,N);

  /*  Call SDRVMD to compute the factorization */
     {  RSP = (double *) ISP;
        PATH = 5; /* SSF and SNF only */
        sdrvmd_(&N,P,IP,IA,JA,A,NULL,NULL,&NSP,ISP,RSP,&ESP, &PATH,&FLAG,&EMAX);
        sdrv_flag_check(ESP,FLAG,N);
     }
#else
  /* my own sparse matrix routines */
  sp_order(N,IA,JA,P,IP);
  IW = (int*)temp_calloc(N+1,sizeof(int));
  fillsize = sp_fill_setup(N,IA,JA,P,IP,IW);
printf("fillsize %d\n",fillsize);
  F = (double*)temp_calloc(fillsize,sizeof(double));
  sp_factor(N,IA,JA,P,IP,IW,A,F);
#endif

  free_matrix(proj);
  free_matrix(qq);
  free_matrix(qqq);
}

/*********************************************************************
*
* function: conbasis_mult()
*
* purpose:  multiply vector (form) by constraint bases of vertices
*
*/

void conbasis_mult(X,Y)
REAL *X;  /* incoming, web.sdim coords per vertex */
REAL *Y;  /* outgoing, degrees of freedom per vertex */
{
  vertex_id v_id;
  int i=0;
  int j;
  int k = 0; /* place in X */
  struct basis *cb;

  FOR_ALL_VERTICES(v_id)
    { int ord = ordinal(v_id);
      int dim = dimf[ord];
      k = ord*web.sdim;
      i = IA_INV[ord];
      if ( (cb = cb_list[ord]) != NULL )
	{ for ( j = 0 ; j < dim ; j++,i++ )
	    Y[i] = dot(X+k,cb->vec[j],web.sdim);
        }
      else /* just copy */
       for ( j = 0 ; j < dim ; j++,i++ )
	 Y[i] = X[k+j];
    }
}


/*********************************************************************
*
* function: conbasis_tmult()
*
* purpose:  form linear combos of constraint bases of vertices
*
*/

void conbasis_tmult(X,Y)
REAL *X;  /* incoming, degrees of freedom per vertex */
REAL *Y;  /* outgoing, web.sdim coords per vertex */
{
  vertex_id v_id;
  int i;
  int j,m;
  int k = 0; /* place in Y */
  struct basis *cb;

  FOR_ALL_VERTICES(v_id)
    { int ord = ordinal(v_id);
      int dim = dimf[ord];
      k = ord*web.sdim;
      i = IA_INV[ord];
      if ( (cb = cb_list[ord]) != NULL )
	{ for ( m = 0 ; m < web.sdim ; m++ )
	    { Y[k+m] = 0.0; 
	      for ( j = 0 ; j < dim ; j++ )
		Y[k+m] += X[i+j]*cb->vec[j][m];
            }
          k += web.sdim;
        }
      else /* just copy */
       for ( j = 0 ; j < dim ; j++,i++ )
	 Y[k+j] = X[i];
    }
}

/*********************************************************************
*
*  function: mobility_mult()
*
*  purpose:  multiply a formfield by the mobility matrix.
*            actually does just one component at a time.
*            conversion done in place.
*
*/

void mobility_mult(B)
doublereal *B;
{
  integer PATH, FLAG=0;
  integer ESP;
  double EMAX;
  double *temp = (double *)temp_calloc(N,sizeof(double));

  conbasis_mult(B,temp);

#ifdef FORT
  RSP = (double *) ISP;
  PATH = 3; /* SNS */
  sdrvmd_(&N,P,IP,IA,JA,A,temp,temp,&NSP,ISP,RSP,&ESP, &PATH,&FLAG,&EMAX);
  sdrv_flag_check(ESP,FLAG,N);
#else
  sp_backsub(N,P,IP,IW,F,temp,temp);
#endif
 
  conbasis_tmult(temp,B);
}

/***********************************************************************
*
* function: mobility_cleanup()
*
*/

void mobility_cleanup()
{
  /* free stuff */
  temp_free((char *)IA); IA = NULL;
  temp_free((char *)IA_INV); IA_INV = NULL;
  temp_free((char *)A);  A = NULL;
  temp_free((char *)JA); JA = NULL;
  temp_free((char *)IP); IP = NULL;
  temp_free((char *)P);  P = NULL;
#ifdef FORT
  temp_free((char *)ISP);   ISP = NULL;
#else
  temp_free((char *)F);  F = NULL;
  temp_free((char *)IW);  IW = NULL;
#endif
  temp_free((char *)cb_list);  cb_list = NULL;
  if ( conbasis ) { temp_free((char *)conbasis); conbasis = NULL; }
  temp_free((char*)dimf); dimf = NULL;
}


/**********************************************************************
*
*  function: approx_curv_calc()
*
*  purpose: converts energy gradients to approximate curvature
*
*/

void approx_curv_calc()
{
  int i,j;
  doublereal *B;
  vertex_id v_id;

  B  = (double *)temp_calloc(web.sdim*web.skel[VERTEX].max_ord,
		  sizeof(double));  

  /* each coordinate gives a right side */
      FOR_ALL_VERTICES(v_id)
        { 
          int vnum = web.sdim*ordinal(v_id);
          for ( j = 0 ; j < web.sdim ; j++ )
            B[vnum+j] = get_force(v_id)[j];
        }
      mobility_mult(B);
      FOR_ALL_VERTICES(v_id)
        { 
          int vnum = web.sdim*ordinal(v_id);
          for ( j = 0 ; j < web.sdim ; j++ )
            get_force(v_id)[j] = B[vnum+j];
        }

  temp_free((char *)B);
}

/********************************************************************
*
* function: stability_test()
*
* purpose: find largest eigenvalue of mobility matrix, by starting
*          with random vector and repeatedly applying mobility
*          matrix.
*/

void stability_test()
{
  doublereal *B;
  int i;
  double oldmag,newmag;
  double drand48();

  mobility_setup();
  B  = (double *)temp_calloc(N,sizeof(double));  
  for ( i = 0 ; i < N ; i++ ) B[i] = drand48(); /* random vector */
  oldmag = dot(B,B,N);
  for ( i = 0 ; i < 20 ; i++ )
   {
      mobility_mult(B);
      newmag = dot(B,B,N);
      printf("%3d. ratio %f \n",i,sqrt(newmag/oldmag));
      oldmag = newmag;
   }
  mobility_cleanup();
  temp_free((char *)B);
}

