(* This program is by Adrian Mariano: adrian@u.washington.edu
 *
 * Copyright (c) 1991, The Geometry Center
 *                     University of Minnesota 
 *                     1300 South Second Street
 *                     Minneapolis, MN  55454
 *
 * email address: software@geom.umn.edu
 *
 * This software is copyrighted as noted above.  It may be freely copied,
 * modified, and redistributed, provided that the copyright notice is
 * preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is".  Bug reports or fixes may be sent
 * to the authors, who may or may not act on them as they desire.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 *     The National Science and Technology Research Center for
 *      Computation and Visualization of Geometric Structures
 *)


Needs["Graphics`ParametricPlot3D`"]
Needs["Algebra`Trigonometry`"]

BeginPackage["ParallelCurves`","Graphics`ParametricPlot3D`",
"Algebra`Trigonometry`"]

ParallelCurves::usage = 
"ParallelCurves[{fx, fy}, <{dfx,dfy}>, {u, umin, umax, ustep}, 
{{xmin, xmax}, {ymin, ymax}}, {tmin, tmax, tstep}, <GraphOpts> ]
Draw parallel curves to the curve specified parametrically
as a function of u over the range umin to umax.  The 
graphs will all be plotted on the region from xmin to xmax
and ymin to ymax.  The variable t determines how far
the parallel curves should be from the original curve.
Each curve appears on a different graph.  The optional
parameter {dfx,dfy} is the derivative of the curve.  If
specified, it is used to determine the direction of
the normals.  GraphOpts is passed to ListPlot.";

ParallelCurves3D::usage = 
"ParallelCurves3D[{fx, fy}, {u, umin, umax, <ustep>},
{{xmin, xmax}, {ymin, ymax}}, {zmin, zmax, <zstep>},<GraphOpts>]
Draw parallel curves to the curve specified 
parametrically as a function of u over the range umin 
to umax.  The parallel curves are stacked into a 
three-dimensional surface.  GraphOpts is passed to
ParametricPlot3D.";

ParallelCurvesOverlay::usage = 
"ParallelCurvesOverlay[{fx, fy}, {u, umin, umax, ustep}, 
{{xmin, xmax}, {ymin, ymax}}, {tmin, tmax, tstep}]
Draw parallel curves to the curve specified parametrically
as a function of u over the range umin to umax.  The 
graphs will all be plotted on the region from xmin to xmax
and ymin to ymax.  The variable t determines how far
the parallel curves should be from the original curve.
All of the curves are superimposed on the same graph";

Evolute::usage =
"Evolute[{fx, fy}, t, <options>]
Return parametric representation of the evolute of
the curve specified parametrically as a function of t.
The only option is to turn off simplification with
Simplify->False.";

GraphEvolute::usage = 
"GraphEvolute[{fx, fy}, {t, tmin, tmax}, <plotoptions>]
Graph the evolute of the curve specified parametrically
as a function of t from t=tmin to t=tmax, passing
plotoptions to ParametricPlot.";

Ellipsoid::usage = 
"Ellipsoid[a, b, c, phi, theta]  Returns the point on 
an ellipsoid with semiaxes a, b, and c, and angles 
phi and theta.";       


NormalLines::usage=
"NormalLines[{fx,fy}, {t, tmin, tmax, tstep}, plotrange,
options]  Draws the lines normal to {fx,fy} at points
ranging from tmin to tmax in steps of tstep.  Plotrange
is the range of the plot, and options are passed on to
Show.";

ParallelSurfaces::usage=
"ParallelSurfaces[{fx,fy,fz}, {u, umin, umax, [ustep]}, 
{v, vmin, vmax, [vstep]}, plotrange, {tmin, tmax, tcount},
options]  Plots a series of surfaces parallel to the
surface specified as a function of u and v.  Will produce
tcount graphs, ranging from tmin to tmax in the direction
normal to the surface.  Any options are passed to
ParametricPlot3D.";

Begin["`private`"]

ParallelCurves[{ffx_, ffy_}, {parm_, umin_, umax_, ustep_}, 
               graphrange_, {tmin_, tmax_, tstep_},
               graphoptions___] :=
   Block[ { f, grad={}, point={}, newpoint, newgrad,t, q, param,
            fx,fy,dfx,dfy,f,n,norm,u}, 
      fx[u_] := Release[ ffx /. parm -> N[u]];
      fy[u_] := Release[ ffy /. parm -> N[u]];
      dfx[u_] := Release[D[ffx, parm] /. parm -> N[u]];
      dfy[u_] := Release[D[ffy, parm] /. parm -> N[u]];
      f[u_] := { fx[u], fy[u] };
      n[u_] := { -dfy[u], dfx[u] };
      norm[u_] := n[u] / Sqrt[ n[u] . n[u] ];
      Do[ AppendTo[point, N[f[u]]];
          AppendTo[grad, N[norm[u]]],
      { u, umin, umax, ustep }];
      Do[ ListPlot[ point - t grad, PlotRange->graphrange,
      graphoptions, PlotJoined->True, Axes->None, 
      AspectRatio->Automatic]
          , {t, tmin, tmax, tstep}]
]     


ParallelCurves[ff_, dff_,
               {parm_, umin_, umax_, ustep_}, 
               graphrange_, {tmin_, tmax_, tstep_},
               graphoptions___] :=
   Block[ {   newpoint, newgrad,t, q, param }, grad={}; point={};
      f[u_] := Release[ ff /. parm -> N[u]];
      df[u_] := Release[dff /. parm -> N[u]];
      n[u_] := ( ii= df[u]; {-ii[[2]],ii[[1]]} );
      norm[u_] := (n[u]) / Sqrt[( n[u]) .(n[u]) ];
      Do[ AppendTo[point, N[f[u]]];
          AppendTo[grad, N[norm[u]]],
      { u, umin, umax, ustep }];
      Do[ ListPlot[ point - t grad, PlotRange->graphrange,
      graphoptions, PlotJoined->True, Axes->None, 
      AspectRatio->Automatic]
          , {t, tmin, tmax, tstep}]
]     


ParallelCurvesOverlay[{ffx_, ffy_}, {parm_, umin_, umax_, ustep_}, 
                 graphrange_, {tmin_, tmax_, tstep_},
                 graphoptions___] :=
   Block[ {  grad={}, point={}, newpoint, newgrad,t, q, param }, 
      fx[u_] := Release[ ffx /. parm -> N[u]];
      fy[u_] := Release[ ffy /. parm -> N[u]];
      dfx[u_] := Release[D[ffx, parm] /. parm -> N[u]];
      dfy[u_] := Release[D[ffy, parm] /. parm -> N[u]];
      f[u_] := { fx[u], fy[u] };
      n[u_] := { -dfy[u], dfx[u] };
      norm[u_] := n[u] / Sqrt[ n[u] . n[u] ];
      Do[  AppendTo[point, f[u]];
           AppendTo[grad, norm[u]]
        , { u, umin, umax, ustep }];
        i=1;
      Do[ graphs[i]=ListPlot[ point - t grad, 
      PlotRange->graphrange, graphoptions, 
      PlotJoined->True,  Axes->None, AspectRatio->Automatic,
      DisplayFunction->Identity]; i++
          , {t, tmin, tmax, tstep}];
      Apply[Show, Release[Append[graphs /@ Table[t, {t,1, i-1}],
         DisplayFunction->$DisplayFunction]]]
     ]

 
ParallelCurvesOverlay[ff_, dff_, 
                 {parm_, umin_, umax_, ustep_}, 
                 graphrange_, {tmin_, tmax_, tstep_},
                 graphoptions___] :=
    Block[ {f,df,n,norm, point={}, grad={}, graphs },
      f[u_] := Release[ ff /. parm -> N[u]];
      df[u_] := Release[dff /. parm -> N[u]];
      n[u_] := ( ii= df[u]; {-ii[[2]],ii[[1]]} );
      norm[u_] := (n[u]) / Sqrt[( n[u]) .(n[u]) ];
      Do[  AppendTo[point, f[u]];
           AppendTo[grad, norm[u]]
        , { u, umin, umax, ustep }];
        i=1;
      Do[ graphs[i]=ListPlot[ point - t grad, 
      PlotRange->graphrange, graphoptions, 
      PlotJoined->True,  Axes->None, AspectRatio->Automatic,
      DisplayFunction->Identity]; i++
          , {t, tmin, tmax, tstep}];
      Apply[Show, Release[Append[graphs /@ Table[t, {t,1, i-1}],
         DisplayFunction->$DisplayFunction]]]
     ]

ParallelCurves3D[{ffx_, ffy_}, {parm_, tmin_, tmax_, tstep___}, 
                 graphrange_, {zmin_, zmax_, zstep___},
                 graphoptions___ ] :=
   Block[ {  u,z,fx,fy,dfx,dfy,f,n,norm,Fx,Fy }, 
      fx[u_] :=  Release[ffx /. parm -> N[u]];
      fy[u_] :=  Release[ffy /. parm -> N[u]];
      dfx[u_] := Release[D[ffx, parm] /. parm -> N[u]];
      dfy[u_] := Release[D[ffy, parm] /. parm -> N[u]];
      f[u_] := { fx[u], fy[u] };
      n[u_] := { -dfy[u], dfx[u] };
      norm[u_] := Sqrt[ n[u] . n[u] ];
      Fx[u_, z_] := fx[u] + z * dfy[u] / norm[u];
      Fy[u_, z_] := fy[u] - z * dfx[u] / norm[u];
      ParametricPlot3D[
        { Fx[u, z], Fy[u, z], z }, {u, tmin, tmax, tstep}, 
          {z, zmin, zmax, zstep},graphoptions] 
          
]   


Options[Evolute] = {Simplify->True};             

Evolute[{x_, y_}, t_, opts___]:=
  Block[ {dx,dy,ddx,ddy,dlen,denom,simplify},
    simplify = Simplify /. {opts} /. Options[Evolute];
    If[simplify, 
        simp[expr_]:=TrigReduce[expr],
        simp[expr_]:=expr];
    dx=simp[D[x,t]];
    dy=simp[D[y,t]];
    ddx=simp[D[dx,t]];
    ddy=simp[D[dy,t]];
    dlen=simp[{dx,dy}.{dx,dy}]; 
    denom=simp[dx ddy - ddx dy];
    simp[  {x - dy dlen / denom, y + dx dlen / denom}]
   ] 
  
  
GraphEvolute[{fx_, fy_}, {var_, tmin_, tmax_}, graphoptions___] :=
 Block[ {f}, 
    f=Evolute[{fx, fy}, var]; 
    ParametricPlot[f, {var, tmin, tmax}, graphoptions]
 ]


Ellipsoid[a_, b_, c_, phi_, theta_] := {
    a Sin[phi] Cos[theta],
    b Sin[phi] Sin[theta],
    c Cos[phi]
 }


NormalLines[{ffx_, ffy_}, {var_, tmin_, tmax_, tstep_},
  plotrange_, graphopts___] :=
  (
    fx[u_] := Release[ ffx /. var -> u];
    fy[u_] := Release[ ffy /. var -> u];
    dfx[u_] := Release[ D[ fx[u], u]];
    dfy[u_] := Release[ D[ fy[u], u]];
    norm[u_] := { -dfy[u], dfx[u]};
    lines={};
    Do [ AppendTo[lines, 
           Line[{  
              {fx[t],fy[t]} // N ,
              {fx[t],fy[t]}+15 norm[t] // N
             }]]
           
                 , {t, tmin, tmax, tstep}];
   Show[Graphics[lines], PlotRange->plotrange,graphopts,
    AspectRatio->Automatic]
    ) 
 
 ParallelSurfaces[func:{_,_,_},  {uvar_, umin_, umax_,ustep___}, 
   {vvar_, vmin_, vmax_,vstep___},plotrange_, {tmin_, tmax_, tcount_},
   opts___]:=
   Block[{ a=0 ,v,u}   ,
      f[u_, v_] := N[Release[func /. {uvar-> N[u], 
                              vvar -> N[v]}]];
      dfu[u_, v_] := N[Release[D[func, uvar] /.
                              {uvar -> N[u], vvar -> N[v]}]];
      dfv[u_, v_] := N[Release[D[func, vvar] /.
                              {uvar -> N[u], vvar -> N[v]}]];
      cross[{x1_, x2_, x3_}, {y1_, y2_, y3_}] :=
        { Det[{{x2, x3}, {y2, y3}}], 
         -Det[{{x1, x3}, {y1, y3}}],
          Det[{{x1, x2}, {y1, y2}}]
        };
      n[u_, v_] := cross[dfu[u, v], dfv[u, v]];
      norm[u_, v_] := n[u, v]/Sqrt[ n[u, v] . n[u, v]];
      F[u_, v_, t_] := f[u, v] + t norm[u,v];
      Do[                
         ParametricPlot3D[F[u,v,t],{u,umin, umax, ustep}, 
           {v, vmin, vmax,vstep}, PlotRange->plotrange,opts,
           AspectRatio->Automatic,Boxed->False]
        ,{t, tmin, tmax, (tmax-tmin)/(tcount-1)}
      ]
]

 
End[]
EndPackage[] 

