/* automata.d/src file wa.c */
/* Constructing the word acceptor. Suppose we have an automatic group,
with a given set of generators,
and that we have already found all the word differences able to arise
from two shortest routes to points distance one away from each other
in the Cayley graph. We can then form a finite state automaton in
which a state consists of three disjoint sets of word differences
called |behind|, |better| and |worse|. This is a full and complete set
of word differences. To complete the description of the finite state
automaton, we need to describe the effect of a generator g. But
before doing this, we give the conceptual background of these disjoint
sets.

Suppose we have a reduced word w and let u be a word, such that
wu is reduced.
Let w'u' be a reduced word, ending at a distance one from wu in
the Cayley graph, and let w' and w have the same length.
Let v be the reduced word representing \overline{w^{-1}w'}.
If s is the state reached by w, then the set |better| associated
with s is the set of
such v's with the property that w' is lexicographically before
w.
The set |worse| associated with s is the set of such v's with the
property that w' is lexicographically after w.
The set |behind| has a similar description, but now w' has length
one less than w, and we only include v's of length one.

For the start state of the finite state automaton the three sets are
empty.
It is not too hard to see how to use the three sets to see which
generators give failure, and, in case none of them give failure to
update the three sets on application of a generator g.

During the first phase of the program, we collect up information about
which word differences arise from which other word differences, and we
only use this information when constructing the automaton.
This short description needs a lot of further explanation.
*/
#include <stdio.h>
#include "defs.h"
#include "list.h"
#include "word.h"
#include "lg.h"
# define DIFF1OP ".diff1"
# define WAOP ".wa"
FILE * rfile=stdin;
FILE * wfile=stdout;



static int ACCEPTORstruct_sgn PARMS ((vindex , vindex));
static ACCEPTORstruct * ACCEPTORstruct_create PARMS ((VOID));
static void ACCEPTORstruct_kill PARMS ((vindex));
static void ACCEPTORstruct_print PARMS ((FILE*,vindex));
static void collect_WA_states PARMS ((VOID));
static void construct_transitions PARMS ((vindex curr));
static void WA_fail_set PARMS ((vindex,list*));
static void mk_WA_target PARMS ((vindex source, gen g,vindex target));
vindex eos_state = UNDEFINED;
vindex failstate=UNDEFINED;
vtxhashlist states;
		/* a state is a triple of ordered lists of word differences*/
list states_in_waiting;
list gendiffs; /* the list of word differences that are actually generators */
list * forbidden;
lg * WA=0;
lg * DIFF=0;
extern word * user_gen_name;
extern gen * inv_of;
extern int num_gens;

main(argc,argv)
	int argc;
	char * argv[];
{
	int M=1;
	int N;
	gen g;
	vindex v;
	char gpname[100];
	char filename[100];
	char * label;


	if (argc>2){
		fprintf(stderr,"Usage: wa [gpname]\n");
		exit(2);
	}
	else if (argc==2){
/* the input and output files are being specified as gpname.diff1 and 
gpname.wa respectively */
		strcpy(gpname,argv[1]);
		strcpy(filename,gpname);
		strcat(filename,DIFF1OP);
		if ((rfile=fopen(filename,"r"))==0)
  			{ fprintf(stderr,"Cannot open %s.\n",filename); exit(2);}
		strcpy(filename,gpname);
		strcat(filename,WAOP);
		wfile=fopen(filename,"w");
	}

	setbuf(stdout, (char*)0);
	setbuf(stderr,(char*)0);
	setbuf(wfile,(char*)0);
	label=vzalloc2(char,9);
	while (read_next_string(label,8,rfile)){
		if (strcmp(label,"Format  ")==0)
			format_echocheck("2.2",rfile,wfile);
		else if (strcmp(label,"fsa     ")==0)
			DIFF=lg_read(GENERIC,BASIC_FSA,rfile);
		else if (strcmp(label,"inverses")==0)
			read_inverse_array(rfile);
		else 
			default_inverse_array();
	}
	Free_dp((dp)label); label=0;
	lg_set_indices(DIFF);
	forbidden=vzalloc1(list);
	list_init(forbidden,GEN,ORDERED);
	list_init(&gendiffs,ASINT_VINDEX,ORDERED);
	for (g=1;g<=num_gens;g++)
		if ((v=get(basept(DIFF),g*(num_gens+1)))==basept(DIFF) ||
			list_insert(&gendiffs,(dp)&v)==FALSE)
/* i.e. if the generator is trivial or equal to an earlier numbered one */
			list_insert(forbidden,(dp)&g);
	if (list_empty(forbidden)){
		list_clear(forbidden);
		Free_dp((dp)forbidden); forbidden=0;
	}
	N=get_num_verts(DIFF)*get_num_verts(DIFF);
	while (M<N)
		M *=2;

	WA=vzalloc1(lg);
	lg_init(WA,ACCEPTOR,BASIC_FSA,num_gens+1);
	vtxhashlist_init(&states,M);
	collect_WA_states();
	vtxhashlist_clear(&states);
	fsa_opt(2,WA);
	lg_print(wfile,WA);
	lg_clear(WA);
	Free_dp((dp)WA); WA=0;
	fprintf(wfile,"\t# Symbols used for inverses of generators\n");
	fprintf(wfile,"inverses \{\n");
	for (g=1;g<=num_gens;g++){
		fprintf(wfile,"inv(");
		gen_print(wfile,g);
		fprintf(wfile,")=");
		gen_print(wfile,inv(g));
		fprintf(wfile," ");
	}
	fprintf(wfile,"\}\n");
	if (forbidden){
		list_clear(forbidden);
		Free_dp((dp)forbidden); forbidden=0;
	}
	list_clear(&gendiffs);
	lg_clear(DIFF);
	Free_dp((dp)DIFF); DIFF=0;
	for (g=0;g<gen_array_size;g++)
		word_clear(user_gen_name+g);
	Free_dp((dp)user_gen_name); user_gen_name=0;
	Free_dp((dp)inv_of); inv_of=0;
	assert(store_ptrs==0);
	exit(0);
}

mp_fntab ACCEPTOR_fntab =
{
	ACCEPTORstruct_sgn,
	(V2DP)ACCEPTORstruct_create,
	ACCEPTORstruct_kill,
	ACCEPTORstruct_print,
	0,
	0
};


static int
ACCEPTORstruct_sgn(v1,v2) 
	vindex v1;
	vindex v2;
{
	int ans = 0;
	if (
		((ans = list_sgn(behindp(v1),behindp(v2)))  == 0) &&
		((ans = list_sgn(betterp(v1),betterp(v2)))  == 0) &&
		((ans = list_sgn(worsep(v1),worsep(v2)))  == 0) 
	)
		;
	/*the comparisons stop at the first non-zero answer of the three*/
	return ans;
}
static void
ACCEPTORstruct_kill(v)
	vindex v;
{
	ACCEPTORstruct *As = (ACCEPTORstruct *)(v->mp);
	assert(v->mp);
	assert((v->fntab) == ACCEPTOR);
	list_clear(&(As->behind));
	list_clear(&(As->better));
	list_clear(&(As->worse));
	Free_dp(v->mp);
	v->mp = 0;
}

static ACCEPTORstruct *
ACCEPTORstruct_create()
{
	ACCEPTORstruct * As = vzalloc1(ACCEPTORstruct);
	list_init(&(As->behind),ASINT_VINDEX,ORDERED);
	list_init(&(As->better),ASINT_VINDEX,ORDERED);
	list_init(&(As->worse),ASINT_VINDEX,ORDERED);
	return As;
}

static void
ACCEPTORstruct_print(wfile,v)
	FILE * wfile;
	vindex v;
{
	ACCEPTORstruct * As = (ACCEPTORstruct *)(v->mp);
	assert((v->fntab) ==ACCEPTOR);
	assert(As);
	fprintf(wfile,"ACCEPTOR \n");
	fprintf(wfile,"better ");
	list_print(wfile,&(As->better));
	fprintf(wfile,"worse ");
	list_print(wfile,&(As->worse));
	fprintf(wfile,"behind ");
	list_print(wfile,&(As->behind));
}

list *
behindp(v)
	vindex v;
{
	ACCEPTORstruct * As = (ACCEPTORstruct *)(v->mp);
	assert((v->fntab) ==ACCEPTOR);
	assert(As);
	return (&(As->behind));
}
list *
betterp(v)
	vindex v;
{
	ACCEPTORstruct * As = (ACCEPTORstruct *)(v->mp);
	assert((v->fntab) ==ACCEPTOR);
	assert(As);
	return (&(As->better));
}
list *
worsep(v)
	vindex v;
{
	ACCEPTORstruct * As = (ACCEPTORstruct *)(v->mp);
	assert((v->fntab) ==ACCEPTOR);
	assert(As);
	return (&(As->worse));
}
/* |lp| is a pointer to an existing list.
*/
static void
set_behind(v,lp)
	vindex v;
	list *lp;
{
	ACCEPTORstruct * As = (ACCEPTORstruct *)(v->mp);
	assert((v->fntab) ==ACCEPTOR);
	assert(As);
	list_cpy(lp,&(As->behind));
}

static void
set_better(v,lp)
	vindex v;
	list *lp;
{
	ACCEPTORstruct * As = (ACCEPTORstruct *)(v->mp);
	assert((v->fntab) ==ACCEPTOR);
	assert(As);
	list_cpy(lp,&(As->better));
}

static void
set_worse(v,lp)
	vindex v;
	list *lp;
{
	ACCEPTORstruct * As = (ACCEPTORstruct *)(v->mp);
	assert((v->fntab) ==ACCEPTOR);
	assert(As);
	list_cpy(lp,&(As->worse));
}

/* We work out the effect of a generator g on an acceptor state. The
routine looks for an existing state with the same three lists. If
there is no such state, it creates a new one.
True or false is returned according to whether the target state waa
already there or not. The target state is filled in at the vertex
pointed to by |target|.
*/
static void
mk_WA_target(source,g,target)
	vindex source;
	gen g;
	vindex target;
{
	gen h;
	list new_better, new_worse, new_behind;
	vindex v;
	vindex image;
	vindex vbetter = UNDEFINED;
	list_traverser tbetter;
	vindex vworse = UNDEFINED;
	list_traverser tworse;
	assert(1 <= g && g <= num_gens);
	assert(source->fntab ==ACCEPTOR);
	assert(get(source,g) == UNDEFINED);
	list_init(&new_better,ASINT_VINDEX,ORDERED);
	list_init(&new_worse,ASINT_VINDEX,ORDERED);
	list_init(&new_behind,ASINT_VINDEX,ORDERED);
	list_traverser_init(&tbetter,betterp(source));
	while (list_next(&tbetter,&vbetter)) {
		for (h=1;h<=num_gens+1;h++){
		if ((image=get(vbetter,(num_gens+1)*(g-1)+h))!=UNDEFINED){
				if (h==num_gens+1
				&& list_find(&gendiffs,(dp)&image,(dp)&v)==TRUE)
		/* that means the word difference is a single generator */
					(void)list_insert(&new_behind,&image);
				else if (image!=basept(DIFF) && h!=num_gens+1)
					(void)list_insert(&new_better,&image);
			}
		}
	}
	list_traverser_clear(&tbetter);
	list_traverser_init(&tworse,worsep(source));
	while (list_next(&tworse,&vworse)) {
		for (h=1;h<=num_gens+1;h++){
		if ((image=get(vworse,(num_gens+1)*(g-1)+h))!=UNDEFINED){
			if (h==num_gens+1
				&& list_find(&gendiffs,(dp)&image,(dp)&v)==TRUE)
					(void)list_insert(&new_behind,&image);
			else if (image != basept(DIFF) && h!=num_gens+1) 
					(void)list_insert(&new_worse,&image);
			}
		}
	}
	list_traverser_clear(&tworse);
	v = get(basept(DIFF),(num_gens+1)*g);
		/* the (g, $) image */
	if (v!=UNDEFINED)
		(void)list_insert(&new_behind,&v);
	for (h=1;h<=num_gens+1;h++){
		if ((image=get(basept(DIFF),(num_gens+1)*(g-1)+h))!=UNDEFINED){
						/* the (g,h) image */
			if (is_basepoint(image))
				continue;
			/*do not include a trivial word difference*/
			else if (g > h)
				(void)list_insert(&new_better,&image);
			else if (h > g && h!=num_gens+1) 
				(void)list_insert(&new_worse,&image);
		}
	}
	set_behind(target,&new_behind);
	set_better(target,&new_better);
	set_worse(target,&new_worse);
	list_clear(&new_behind);
	list_clear(&new_worse);
	list_clear(&new_better);
}




static void
collect_WA_states()
{
	vindex curr = UNDEFINED;
	gen g;
	if (num_gens>0){
		failstate=lg_vertex_create(WA);
		set_category(failstate,NONACCEPTSTATE);
	}
	eos_state=lg_vertex_create(WA);
	set_category(eos_state,ACCEPTSTATE);
	for (g=1;g<=num_gens;g++)
		set(eos_state,g,failstate);
	set(eos_state,num_gens+1,eos_state);
	list_init(&states_in_waiting,VINDEX,FIFO);
	(void)list_insert(&states_in_waiting,&(basept(WA)));
	while(list_delget_first(&states_in_waiting,&curr)) 
		construct_transitions(curr);
	list_clear(&states_in_waiting);
}

	
static void
construct_transitions(curr)
	vindex curr;
{
	int hash;
	int M= states.maxhash + 1;
	gen g;
	assert((curr->fntab) == ACCEPTOR);
	WA_fail_set(curr,behindp(curr));
	WA_fail_set(curr,betterp(curr));
	WA_fail_set(curr,worsep(curr));
	if (forbidden!=0){
		list_traverser lt;
		list_traverser_init(&lt,forbidden);
		while (list_next(&lt,(dp)&g))
			set(curr,g,failstate);
		list_traverser_clear(&lt);
	}
	for (g=1;g<=num_gens;g++) {
		if (get(curr,g)==UNDEFINED) {
			vindex target = lg_vertex_create(WA);
			vindex out = UNDEFINED;
			set_category(target,NONACCEPTSTATE);
			mk_WA_target(curr,g,target);
			hash = statelist_hash(behindp(target),M-1);
			hash = hash + statelist_hash(betterp(target),M-1);
		hash = (hash + statelist_hash(worsep(target),M-1))&(M-1);
			if (vtxhashlist_find(&states,&target,&out,hash)) {
					/*this state already exists*/
				lg_vertex_kill(WA,target);
				target = out;
			}
			else {
				(void)vtxhashlist_insert(&states,&target,hash);
				(void)list_insert(&states_in_waiting,&target);
			}
			set(curr,g,target);
		}
	}
	set(curr,num_gens+1,eos_state);
}


/* Let |v| be a |vindex| representing some state in |WA|.
Let |lp| be a pointer to a list of |vindex|'s in |DIFF|.
We run through, looking for word differences w of length one.
If \overline{g^{-1}w}  is trivial, then we fill in the image of |v|
under |g| in |WA| as being the fail state.
*/
static void
WA_fail_set(source,lp)
	list *lp;
	vindex source;
{
	list_traverser tlp;
	vindex temp;
	gen g,h;
	assert(source->fntab == ACCEPTOR);
assert(lp==behindp(source) || lp==betterp(source) || lp==worsep(source));
	list_traverser_init(&tlp,lp);
	while (list_next(&tlp,&temp)){
		vindex v;
	if (list_find(&gendiffs,(dp)&temp,(dp)&v)==FALSE&&lp!=betterp(source)) 
			continue;
	/* unless this is the "better" list we're only interested in
	looking at word differences which are single generators. */ 
		assert(temp != basept(DIFF));
		for (g=1;g<=num_gens+1;g++){
			for (h=1;h<=num_gens+1;h++){
				vindex image=UNDEFINED;
				if (get(source,g)!=UNDEFINED)
					break;
		else if ((image=get(temp,(num_gens+1)*(g-1)+h))!=UNDEFINED){
					if (image==basept(DIFF))
					if (h==num_gens+1||lp==betterp(source))
							set(source,g,failstate);
				}
			}
		}
	}
	list_traverser_clear(&tlp);
}


