/****************************************************************************
 * vector.c
 * Author Joel Welling
 * Copyright 1992, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* 
This module implements vectors as a user data type.
*/

#include <stdio.h>
#include "ge_error.h"
#include "alisp.h"
#include "p3d.h"
#include "ren.h"

/* Notes-
*/

#define DEFAULT_X 0.0
#define DEFAULT_Y 0.0
#define DEFAULT_Z 0.0
#define INITIAL_VECTORS 5000;

/* The USERDATA struct referenced by all vectors */
static USERDATA vectordef;

/* The list of free vector data structures, and total number allocated. */
static Vector_body *free_list= (Vector_body *)0;
static int total_vectors= 0;

static void vector_print( data )
char *data;
/* This is the print method for the vector */
{
  Vector_body *vec;
  vec= (Vector_body *)data;
  ger_debug("vector_print");
  fprintf(stderr,"*vector* ");
#ifdef never
  fprintf(stderr,"vector: XYZ= (%f, %f, %f)\n",
	  vec->d.x, vec->d.y, vec->d.z);
#endif
}

void vector_destroy( data )
char *data;
/* This routine destroys a vector instance */
{
  Vector_body *thisvectordata;

  ger_debug("vector_destroy");

  thisvectordata= (Vector_body *)data;
  thisvectordata->next= free_list;
  free_list= thisvectordata;
}

static void vector_setf( place, location, value )
NODE *place, *location, *value;
/* This routine provides setf methods for vector 'structs'. */
{
  float val;
  char *fname, *cname;

  ger_debug("vector_setf");

  if ( realp(value) ) val= get_number(value);
  else {
    ger_error("vector_setf: argument value not int or real!");
    return;
  }

  fname= getname(place);
  cname= callout_name(place);

  if (!strcmp(fname,"vector-x") || !strcmp(cname,"~vector-x"))
    set_vector_x((Vector)location,val);
  else if (!strcmp(fname,"vector-y") || !strcmp(cname,"~vector-y"))
    set_vector_y((Vector)location,val);
  else if (!strcmp(fname,"vector-z") || !strcmp(cname,"~vector-z"))
    set_vector_z((Vector)location,val);
  else ger_error("vector_setf: %s is not a setf-able field of a vector.",
		 fname);
}

static void init_vector()
/* This routine initializes this module and plugs it into Alisp. */
{
  ger_debug("init_vector");
  vectordef.type= "vector";
  vectordef.destroy= vector_destroy;
  vectordef.print= vector_print;
  vectordef.setf_method= vector_setf;
}

static void allocate_vectors()
/* This routine adds some new vectors to the free list */
{
  int num_to_alloc;
  Vector_body *new_block;
  int i;

  if (total_vectors) num_to_alloc= total_vectors/2;
  else num_to_alloc= INITIAL_VECTORS;

  ger_debug("allocate_vectors: current total %d; allocating %d more.",
	    total_vectors, num_to_alloc);

  if ( !(new_block= (Vector_body *)malloc(num_to_alloc*sizeof(Vector_body))) )
    ger_fatal("allocate_vectors: unable to allocate %d bytes!",
	      num_to_alloc*sizeof(Vector_body));

  /* String them together and add them to the free list */
  new_block[0].next= free_list;
  for (i=1; i<num_to_alloc; i++) new_block[i].next= &(new_block[i-1]);
  free_list= &(new_block[num_to_alloc - 1]);

  total_vectors += num_to_alloc;
}

static Vector new_vector()
/*
This function creates a vector.  Its x, y, and z components are initially
set to 0.0.
*/
{
  Vector_body *thisvectordata;
  Vector thisvector;
  static int initialized= 0;
  
  ger_debug("new_vector:");

  if (!initialized) {
    init_vector();
    initialized= 1;
  }

  /* Pop a cell off the free list */
  if (!free_list) allocate_vectors();
  thisvectordata= free_list;
  free_list= free_list->next;

  thisvectordata->d.x= DEFAULT_X;
  thisvectordata->d.y= DEFAULT_Y;
  thisvectordata->d.z= DEFAULT_Z;
  
  /* Build the node */
  thisvector= new_node(N_USERDATA);
  ((NODE *)thisvector)->val.userdef.data= (char *)thisvectordata;
  ((NODE *)thisvector)->val.userdef.methods= &vectordef;

  return( thisvector );
}

NODE *vector_create( arglist )
NODE *arglist;
/* This routine creates a new vector instance */
{
  Vector thisvector;
  Vector_body *thisvectordata;

  ger_debug("vector_create:");

  thisvector= new_vector();
  thisvectordata= (Vector_body *)((NODE *)thisvector)->val.userdef.data;

  /* Decode the NODE * here.  It is a list containing the data
   * entries in order.
   */
  if (length_list(arglist) < 3) {
    ger_error("vector_create: need at least 3 arguments!");
    return( NIL );
  }

  if ( realp(safe_car(arglist)) )
    thisvectordata->d.x= get_number(safe_car(arglist));
  else {
    ger_error("vector_create: argument for x not int or real!");
    thisvectordata->d.x= DEFAULT_X;
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thisvectordata->d.y= get_number(safe_car(arglist));
  else {
    ger_error("vector_create: argument for y not int or real!");
    thisvectordata->d.y= DEFAULT_Y;
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thisvectordata->d.z= get_number(safe_car(arglist));
  else {
    ger_error("vector_create: argument for z not int or real!");
    thisvectordata->d.z= DEFAULT_Z;
  }
  arglist= safe_cdr(arglist);

  return(thisvector);
}

Vector create_vector()
/*
This function creates a vector.  It's x, y, and z components are initially
set to 0.0.
*/
{
  Vector thisvector;
  
  ger_debug("vector_create:");

  thisvector= new_vector();
  incr_ref( (NODE *)thisvector );
  return( thisvector );
}

void free_vector( thisvector )
Vector thisvector;
/*
This routine frees the given vector.  If this was the last reference to
this vector, it will be deleted.
*/
{
  ger_debug("free_vector:");
  decr_elem( (NODE *)thisvector );
}

Vector set_vector_x( thisvector, value )
Vector thisvector;
float value;
/* This routine sets the x value of a vector */
{
  Vector_body *thisvectordata;

  ger_debug("set_vector_x:");

  if ((Type_Of((NODE *)thisvector) != N_USERDATA) 
      || (((NODE *)thisvector)->val.userdef.methods != &vectordef))
    ger_error("set_vector_x: passed a non-vector!");
  else {
    Vector_body *thisvectordata;
    thisvectordata= (Vector_body *)((NODE *)thisvector)->val.userdef.data;
    thisvectordata->d.x= value;
  }

  return(thisvector);
}

Vector set_vector_y( thisvector, value )
Vector thisvector;
float value;
/* This routine sets the y value of a vector */
{
  Vector_body *thisvectordata;

  ger_debug("set_vector_y:");

  if ((Type_Of((NODE *)thisvector) != N_USERDATA) 
      || (((NODE *)thisvector)->val.userdef.methods != &vectordef))
    ger_error("set_vector_y: passed a non-vector!");
  else {
    Vector_body *thisvectordata;
    thisvectordata= (Vector_body *)((NODE *)thisvector)->val.userdef.data;
    thisvectordata->d.y= value;
  }

  return(thisvector);
}

Vector set_vector_z( thisvector, value )
Vector thisvector;
float value;
/* This routine sets the z value of a vector */
{
  Vector_body *thisvectordata;

  ger_debug("set_vector_z:");

  if ((Type_Of((NODE *)thisvector) != N_USERDATA) 
      || (((NODE *)thisvector)->val.userdef.methods != &vectordef))
    ger_error("set_vector_z: passed a non-vector!");
  else {
    Vector_body *thisvectordata;
    thisvectordata= (Vector_body *)((NODE *)thisvector)->val.userdef.data;
    thisvectordata->d.z= value;
  }

  return(thisvector);
}

NODE *lisp_vector_x( arglist )
NODE *arglist;
/* This routine provides the (vector-x thisvec) function in lisp */
{
  Vector thisvector;
  float val;
  NODE *result;

  ger_debug("lisp_vector_x:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_vector_x: passed empty list!");
    return(NIL);
  }

  thisvector= (Vector)safe_car(arglist);
  val= vector_x(thisvector);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_vector_y( arglist )
NODE *arglist;
/* This routine provides the (vector-y thisvec) function in lisp */
{
  Vector thisvector;
  float val;
  NODE *result;

  ger_debug("lisp_vector_y:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_vector_y: passed empty list!");
    return(NIL);
  }

  thisvector= (Vector)safe_car(arglist);
  val= vector_y(thisvector);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_vector_z( arglist )
NODE *arglist;
/* This routine provides the (vector-z thisvec) function in lisp */
{
  Vector thisvector;
  float val;
  NODE *result;

  ger_debug("lisp_vector_z:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_vector_z: passed empty list!");
    return(NIL);
  }

  thisvector= (Vector)safe_car(arglist);
  val= vector_z(thisvector);
  result= get_floatrep(val);
  return(result);
}

