/*
 * surface.c
 *
 * Copyright (C) 1989, Craig E. Kolb
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely .  Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Id: surface.c,v 1.1 1991/01/04 18:35:18 welling Exp $
 *
 * $Log: surface.c,v $
 * Revision 1.1  1991/01/04  18:35:18  welling
 * Initial revision
 *
 * Revision 3.0.1.1  89/12/06  16:33:11  craig
 * patch2: Added calls to new error/warning routines.
 * 
 * Revision 3.0  89/10/27  02:06:05  craig
 * Baseline for first official release.
 * 
 */
#include <stdio.h>
#include "constants.h"
#include "typedefs.h"
#include "funcdefs.h"

#define blend(a, b, p, q)	(a * p + b * q)
#define NORMCONST		(1. / 255.)

ray_SurfaceList *Surfaces;		/* Linked list of defined surfaces */

/*
 * Create and return pointer to surface with given properties.
 */
ray_Surface *
make_surface(name, amb, diff, spec, coef, refl, refr, k, translu, stcoef)
char *name;
ray_Color *amb, *diff, *spec;
double coef, refl, refr, k, translu, stcoef;
{
	ray_Surface *stmp;

	/*
	 * Complain if named surface already exists.
	 */
	if ((stmp = get_surface(name)) != (ray_Surface *)0)
		fprintf(stderr,"Surface \"%s\" redefined.", name);

	stmp = (ray_Surface *)Malloc(sizeof(ray_Surface));
	stmp->name = strsave(name);

	stmp->amb = *amb;
	stmp->diff = *diff;
	stmp->spec = *spec;

	stmp->coef = coef; stmp->refl = refl; stmp->transp = refr;
	stmp->kref = k; stmp->translucency = translu;
	stmp->stcoef = stcoef;

	return stmp;
}

ray_SurfaceList *
add_surface(surf, list)
ray_Surface *surf;
ray_SurfaceList *list;
{
	ray_SurfaceList *res;

	res = (ray_SurfaceList *)Malloc(sizeof(ray_SurfaceList));
	res->surf = surf;
	res->next = list;

	return res;
}

/*
 * Search for surface with given name.  If not found, complain and exit.
 */
ray_Surface *
find_surface(name)
char *name;
{
	ray_Surface *stmp;

	stmp = get_surface(name);
	if (stmp == (ray_Surface *)0)
		fprintf(stderr,"Undefined surface \"%s\".", name);

	return stmp;
}

/*
 * Return pointer to surface with given name, NULL if no such surface.
 */
ray_Surface *
get_surface(name)
char *name;
{
	ray_SurfaceList *stmp;

	for (stmp = Surfaces; stmp ; stmp = stmp->next)
		if(strcmp(name, stmp->surf->name) == 0)
			return stmp->surf;
	/*
	 * No surface named "name".
	 */
	return (ray_Surface *)0;
}

/*
 * Compute combination of two surfaces. Resulting surface is copied into surf1.
 */
blend_surface(surf1, surf2, p, q)
ray_Surface *surf1, *surf2;
double p, q;
{
	/*
 	 * P is weight of surf1.  q is weight of surf2.
	 * Result is placed in surf1.
	 */

	blend_color(&surf1->amb, &surf2->amb, p, q);
	blend_color(&surf1->diff, &surf2->diff, p, q);
	blend_color(&surf1->spec, &surf2->spec, p, q);

	surf1->coef = blend(surf1->coef, surf2->coef, p, q);
	surf1->refl = blend(surf1->refl, surf2->refl, p, q);
	surf1->transp = blend(surf1->transp, surf2->transp, p, q);
	surf1->kref = blend(surf1->kref, surf2->kref, p, q);
}

/*
 * Blend two colors.  Result is placed in color1.
 */
blend_color(color1, color2, p, q)
ray_Color *color1, *color2;
double p, q;
{
	color1->r = blend(color1->r, color2->r, p, q);
	color1->g = blend(color1->g, color2->g, p, q);
	color1->b = blend(color1->b, color2->b, p, q);
}

/*
 * Scale color by given red, green and blue factors.
 */
ScaleColor(scalar, color, res)
double scalar;
ray_Color color, *res;
{
	res->r = color.r * scalar;
	res->g = color.g * scalar;
	res->b = color.b * scalar;
}

AddScaledColor(color1, scalar, color2, res)
ray_Color color1, color2, *res;
double scalar;
{
	res->r = color1.r + scalar * color2.r;
	res->g = color1.g + scalar * color2.g;
	res->b = color1.b + scalar * color2.b;
}
