/****************************************************************************
 * point.c
 * Author Joel Welling
 * Copyright 1992, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* 
This module implements points as a user data type.
*/

#include <stdio.h>
#include "ge_error.h"
#include "alisp.h"
#include "p3d.h"
#include "ren.h"

/* Notes-
*/

#define DEFAULT_X 0.0
#define DEFAULT_Y 0.0
#define DEFAULT_Z 0.0
#define INITIAL_POINTS 5000;

/* The USERDATA struct referenced by all points */
static USERDATA pointdef;

/* The list of free point data structures, and total number allocated. */
static Point_body *free_list= (Point_body *)0;
static int total_points= 0;

static void point_print( data )
char *data;
/* This is the print method for the point */
{
  Point_body *vec;
  vec= (Point_body *)data;
  ger_debug("point_print");
  fprintf(stderr,"*point* ");
#ifdef never
  fprintf(stderr,"point: XYZ= (%f, %f, %f)\n",
	  vec->d.x, vec->d.y, vec->d.z);
#endif
}

void point_destroy( data )
char *data;
/* This routine destroys a point instance */
{
  Point_body *thispointdata;

  ger_debug("point_destroy");

  thispointdata= (Point_body *)data;
  thispointdata->next= free_list;
  free_list= thispointdata;
}

static void point_setf( place, location, value )
NODE *place, *location, *value;
/* This routine provides setf methods for point 'structs'. */
{
  float val;
  char *fname, *cname;

  ger_debug("point_setf");

  if ( realp(value) ) val= get_number(value);
  else {
    ger_error("point_setf: argument value not int or real!");
    return;
  }

  fname= getname(place);
  cname= callout_name(place);

  if (!strcmp(fname,"point-x") || !strcmp(cname,"~point-x"))
    set_point_x((Point)location,val);
  else if (!strcmp(fname,"point-y") || !strcmp(cname,"~point-y"))
    set_point_y((Point)location,val);
  else if (!strcmp(fname,"point-z") || !strcmp(cname,"~point-z"))
    set_point_z((Point)location,val);
  else ger_error("point_setf: %s is not a setf-able field of a point.",
		 fname);
}

static void init_point()
/* This routine initializes this module and plugs it into Alisp. */
{
  ger_debug("init_point");
  pointdef.type= "point";
  pointdef.destroy= point_destroy;
  pointdef.print= point_print;
  pointdef.setf_method= point_setf;
}

static void allocate_points()
/* This routine adds some new points to the free list */
{
  int num_to_alloc;
  Point_body *new_block;
  int i;

  if (total_points) num_to_alloc= total_points/2;
  else num_to_alloc= INITIAL_POINTS;

  ger_debug("allocate_points: current total %d; allocating %d more.",
	    total_points, num_to_alloc);

  if ( !(new_block= (Point_body *)malloc(num_to_alloc*sizeof(Point_body))) )
    ger_fatal("allocate_points: unable to allocate %d bytes!",
	      num_to_alloc*sizeof(Point_body));

  /* String them together and add them to the free list */
  new_block[0].next= free_list;
  for (i=1; i<num_to_alloc; i++) new_block[i].next= &(new_block[i-1]);
  free_list= &(new_block[num_to_alloc - 1]);

  total_points += num_to_alloc;
}

static Point new_point()
/*
This function creates a point.  Its x, y, and z components are initially
set to 0.0.
*/
{
  Point_body *thispointdata;
  Point thispoint;
  static int initialized= 0;
  
  ger_debug("new_point:");

  if (!initialized) {
    init_point();
    initialized= 1;
  }

  /* Pop a cell off the free list */
  if (!free_list) allocate_points();
  thispointdata= free_list;
  free_list= free_list->next;

  thispointdata->d.x= DEFAULT_X;
  thispointdata->d.y= DEFAULT_Y;
  thispointdata->d.z= DEFAULT_Z;
  
  /* Build the node */
  thispoint= new_node(N_USERDATA);
  ((NODE *)thispoint)->val.userdef.data= (char *)thispointdata;
  ((NODE *)thispoint)->val.userdef.methods= &pointdef;

  return( thispoint );
}

NODE *point_create( arglist )
NODE *arglist;
/* This routine creates a new point instance */
{
  Point thispoint;
  Point_body *thispointdata;

  ger_debug("point_create:");

  thispoint= new_point();
  thispointdata= (Point_body *)((NODE *)thispoint)->val.userdef.data;

  /* Decode the NODE * here.  It is a list containing the data
   * entries in order.
   */
  if (length_list(arglist) < 3) {
    ger_error("point_create: need at least 3 arguments!");
    return( NIL );
  }

  if ( realp(safe_car(arglist)) )
    thispointdata->d.x= get_number(safe_car(arglist));
  else {
    ger_error("point_create: argument for x not int or real!");
    thispointdata->d.x= DEFAULT_X;
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thispointdata->d.y= get_number(safe_car(arglist));
  else {
    ger_error("point_create: argument for y not int or real!");
    thispointdata->d.y= DEFAULT_Y;
  }
  arglist= safe_cdr(arglist);

  if ( realp(safe_car(arglist)) )
    thispointdata->d.z= get_number(safe_car(arglist));
  else {
    ger_error("point_create: argument for z not int or real!");
    thispointdata->d.z= DEFAULT_Z;
  }
  arglist= safe_cdr(arglist);

  return(thispoint);
}

Point create_point()
/*
This function creates a point.  It's x, y, and z components are initially
set to 0.0.
*/
{
  Point thispoint;
  
  ger_debug("point_create:");

  thispoint= new_point();
  incr_ref( (NODE *)thispoint );
  return( thispoint );
}

void free_point( thispoint )
Point thispoint;
/*
This routine frees the given point.  If this was the last reference to
this point, it will be deleted.
*/
{
  ger_debug("free_point:");
  decr_elem( (NODE *)thispoint );
}

Point set_point_x( thispoint, value )
Point thispoint;
float value;
/* This routine sets the x value of a point */
{
  Point_body *thispointdata;

  ger_debug("set_point_x:");

  if ((Type_Of((NODE *)thispoint) != N_USERDATA) 
      || (((NODE *)thispoint)->val.userdef.methods != &pointdef))
    ger_error("set_point_x: passed a non-point!");
  else {
    Point_body *thispointdata;
    thispointdata= (Point_body *)((NODE *)thispoint)->val.userdef.data;
    thispointdata->d.x= value;
  }

  return(thispoint);
}

Point set_point_y( thispoint, value )
Point thispoint;
float value;
/* This routine sets the y value of a point */
{
  Point_body *thispointdata;

  ger_debug("set_point_y:");

  if ((Type_Of((NODE *)thispoint) != N_USERDATA) 
      || (((NODE *)thispoint)->val.userdef.methods != &pointdef))
    ger_error("set_point_y: passed a non-point!");
  else {
    Point_body *thispointdata;
    thispointdata= (Point_body *)((NODE *)thispoint)->val.userdef.data;
    thispointdata->d.y= value;
  }

  return(thispoint);
}

Point set_point_z( thispoint, value )
Point thispoint;
float value;
/* This routine sets the z value of a point */
{
  Point_body *thispointdata;

  ger_debug("set_point_z:");

  if ((Type_Of((NODE *)thispoint) != N_USERDATA) 
      || (((NODE *)thispoint)->val.userdef.methods != &pointdef))
    ger_error("set_point_z: passed a non-point!");
  else {
    Point_body *thispointdata;
    thispointdata= (Point_body *)((NODE *)thispoint)->val.userdef.data;
    thispointdata->d.z= value;
  }

  return(thispoint);
}

NODE *lisp_point_x( arglist )
NODE *arglist;
/* This routine provides the (point-x thisvec) function in lisp */
{
  Point thispoint;
  float val;
  NODE *result;

  ger_debug("lisp_point_x:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_point_x: passed empty list!");
    return(NIL);
  }

  thispoint= (Point)safe_car(arglist);
  val= point_x(thispoint);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_point_y( arglist )
NODE *arglist;
/* This routine provides the (point-y thisvec) function in lisp */
{
  Point thispoint;
  float val;
  NODE *result;

  ger_debug("lisp_point_y:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_point_y: passed empty list!");
    return(NIL);
  }

  thispoint= (Point)safe_car(arglist);
  val= point_y(thispoint);
  result= get_floatrep(val);
  return(result);
}

NODE *lisp_point_z( arglist )
NODE *arglist;
/* This routine provides the (point-z thisvec) function in lisp */
{
  Point thispoint;
  float val;
  NODE *result;

  ger_debug("lisp_point_z:");

  if (length_list(arglist) < 1) {
    ger_error("lisp_point_z: passed empty list!");
    return(NIL);
  }

  thispoint= (Point)safe_car(arglist);
  val= point_z(thispoint);
  result= get_floatrep(val);
  return(result);
}

