/****************************************************************************
 * p3d.h
 * Author Joel Welling, Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This file contains type definitions for P3D structures, and
definitions needed to access the functions in struct_ops.c, which 
provide C access to P3D lisp structures.  If the lisp underlieing
P3D or p3d.lsp is changed, this file will probably need to be revised.
*/

/* Typedefs, used to make drivers less lisp-specific. */
typedef NODE *Primitive;	/* primitive struct */
typedef NODE *Gob;		/* gob struct */
typedef NODE *Point;		/* point struct */
typedef NODE *Vector;		/* vector struct */
typedef NODE *Vertex;		/* vertex struct */
typedef NODE *Color;		/* color struct */
typedef NODE *Camera;		/* camera struct */
typedef NODE *Facet_list;	/* list of Vertex_list's */
typedef NODE *Vertex_list;	/* list of vertex structs */
typedef NODE *Child_list;	/* list of Gob's */
typedef NODE *Transformation;	/* 4 by 4 transformation matrix */
typedef NODE *Function;		/* evaluatable list */
typedef NODE *Symbol;		/* SYM inode */
typedef NODE *Pair;		/* dotted pair */
typedef NODE *Attribute_list;	/* assoc list of Pair's */

/* Structure to hold material info */
struct Material_struct {
  float ka; /* ambient reflection coefficient */
  float kd; /* diffuse */
  float ks; /* specular */
  float exp; /* specular exponent */
  float reflect; /* reflectivity */
  float refract; /* index of refraction */
  Color energy; /* surface energy (for radiosity) */
  char *ren_data; /* hook on which to hang renderer data */
};
typedef struct Material_struct *Material;

/* Structure to hold color info */
typedef union Color_body_struct {
  struct {
    float r; /* red */
    float g; /* green */
    float b; /* blue */
    float a; /* alpha */
  } d;
  union Color_body_struct *next;
} Color_body;

/* Structure to hold point info */
typedef union Point_body_struct {
  struct {
    float x; /* x component */
    float y; /* y component */
    float z; /* z component */
  } d;
  union Point_body_struct *next;
} Point_body;

/* Structure to hold vector info */
typedef union Vector_body_struct {
  struct {
    float x; /* x component */
    float y; /* y component */
    float z; /* z component */
  } d;
  union Vector_body_struct *next;
} Vector_body;

/* Structure to hold vertex info */
typedef union Vertex_body_struct {
  struct {
    float x; /* x component */
    float y; /* y component */
    float z; /* z component */
    Color color; /* handle for the color node */
    Vector normal; /* handle for the vertex normal node */
    int index;
  } d;
  union Vertex_body_struct *next;
} Vertex_body;

/* macros and functions for manipulating the above types */

/* General list and object manipulation functions */
extern int length_list();
extern void grab();
extern void ungrab();

/* Transformation access functions */
extern float *array2d_to_c();
extern Transformation array2d_to_lisp();
extern void free_transformation();

/* Safe lisp accessors */
#define realp(x) ( floatp(x) || fixp(x) )
#define get_number(x) ( floatp(x) ? getflonum(x) : (fixp(x) ? getfixnum(x) : 0))
#define safe_getname(x) ( symbolp(x) ? getname(x) : "undefined")
#define safe_cdr(x) ( consp(x) ? cdr(x) : NIL)
#define safe_car(x) ( consp(x) ? car(x) : NIL)
#define safe_aref(x,n) ( (arrayp(x) && (array_max(x) > n)) ? array_ref(x,n) : NIL)

/* Child_list access functions and macros */
#define first_child( childlist ) (Gob)safe_car( (NODE *)(childlist) )
#define rest_of_children( childlist ) (Child_list)safe_cdr( (NODE *)(childlist) )

/* Vertex_list access functions and macros */
extern Vertex_list create_vertex_list();
extern void free_vertex_list();
#define first_vertex( vlist ) (Vertex)car( (NODE *)(vlist) )
#define u_first_vertex( vlist ) (Vertex)car( (NODE *)(vlist) )
extern Vertex_list set_first_vertex();
#define rest_of_vertices( vlist ) (Vertex_list)cdr( (NODE *)(vlist) )

/* Facet_list access functions and macros */
extern Facet_list create_facet_list();
extern void free_facet_list();
#define first_facet( flist ) (Vertex_list)car( (NODE *)(flist) )
#define u_first_facet( flist ) (Vertex_list)car( (NODE *)(flist) )
extern Facet_list set_first_facet();
#define rest_of_facets( flist ) (Facet_list)cdr( (NODE *)(flist) )

/* Attribute list, Symbol and Pair access functions and macros */
extern Symbol create_symbol();
extern Attribute_list merge_attribute_lists();
extern void free_attribute_list();
#define first_pair( attrlist ) (Pair)safe_car( (NODE *)(attrlist) )
#define rest_of_pairs( attrlist ) (Attribute_list)safe_cdr( (NODE *)(attrlist) )
#define symbol_equality( sym1, sym2 ) ( sym1 == sym2 )
#define symbol_string( symbol ) (char *)getstring( lookup( (NODE *)(symbol) ) )
#define symbol_attr( symbol, attr ) \
	(Pair)assoc( (NODE *)(symbol) , (NODE *)(attr) )
#define pair_symbol( thispair ) (Symbol)safe_car( (NODE *)(thispair) )
#define pair_color( thispair ) (Color)safe_cdr( (NODE *)(thispair) )
#define pair_boolean( thispair ) !null( safe_cdr( (NODE *)(thispair) ) )
#define pair_float( thispair ) (float)get_number( safe_cdr( (NODE *)(thispair) ) ) 
#define pair_int( thispair ) (int)get_number( safe_cdr( (NODE *)(thispair) ) ) 
#define pair_material( thispair ) \
  ( consp(thispair) ? (Material)(cdr((NODE *)(thispair))->val.userdef.data) \
   : (Material)0 )
#define pair_string( thispair ) \
  ( ( consp(thispair) && stringp( cdr(thispair) ) ) ? \
   (char *)getstring(cdr((NODE *)(thispair))) : (char *)0 )

/* Funtion access storage and macro */
#define eval_function( thisfun ) \
	(void)eval_ilisp( (NODE *)(thisfun) )

/* Primitive access macros */
#define primitive_id(aprim)  ( (int)get_number( safe_aref((NODE *)aprim,1) ) )
#define primitive_op(aprim)  ( (Function)( safe_aref((NODE *)aprim,2) ) )

/* gob structure access functions */
#define gob_primitive(agob) ( (Primitive)safe_aref((NODE *)agob,3) )
#define gob_idnum(agob) ( (int)get_number(safe_aref((NODE *)agob,1)) )
#define gob_children(agob) ( (Child_list)safe_aref((NODE *)agob,6) )
#define gob_attr(agob) ( (Attribute_list)safe_aref((NODE *)agob,4) )
#define gob_trans(agob) ( (Transformation)safe_aref((NODE *)agob,5) )
#define gob_parentcount(agob) ( (int)get_number(safe_aref((NODE *)agob,2)) )

/* color structure access functions and macros */
extern Color create_color();
extern void free_color();
#define color_red( colornode ) \
  (((Color_body *)(((NODE *)colornode)->val.userdef.data))->d.r)
#define color_green( colornode ) \
  (((Color_body *)(((NODE *)colornode)->val.userdef.data))->d.g)
#define color_blue( colornode ) \
  (((Color_body *)(((NODE *)colornode)->val.userdef.data))->d.b)
#define color_alpha( colornode ) \
  (((Color_body *)(((NODE *)colornode)->val.userdef.data))->d.a)
extern Color set_color_red(); /* These supply setting functions from C */
extern Color set_color_green();
extern Color set_color_blue();
extern Color set_color_alpha();
extern NODE *color_create(); /* This is the lisp creation function */
extern NODE *lisp_color_r();  /* These supply access functions from lisp */
extern NODE *lisp_color_g();
extern NODE *lisp_color_b();
extern NODE *lisp_color_a();

/* point structure access functions and macros */
extern Point create_point();
extern void free_point();
#define point_x( pointnode ) \
  (((Point_body *)(((NODE *)pointnode)->val.userdef.data))->d.x)
#define point_y( pointnode ) \
  (((Point_body *)(((NODE *)pointnode)->val.userdef.data))->d.y)
#define point_z( pointnode ) \
  (((Point_body *)(((NODE *)pointnode)->val.userdef.data))->d.z)
extern Point set_point_x(); /* These supply setting functions from C */
extern Point set_point_y();
extern Point set_point_z();
extern NODE *point_create(); /* This is the lisp creation function */
extern NODE *lisp_point_x();  /* These supply access functions from lisp */
extern NODE *lisp_point_y();
extern NODE *lisp_point_z();

/* vector structure access functions and macros */
extern Vector create_vector();
extern void free_vector();
#define vector_x( vectornode ) \
  (((Vector_body *)(((NODE *)vectornode)->val.userdef.data))->d.x)
#define vector_y( vectornode ) \
  (((Vector_body *)(((NODE *)vectornode)->val.userdef.data))->d.y)
#define vector_z( vectornode ) \
  (((Vector_body *)(((NODE *)vectornode)->val.userdef.data))->d.z)
extern Vector set_vector_x(); /* These supply setting functions from C */
extern Vector set_vector_y();
extern Vector set_vector_z();
extern NODE *vector_create(); /* This is the lisp creation function */
extern NODE *lisp_vector_x();  /* These supply access functions from lisp */
extern NODE *lisp_vector_y();
extern NODE *lisp_vector_z();

/* vertex structure access functions and macros */
extern Vertex create_vertex();
extern void free_vertex();
#define vertex_x( vertexnode ) \
  (((Vertex_body *)(((NODE *)vertexnode)->val.userdef.data))->d.x)
#define vertex_y( vertexnode ) \
  (((Vertex_body *)(((NODE *)vertexnode)->val.userdef.data))->d.y)
#define vertex_z( vertexnode ) \
  (((Vertex_body *)(((NODE *)vertexnode)->val.userdef.data))->d.z)
#define vertex_color( vertexnode ) \
  (((Vertex_body *)(((NODE *)vertexnode)->val.userdef.data))->d.color)
#define vertex_normal( vertexnode ) \
  (((Vertex_body *)(((NODE *)vertexnode)->val.userdef.data))->d.normal)
#define vertex_index( vertexnode ) \
  (((Vertex_body *)(((NODE *)vertexnode)->val.userdef.data))->d.index)
extern Vertex set_vertex_x(); /* These supply setting functions from C */
extern Vertex set_vertex_y();
extern Vertex set_vertex_z();
extern Vertex set_vertex_color();
extern Vertex set_vertex_normal();
extern Vertex set_vertex_index();
extern NODE *vertex_create(); /* This is the lisp creation function */
extern NODE *lisp_vertex_x();  /* These supply access functions from lisp */
extern NODE *lisp_vertex_y();
extern NODE *lisp_vertex_z();
extern NODE *lisp_vertex_color();
extern NODE *lisp_vertex_normal();
extern NODE *lisp_vertex_index();

/* camera structure access functions and macros */
extern Camera create_camera();
extern void free_camera();
#define camera_lookfrom( camera ) ( (Point)safe_aref((NODE *)camera,1) )
#define u_camera_lookfrom( camera ) ( (Point)array_ref((NODE *)camera,1) )
extern Camera set_camera_lookfrom();
#define camera_lookat(camera) ( (Point)safe_aref((NODE *)camera,2) )
#define u_camera_lookat(camera) ( (Point)array_ref((NODE *)camera,2) )
extern Camera set_camera_lookat();
#define camera_up(camera) ( (Vector)safe_aref((NODE *)camera,3) )
#define  u_camera_up(camera) ( (Vector)array_ref((NODE *)camera,3) )
extern Camera set_camera_up();
#define camera_fovea(camera) \
	 ( (float)get_number( safe_aref((NODE *)camera,4) ) )
#define u_camera_fovea(camera) \
	 ( (float)getflonum( array_ref((NODE *)camera,4) ) )
extern Camera set_camera_fovea();
#define camera_hither(camera) \
	 ( (float)get_number( safe_aref((NODE *)camera,5) ) )
extern Camera set_camera_hither();
#define camera_yon(camera) ( (float)get_number( safe_aref((NODE *)camera,6) ) )
extern Camera set_camera_yon();
#define camera_background(camera) ( (Color)safe_aref((NODE *)camera,7) )
extern Camera set_camera_background();

/* Material creation routine and related macros */
extern NODE *material_create();
#define material_ka( thismat ) ( thismat->ka )
#define material_kd( thismat ) ( thismat->kd )
#define material_ks( thismat ) ( thismat->ks )
#define material_exp( thismat ) ( thismat->exp )
#define material_reflect( thismat ) ( thismat->reflect )
#define material_refract( thismat ) ( thismat->refract )
#define material_energy( thismat ) ( thismat->energy )
#define material_ren_data( thismat ) (thismat->ren_data)
