/****************************************************************************
 * motif_view.c
 * Authors Brian Welcker, Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* This module provides functions associated with the 'view' menu in
 * the Motif user interface.
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#ifndef VMS
#include <unistd.h>
#include <sys/param.h>
#else
#include stdlib
#define R_OK 4  /* needed for access() */
#endif

#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/MainW.h>
#include <Xm/DialogS.h>
#include <Xm/RowColumnP.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/CascadeB.h>
#include <Xm/MessageB.h>
#include <Xm/Form.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>
#include <Xm/TextF.h>
#include <Xm/LabelG.h>

#include "ge_error.h"
#include "mi.h"
#include "motif_mi.h"

static char cam_help_string[]= 
"This dialog contains information about the camera that \
was in use at the moment the dialog was created.  It will \
stay around until you press the 'close' button, at which \
point the camera information in the dialog is forgotten.\n \
If the view in the main window is changed, the new camera  \
information can be seen either by popping up another camera \
dialog or by pressing the 'update' button in this dialog.  \
To set the camera in the main window to the values shown  \
in this window, press the 'set' button.  Note that the data \
in this dialog can be edited, and the 'set' button can then \
be used to set the camera to the newly edited values.";

static float val_from_named_child(w,name)
Widget w;
char *name;
/* This routine searches the decendents of the given widget, looking
 * for a child of the given name.  When such a child is found, its
 * value is interpreted as a character string denoting a float.
 */
{
  char *vstring;
  float val;

  ger_debug("val_from_named_child:");

  w= XtNameToWidget(w,name);
  if (!w) 
    ger_fatal("val_from_named_child: couldn't find the needed child <%s>!",
	      name);
  vstring= XmTextGetString(w);
  if (!vstring) 
    ger_fatal("val_from_named_child: couldn't get string from child <%s>!");
  val= atof(vstring);
  XtFree(vstring);
  return(val);
}

static void val_into_named_child(w,name,val)
Widget w;
char *name;
float val;
/* This routine searches the decendents of the given widget, looking
 * for a child of the given name.  When such a child is found, the 
 * given float is written into its value.
 */
{
  char buf[20];

  ger_debug("val_into_named_child:");

  w= XtNameToWidget(w,name);
  if (!w) 
    ger_fatal("val_into_named_child: couldn't find the needed child <%s>!",
	      name);
  sprintf(buf,"%10f",val);
  XmTextSetString(w,buf);
}

static void set_cam_callback(w, settings, call_data)
Widget w;
Widget settings;
caddr_t call_data;
/* This sets the current camera to that given in the widget. */
{
  Viewpoint view;

  ger_debug("set_cam_callback:");

  view.from.x= val_from_named_child(settings,"from_x");
  view.from.y= val_from_named_child(settings,"from_y");
  view.from.z= val_from_named_child(settings,"from_z");
  view.at.x= val_from_named_child(settings,"at_x");
  view.at.y= val_from_named_child(settings,"at_y");
  view.at.z= val_from_named_child(settings,"at_z");
  view.up.x= val_from_named_child(settings,"up_x");
  view.up.y= val_from_named_child(settings,"up_y");
  view.up.z= val_from_named_child(settings,"up_z");
  view.fovea= val_from_named_child(settings,"fovea");
  view.hither= val_from_named_child(settings,"hither");
  view.yon= val_from_named_child(settings,"yon");
  
  miu_set_viewpoint(&view);
}

static void update_cam_callback(w, settings, call_data)
Widget w;
Widget settings;
caddr_t call_data;
/* This sets the current camera to that given in the widget. */
{
  Viewpoint *view;

  ger_debug("update_cam_callback:");
  view= miu_viewpoint();

  val_into_named_child(settings,"from_x",view->from.x);
  val_into_named_child(settings,"from_y",view->from.y);
  val_into_named_child(settings,"from_z",view->from.z);
  val_into_named_child(settings,"at_x",view->at.x);
  val_into_named_child(settings,"at_y",view->at.y);
  val_into_named_child(settings,"at_z",view->at.z);
  val_into_named_child(settings,"up_x",view->up.x);
  val_into_named_child(settings,"up_y",view->up.y);
  val_into_named_child(settings,"up_z",view->up.z);
  val_into_named_child(settings,"fovea",view->fovea);
  val_into_named_child(settings,"hither",view->hither);
  val_into_named_child(settings,"yon",view->yon);

  free(view);
}

static void destroy_callback(w, target_of_destruction, call_data)
Widget w;
Widget target_of_destruction;
caddr_t call_data;
/* This destroys the calling widget */
{
  ger_debug("destroy_callback:");

  XtDestroyWidget(target_of_destruction);
}

static void cam_help_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
/* This gives information on this dialog. */
{
  static Widget help_box= (Widget)0;
  char message[BUFSIZ];

  ger_debug("cam_help_callback:");

  if (!help_box) 
    help_box= mil_motif_help_box(w,"CameraInfo Help",cam_help_string);
		      
  XtManageChild(help_box);
}

static Widget add_managed_label_gadget(parent,string)
Widget parent;
char *string;
{
  Arg args[MAX_ARGS];
  int n;
  Widget w;

  ger_debug("add_managed_label_gadget:");

  n= 0;
  w= XmCreateLabelGadget(parent,string, args, n);
  XtManageChild(w);
  return(w);
}

static void num_text_check(text_w, unused, cbs)
Widget text_w;
caddr_t unused;
XmTextVerifyCallbackStruct *cbs;
/* This callback verifies that a text field change is leaving a valid number */
{
  int len= 0;
  int c;

  /* Ignore backspace */
  if (cbs->startPos < cbs->currInsert) return;

  /* Make sure it's all either numbers or '.', deleting anything that isn't */
  for (len=0; len<cbs->text->length; len++) {
    c= cbs->text->ptr[len];
    if (!isdigit(c) && (c != '.') && (c != '-')) { /* reject it */
      int i;
      for (i=len; (i+1)<cbs->text->length; i++)
	cbs->text->ptr[i]= cbs->text->ptr[i+1];
      cbs->text->length--;
      len--;
    }
  }
  if (cbs->text->length == 0) cbs->doit= False;
}

static Widget add_managed_text_widget(parent,string,val)
Widget parent;
char *string;
float val;
{
  Arg args[MAX_ARGS];
  int n;
  Widget w;
  char buf[20];

  ger_debug("add_managed_text_widget:");

  n= 0;
  XtSetArg(args[n], XmNcolumns, 10); n++;
  w= XmCreateText(parent, string, args, n);
  sprintf(buf,"%10f",val);
  XmTextSetString(w,buf);
  XtAddCallback(w, XmNmodifyVerifyCallback, num_text_check, NULL);
  XtManageChild(w);
  return(w);
}

static void cam_info_callback(w, client_data, call_data)
/* This pops up a camera info window */
Widget w;
caddr_t client_data;
caddr_t call_data;
{
  Viewpoint *cam;
  Widget dialogshell, dialog, action_area, separator, settings, headings;
  Widget close_button, set_button, help_button, update_button;
  XmString title_string;
  Arg args[MAX_ARGS];
  int i, j, n= 0;
  static char *text_labels[]= {
    "look from:", "look at:", "up:", "fovea:", "hither:", "yon:"
    };

  ger_debug("cam_info_callback:");

  cam= miu_viewpoint();

  /* Create the main dialogshell and form dialog */
  n = 0;
  XtSetArg(args[n], XmNdeleteResponse, XmDESTROY); n++;
  XtSetArg(args[n], XmNautoUnmanage, False); n++;
  dialogshell= XmCreateDialogShell(w,"CameraInfo",args,n);
  n = 0;
  XtSetArg(args[n], XmNnoResize, True); n++;
  dialog= XmCreateForm(dialogshell,"camera_info_dialog",args,n);

  /* Create headings */
  n= 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNfractionBase, 8); n++;
  headings= XmCreateForm(dialog,"headings",args,n);
  n= 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNrightPosition, 3); n++;
  XtManageChild( XmCreateLabelGadget(headings,"x", args, n) );
  n= 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNrightPosition, 5); n++;
  XtManageChild( XmCreateLabelGadget(headings,"y", args, n) );
  n= 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNrightPosition, 7); n++;
  XtManageChild( XmCreateLabelGadget(headings,"z", args, n) );
  XtManageChild(headings);

  /* Create the value array */
  n = 0;
  XtSetArg(args[n], XmNpacking, XmPACK_COLUMN); n++;
  XtSetArg(args[n], XmNnumColumns, XtNumber(text_labels)); n++;
  XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
  XtSetArg(args[n], XmNisAligned, True); n++;
  XtSetArg(args[n], XmNentryAlignment, XmALIGNMENT_END); n++;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(args[n], XmNtopWidget, headings); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  settings= XmCreateRowColumn(dialog, "settings", args, n);
  /* from line */
  (void)add_managed_label_gadget(settings,text_labels[0]);
  (void)add_managed_text_widget(settings,"from_x",cam->from.x);
  (void)add_managed_text_widget(settings,"from_y",cam->from.y);
  (void)add_managed_text_widget(settings,"from_z",cam->from.z);
  /* at line */
  (void)add_managed_label_gadget(settings,text_labels[1]);
  (void)add_managed_text_widget(settings,"at_x",cam->at.x);
  (void)add_managed_text_widget(settings,"at_y",cam->at.y);
  (void)add_managed_text_widget(settings,"at_z",cam->at.z);
  /* up line */
  (void)add_managed_label_gadget(settings,text_labels[2]);
  (void)add_managed_text_widget(settings,"up_x",cam->up.x);
  (void)add_managed_text_widget(settings,"up_y",cam->up.y);
  (void)add_managed_text_widget(settings,"up_z",cam->up.z);
  /* fovea */
  (void)add_managed_label_gadget(settings,text_labels[3]);
  (void)add_managed_text_widget(settings,"fovea",cam->fovea);
  (void)add_managed_label_gadget(settings,"");
  (void)add_managed_label_gadget(settings,"");
  /* hither */
  (void)add_managed_label_gadget(settings,text_labels[4]);
  (void)add_managed_text_widget(settings,"hither",cam->hither);
  (void)add_managed_label_gadget(settings,"");
  (void)add_managed_label_gadget(settings,"");
  /* yon */
  (void)add_managed_label_gadget(settings,text_labels[5]);
  (void)add_managed_text_widget(settings,"yon",cam->yon);
  (void)add_managed_label_gadget(settings,"");
  (void)add_managed_label_gadget(settings,"");
  /* manage the whole settings widget */
  XtManageChild(settings);

  /* Separator */
  n= 0;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(args[n], XmNtopWidget, settings); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  separator= XmCreateSeparator(dialog,"separator",args,n);
  XtManageChild(separator);

  /* Action area */
  n= 0;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
  XtSetArg(args[n], XmNtopWidget, separator); n++;
  XtSetArg(args[n], XmNfractionBase, 7); n++;
  action_area= XmCreateForm(dialog,"action_area",args,n);

  /* Close button */
  title_string= XmStringCreate("Close", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'C'); n++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNleftPosition, 0); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNrightPosition, 1); n++;
  close_button= XmCreatePushButton(action_area,"Close",args,n);
  XtAddCallback(close_button, XmNactivateCallback, 
		destroy_callback, dialogshell);
  if (title_string) XmStringFree(title_string);
  XtManageChild(close_button);

  /* Update button */
  title_string= XmStringCreate("Update", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'U'); n++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNleftPosition, 2); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNrightPosition, 3); n++;
  update_button= XmCreatePushButton(action_area,"Update",args,n);
  XtAddCallback(update_button, XmNactivateCallback, 
		update_cam_callback, settings);
  if (title_string) XmStringFree(title_string);
  XtManageChild(update_button);

  /* Set button */
  title_string= XmStringCreate("Set", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'S'); n++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNleftPosition, 4); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNrightPosition, 5); n++;
  set_button= XmCreatePushButton(action_area,"Set",args,n);
  XtAddCallback(set_button, XmNactivateCallback, 
		set_cam_callback, settings);
  if (title_string) XmStringFree(title_string);
  XtManageChild(set_button);

  /* Help button */
  title_string= XmStringCreate("Help", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'H'); n++;
  XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
  XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNleftPosition, 6); n++;
  XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
  XtSetArg(args[n], XmNrightPosition, 7); n++;
  help_button= XmCreatePushButton(action_area,"Help",args,n);
  XtAddCallback(help_button, XmNactivateCallback, cam_help_callback, NULL);
  if (title_string) XmStringFree(title_string);
  XtManageChild(help_button);

  XtManageChild(action_area);

  /* Set default button and manage the dialog */
  XtVaSetValues(dialog, XmNdefaultButton, close_button, NULL);
  XtManageChild(dialog);

  free(cam);
}

void mil_motif_view_setup( menu_bar )
Widget menu_bar;
/* This routine creates the View pulldown menu. */
{
  Widget button;
  Widget menu_pane;
  XmString title_string;

  Arg    args[MAX_ARGS];
  int    n;

  ger_debug("mil_motif_view_setup:");

  /* Create new menu pane */
  n = 0;
  menu_pane = XmCreatePulldownMenu(menu_bar, "menu_pane", args, n);
  
  /* Create View Menu */
  title_string= XmStringCreate("View", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNsubMenuId, menu_pane); n++;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'V'); n++;
  button = XmCreateCascadeButton(menu_bar, "View", args, n);
  if (title_string) XmStringFree(title_string);
  XtManageChild(button);
  
  /* Create Camera Info button */
  title_string= XmStringCreate("Camera Info", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'C'); n++;
  button = XmCreatePushButton(menu_pane, "camera info", args, n);
  XtAddCallback(button, XmNactivateCallback, cam_info_callback, NULL);
  if (title_string) XmStringFree(title_string);
  XtManageChild(button);
  
  /* Create Save Trace button */
  title_string= XmStringCreate("Save Trace", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'T'); n++;
  button = XmCreateToggleButton(menu_pane, "save trace", args, n);
  XtAddCallback(button, XmNvalueChangedCallback, miu_toggle_trace, NULL);
  if (title_string) XmStringFree(title_string);
  XtManageChild(button);
}
