/****************************************************************************
 * motif_flyby.c
 * Authors Brian Welcker, Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* This module provides functions associated with the 'view' menu in
 * the Motif user interface.
 */

#include <stdio.h>
#include <ctype.h>
#include <math.h>
#ifndef VMS
#include <unistd.h>
#include <sys/param.h>
#else
#define R_OK 4  /* needed for access() */
#define F_OK 0  /* needed for access() */
#endif

#include <X11/Xatom.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#include <Xm/Xm.h>
#include <Xm/RowColumn.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/CascadeB.h>
#include <Xm/DialogS.h>
#include <Xm/Form.h>
#include <Xm/Separator.h>
#include <Xm/Text.h>
#include <Xm/LabelG.h>

#include "ge_error.h"
#include "mi.h"
#include "motif_mi.h"
#include "flyby.h"

/* Default number of frames in flyby */
#define INITIAL_FLYBY_FRAMES 300

/* Current setting for flyby frames */
static int flyby_frames= INITIAL_FLYBY_FRAMES;

/* Some status flags */
static int path_ready= 0;
static int path_generated= 0;
static int option_flags= 0;

static char flyby_help_string[]= 
"This dialog contains settings which describe the flyby \
to be generated. \n\n\
The 'frames' field gives the number of frames in the flyby. \
For video animation, 30 frames are needed for each second \
of video. \n\n\
If the 'constant time between keyframes' box is set, roughly \
the same number of views will be generated between each pair \
of keyframes, regardless of how far apart in space the keyframes \
are.  If the 'steady motion' box is set, more views will be \
generated between keyframes which are farther apart.  These \
two options are mutually exclusive. \n\n\
If the 'start from rest' box is set, a flyby will be generated \
in which the model is initially stationary and accelerates up \
to speed.  If 'end at rest' is checked, the model will slow \
down and stop at the end of the flyby. \n\n\
The 'OK' button puts any changes you make into effect, and \
closes the dialog.  The 'Cancel' button closes the dialog \
and forgets about your changes.  The 'Reset' button sets \
the values back to those that were in force when the appeared.";

static void int_into_named_child(w,name,val)
Widget w;
char *name;
int val;
/* This routine searches the decendents of the given widget, looking
 * for a child of the given name.  When such a child is found, the 
 * given int is written into its value.
 */
{
  char buf[20];

  ger_debug("int_into_named_child:");

  w= XtNameToWidget(w,name);
  if (!w) 
    ger_fatal("int_into_named_child: couldn't find the needed child <%s>!",
	      name);
  sprintf(buf,"%10d",val);
  XmTextSetString(w,buf);
}

static int int_from_named_child(w,name)
Widget w;
char *name;
/* This routine searches the decendents of the given widget, looking
 * for a child of the given name.  When such a child is found, its
 * value is interpreted as a character string denoting an int.
 */
{
  char *vstring;
  int val;

  ger_debug("int_from_named_child:");

  w= XtNameToWidget(w,name);
  if (!w) 
    ger_fatal("int_from_named_child: couldn't find the needed child <%s>!",
	      name);
  vstring= XmTextGetString(w);
  if (!vstring) 
    ger_fatal("int_from_named_child: couldn't get string from child <%s>!");
  val= atoi(vstring);
  XtFree(vstring);
  return(val);
}

static void settings_to_widget(w)
Widget w;
/* This routine copies the settings into the widget */
{
  ger_debug("settings_to_widget:");

  XmToggleButtonSetState( XtNameToWidget(w,"time_scale.button_0"), 
			 !(option_flags & STEADY_MOTION_FLAG) , False);
  XmToggleButtonSetState( XtNameToWidget(w,"time_scale.button_1"), 
			 (option_flags & STEADY_MOTION_FLAG) , False);
  XmToggleButtonSetState( XtNameToWidget(w,"accelerations.button_0"), 
			 (option_flags & START_FROM_REST_FLAG) , False);
  XmToggleButtonSetState( XtNameToWidget(w,"accelerations.button_1"), 
			 (option_flags & END_AT_REST_FLAG) , False);
  int_into_named_child(w,"frames.frames_text",flyby_frames);
}

static int settings_from_widget(w)
Widget w;
/* This routine copies the settings out of the widget.  If everything
 * is OK, it returns non-zero.
 */
{
  int newframes;

  ger_debug("settings_from_widget:");

  if (XmToggleButtonGetState( XtNameToWidget(w,"time_scale.button_0") ))
    option_flags &= ~STEADY_MOTION_FLAG;
  if (XmToggleButtonGetState( XtNameToWidget(w,"time_scale.button_1") ))
    option_flags |= STEADY_MOTION_FLAG;
  if (XmToggleButtonGetState( XtNameToWidget(w,"accelerations.button_0") ))
    option_flags |= START_FROM_REST_FLAG;
  else option_flags &= ~START_FROM_REST_FLAG;
  if (XmToggleButtonGetState( XtNameToWidget(w,"accelerations.button_1") ))
    option_flags |= END_AT_REST_FLAG;
  else option_flags &= ~END_AT_REST_FLAG;

  newframes= int_from_named_child(w,"frames.frames_text");
  if (newframes<2) {
    mil_motif_alert_box(w,"There must be at least 2 frames in the flyby!");
    return(0);
  }
  else flyby_frames= newframes;

  path_generated= 0; /* to force regeneration with new settings */
  return(1);
}

static void fly_settings_help_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
/* This gives information on the flyby settings dialog. */
{
  static Widget help_box= (Widget)0;
  char message[BUFSIZ];

  ger_debug("fly_settings_help_callback:");

  if (!help_box) 
    help_box= mil_motif_help_box(w,"Flyby Help",flyby_help_string);
		      
  XtManageChild(help_box);
}

static void ok_callback(w, dialog, call_data)
Widget w;
Widget dialog;
caddr_t call_data;
/* This routine clears the flyby */
{
  ger_debug("new_path_callback:");

  if (settings_from_widget(dialog))
    XtUnmanageChild(dialog);
}

static void reset_callback(w, dialog, call_data)
Widget w;
Widget dialog;
caddr_t call_data;
/* This routine clears the flyby */
{
  ger_debug("new_path_callback:");

  settings_to_widget(dialog);
}

static void cancel_callback(w, dialog, call_data)
Widget w;
Widget dialog;
caddr_t call_data;
/* This routine clears the flyby */
{
  ger_debug("cancel_callback:");

  settings_to_widget(dialog);
  XtUnmanageChild(dialog);
}

static void num_text_check(text_w, unused, cbs)
Widget text_w;
caddr_t unused;
XmTextVerifyCallbackStruct *cbs;
/* This callback verifies that a text field change is leaving a valid 
 * positive number. 
 */
{
  int len= 0;
  int c;

  /* Ignore backspace */
  if (cbs->startPos < cbs->currInsert) return;

  /* Make sure it's all either numbers or '.', deleting anything that isn't */
  for (len=0; len<cbs->text->length; len++) {
    c= cbs->text->ptr[len];
    if (!isdigit(c) && (c != '.')) { /* reject it */
      int i;
      for (i=len; (i+1)<cbs->text->length; i++)
	cbs->text->ptr[i]= cbs->text->ptr[i+1];
      cbs->text->length--;
      len--;
    }
  }
  if (cbs->text->length == 0) cbs->doit= False;
}

static void settings_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
/* This routine generates (if necessary) the settings panel, and displays it */
{
  static Widget dialog= (Widget)0;
  Widget dialogshell, action_area, separator;
  Widget frames, frames_text, time_scale_box, accel_box;
  Widget ok_button, reset_button, help_button, cancel_button;
  XmString title_string, str1, str2;
  Arg args[MAX_ARGS];
  int n= 0;

  ger_debug("settings_callback:");

  if (!dialog) { 

    /* Create the main dialogshell and form dialog */
    n = 0;
    XtSetArg(args[n], XmNdeleteResponse, XmDESTROY); n++;
    XtSetArg(args[n], XmNautoUnmanage, False); n++;
    dialogshell= XmCreateDialogShell(w,"FlybySettings",args,n);
    n = 0;
    XtSetArg(args[n], XmNnoResize, True); n++;
    dialog= XmCreateForm(dialogshell,"flyby_settings_dialog",args,n);

    /* Create the frames field */
    n = 0;
    XtSetArg(args[n], XmNpacking, XmPACK_COLUMN); n++;
    XtSetArg(args[n], XmNnumColumns, 1); n++;
    XtSetArg(args[n], XmNorientation, XmHORIZONTAL); n++;
    XtSetArg(args[n], XmNisAligned, True); n++;
    XtSetArg(args[n], XmNentryAlignment, XmALIGNMENT_END); n++;
    XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
    frames= XmCreateRowColumn(dialog, "frames", args, n);
    n= 0;
    XtManageChild( XmCreateLabelGadget(frames,"Frames in flyby:", args, n) );
    n= 0;
    XtSetArg(args[n], XmNcolumns, 10); n++;
    frames_text= XmCreateText(frames, "frames_text", args, n);
    XtAddCallback(frames_text, XmNmodifyVerifyCallback, num_text_check, NULL);
    XtManageChild(frames_text);
    XtManageChild(frames);

    /* Create the time scale options radio box */
    str1= XmStringCreateSimple("constant time between keyframes");
    str2= XmStringCreateSimple("steady motion");
    time_scale_box= XmVaCreateSimpleRadioBox(dialog,"time_scale",
		       0, NULL,
		       XmVaRADIOBUTTON, str1, NULL, NULL, NULL,
		       XmVaRADIOBUTTON, str2, NULL, NULL, NULL,
		       XmNtopAttachment, XmATTACH_WIDGET,
		       XmNtopWidget, frames,
		       XmNleftAttachment, XmATTACH_FORM,
		       XmNrightAttachment, XmATTACH_FORM,
		       NULL);
    XmStringFree(str1);
    XmStringFree(str2);
    XtManageChild(time_scale_box);

    /* Create the acceleration button check box */
    str1= XmStringCreateSimple("start from rest");
    str2= XmStringCreateSimple("end at rest");
    accel_box= XmVaCreateSimpleCheckBox(dialog,"accelerations",
		       NULL,
		       XmVaCHECKBUTTON, str1, NULL, NULL, NULL,
		       XmVaCHECKBUTTON, str2, NULL, NULL, NULL,
		       XmNtopAttachment, XmATTACH_WIDGET,
		       XmNtopWidget, time_scale_box,
		       XmNleftAttachment, XmATTACH_FORM,
		       XmNrightAttachment, XmATTACH_FORM,
		       NULL);
    XmStringFree(str1);
    XmStringFree(str2);
    XtManageChild(accel_box);

    /* Separator */
    n= 0;
    XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
    XtSetArg(args[n], XmNtopWidget, accel_box); n++;
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
    separator= XmCreateSeparator(dialog,"separator",args,n);
    XtManageChild(separator);
    
    /* Action area */
    n= 0;
    XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
    XtSetArg(args[n], XmNtopWidget, separator); n++;
    XtSetArg(args[n], XmNfractionBase, 7); n++;
    action_area= XmCreateForm(dialog,"action_area",args,n);
    
    /* OK button */
    title_string= XmStringCreate("OK", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'O'); n++;
    XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
    XtSetArg(args[n], XmNleftPosition, 0); n++;
    XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
    XtSetArg(args[n], XmNrightPosition, 1); n++;
    ok_button= XmCreatePushButton(action_area,"Close",args,n);
    XtAddCallback(ok_button, XmNactivateCallback, 
		  ok_callback, dialog);
    if (title_string) XmStringFree(title_string);
    XtManageChild(ok_button);
    
    /* Reset button */
    title_string= XmStringCreate("Reset", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'R'); n++;
    XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
    XtSetArg(args[n], XmNleftPosition, 2); n++;
    XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
    XtSetArg(args[n], XmNrightPosition, 3); n++;
    reset_button= XmCreatePushButton(action_area,"Update",args,n);
    XtAddCallback(reset_button, XmNactivateCallback, 
		  reset_callback, dialog);
    if (title_string) XmStringFree(title_string);
    XtManageChild(reset_button);
    
    /* Cancel button */
    title_string= XmStringCreate("Cancel", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'C'); n++;
    XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
    XtSetArg(args[n], XmNleftPosition, 4); n++;
    XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
    XtSetArg(args[n], XmNrightPosition, 5); n++;
    cancel_button= XmCreatePushButton(action_area,"Set",args,n);
    XtAddCallback(cancel_button, XmNactivateCallback, 
		  cancel_callback, dialog);
    if (title_string) XmStringFree(title_string);
    XtManageChild(cancel_button);
    
    /* Help button */
    title_string= XmStringCreate("Help", XmSTRING_DEFAULT_CHARSET);
    n = 0;
    XtSetArg(args[n], XmNlabelString, title_string); n++;
    XtSetArg(args[n], XmNmnemonic, 'H'); n++;
    XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
    XtSetArg(args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
    XtSetArg(args[n], XmNleftPosition, 6); n++;
    XtSetArg(args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
    XtSetArg(args[n], XmNrightPosition, 7); n++;
    help_button= XmCreatePushButton(action_area,"Help",args,n);
    XtAddCallback(help_button, XmNactivateCallback, 
		  fly_settings_help_callback, NULL);
    if (title_string) XmStringFree(title_string);
    XtManageChild(help_button);
    
    XtManageChild(action_area);
    
    /* Set default button and manage the dialog */
    XtVaSetValues(dialog, XmNdefaultButton, cancel_button, NULL);
  }

  settings_to_widget(dialog); /* copy everything in */
  XtManageChild(dialog);
}

static void new_path_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
/* This routine clears the flyby */
{
  ger_debug("new_path_callback:");

  fb_new_path();
  path_ready= 0;
  path_generated= 0;
}

static void add_keyframe_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
/* This routine clears the flyby */
{
  ger_debug("add_keyframe_callback:");

  fb_add_keyframe();
  path_ready= 1;
  path_generated= 0;
}

static void preview_whole_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
/* This routine clears the flyby */
{
  ger_debug("preview_whole_callback:");

  if (!path_ready) 
    mil_motif_alert_box(w,"No keyframes are defined yet!");
  else {
    if (!path_generated) {
      fb_generate_path(option_flags,flyby_frames);
      path_generated= 1;
    }
    fb_preview(0);
  }
}

static void preview_keyframes_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
/* This routine clears the flyby */
{
  ger_debug("preview_keyframes_callback:");

  if (!path_ready) 
    mil_motif_alert_box(w,"No keyframes are defined yet!");
  else {
    fb_preview(1);
  }
}

static int flyby_filename_ok(fname)
char *fname;
/* This routine checks for a filename which is valid for writing. */
{
  char *runner, *last;

  ger_debug("flyby_filename_ok:");

  runner= fname;
  while (*runner) last= runner++;
  if ( *last == '/' ) return(0);

  return(1);
}

static void flyby_save_overwrite_callback(w, data, call_data)
Widget w;
Warning_data *data;
caddr_t call_data;
/* Called when 'OK' is pressed in the confirm warning of the file box */
{
  ger_debug("flyby_save_overwrite_callback:");
  
  /* Pop down the file selection dialog */
  XtUnmanageChild( data->to_be_closed );

  fb_save_flyby(data->fname);
  (void)free(data->fname);
  (void)free(data);
}

static void flyby_save_cancel_callback(w, data, call_data)
Widget w;
Warning_data *data;
caddr_t call_data;
/* Called when 'cancel' is pressed in the confirm warning of the file box */
{
  ger_debug("flyby_save_cancel_callback:");
  
  (void)free(data->fname);
  (void)free(data);
}

static void flyby_save_ok_callback(file_box, client_data, call_data)
Widget file_box;
caddr_t client_data;
XmFileSelectionBoxCallbackStruct *call_data;
/* Called when 'OK' is selected in the file box. */
{
  Widget shell = XtParent(file_box);
  char *fname;
  char buf[256];
  Warning_data *data;

  ger_debug("flyby_save_ok_callback:");

  if ( !XmStringGetLtoR(call_data->value, XmSTRING_DEFAULT_CHARSET,
			&fname) ) return;
  if ( flyby_filename_ok(fname) ) {
    if (access(fname,F_OK)) { /* file doesn't exist */
      XtUnmanageChild(file_box);
      fb_save_flyby(fname);
      (void)free(fname);
    }
    else {
      if ( !(data= (Warning_data *)malloc(sizeof(Warning_data))) )
	ger_fatal("flyby_save_ok_callback: unable to allocate %d bytes!",
		  sizeof(Warning_data));
      data->fname= fname;
      data->to_be_closed= file_box;
      sprintf(buf,"WARNING:\n overwrite the file %s?",fname);
      mil_motif_warning_box(file_box, buf,
			    flyby_save_overwrite_callback,
			    flyby_save_cancel_callback,
			    data);
    }
  }
  else {
    mil_motif_alert_box(file_box,
			"The file selected is invalid or is not writable!");
    (void)free(fname);
  }
}

static void save_flyby_callback(w, client_data, call_data)
Widget w;
caddr_t client_data;
caddr_t call_data;
/* This routine clears the flyby */
{
  static Widget file_box= NULL;

  ger_debug("save_flyby_callback:");

  if (!path_ready) 
    mil_motif_alert_box(w,"No keyframes are defined yet!");
  else {
    if (!path_generated) {
      fb_generate_path(option_flags,flyby_frames);
      path_generated= 1;
    }
    if (!file_box) file_box= mil_motif_file_box(w,"SaveFlyby","*_flyby.p3d",
						flyby_save_ok_callback);
    XtManageChild(file_box);
  }
}

void mil_motif_flyby_setup( menu_bar )
Widget menu_bar;
/* This routine creates the Flyby pulldown menu. */
{
  Widget button;
  Widget menu_pane;
  Widget submenu_pane;
  Widget submenu_button;
  XmString title_string;

  Arg    args[MAX_ARGS];
  int    n;

  ger_debug("mil_motif_flyby_setup:");

  /* Initialize the flyby package and status flags */
  fb_new_path();
  path_ready= 0;
  path_generated= 0;

  /* Create new menu pane */
  n = 0;
  menu_pane = XmCreatePulldownMenu(menu_bar, "menu_pane", args, n);
  
  /* Create Flyby Menu */
  title_string= XmStringCreate("FlyBy", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNsubMenuId, menu_pane); n++;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'B'); n++;
  button = XmCreateCascadeButton(menu_bar, "Flyby", args, n);
  if (title_string) XmStringFree(title_string);
  XtManageChild(button);
  
  /* Create Settings button */
  title_string= XmStringCreate("Settings", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'S'); n++;
  button = XmCreatePushButton(menu_pane, "settings", args, n);
  XtAddCallback(button, XmNactivateCallback, settings_callback, NULL);
  if (title_string) XmStringFree(title_string);
  XtManageChild(button);

  /* Create New Path button */
  title_string= XmStringCreate("New Path", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'N'); n++;
  button = XmCreatePushButton(menu_pane, "new path", args, n);
  XtAddCallback(button, XmNactivateCallback, new_path_callback, NULL);
  if (title_string) XmStringFree(title_string);
  XtManageChild(button);
  
  /* Create Add Keyframe button */
  title_string= XmStringCreate("Add Keyframe", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'A'); n++;
  button = XmCreatePushButton(menu_pane, "add keyframe", args, n);
  XtAddCallback(button, XmNactivateCallback, add_keyframe_callback, NULL);
  if (title_string) XmStringFree(title_string);
  XtManageChild(button);

  /* Create Preview submenu and items */
  n = 0;
  submenu_pane = XmCreatePulldownMenu(menu_pane, "menu_pane", args, n);
  title_string= XmStringCreate("Preview", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNsubMenuId, submenu_pane); n++;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'P'); n++;
  button = XmCreateCascadeButton(menu_pane, "Flyby", args, n);
  if (title_string) XmStringFree(title_string);

  title_string= XmStringCreate("Whole Flyby", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'W'); n++;
  submenu_button = XmCreatePushButton(submenu_pane, "whole_flyby", args, n);
  XtAddCallback(submenu_button, XmNactivateCallback, 
		preview_whole_callback, NULL);
  if (title_string) XmStringFree(title_string);
  XtManageChild(submenu_button);

  title_string= XmStringCreate("Keyframes Only", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'K'); n++;
  submenu_button = XmCreatePushButton(submenu_pane, "keyframes_only", args, n);
  XtAddCallback(submenu_button, XmNactivateCallback, 
		preview_keyframes_callback, NULL);
  if (title_string) XmStringFree(title_string);
  XtManageChild(submenu_button);

  XtManageChild(button); /* Preview submenu CascadeButton */

  /* Create Save Flyby button */
  title_string= XmStringCreate("Save Flyby", XmSTRING_DEFAULT_CHARSET);
  n = 0;
  XtSetArg(args[n], XmNlabelString, title_string); n++;
  XtSetArg(args[n], XmNmnemonic, 'S'); n++;
  button = XmCreatePushButton(menu_pane, "save flyby", args, n);
  XtAddCallback(button, XmNactivateCallback, save_flyby_callback, NULL);
  if (title_string) XmStringFree(title_string);
  XtManageChild(button);
}
