/****************************************************************************
 * make_hershey.c
 * Author Joel Welling and Phil Andrews
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

/* 
This routine translates NCAR Hershey font information into more
useable form, and outputs a header file with the desired fonts in it.
The header file produced is called hershey.h, as defined by out_file below.
*/
#include <stdio.h>
#include "hfonts.h"

/* Fonts to include, and how many there are: modify to add fonts */
#define NO_FONTS 4
static char *fonts_wanted[NO_FONTS] = {
"HERSHEY:SIMPLEX_ROMAN",
"HERSHEY:TRIPLEX_ROMAN",
"HERSHEY:SIMPLEX_SCRIPT",
"HERSHEY:SIMPLEX_GREEK"};

/* 
The following value rescales the fonts so the characters lie in roughly
a unit rectangle
*/
#define FONT_SCALE 1.0/24.0

/* The following sets the allowed length of font names */
#define NAME_LENGTH 64

/*
The following gets subtracted from the (unsigned char) vertex data
to produce the actual vertex coordinates.
*/
#define COORD_BIAS 128

/* Sizes of expected input data */
#define INDEX_SIZE 8000
#define DATA_SIZE 128000

/* Structure arrays to hold font, character, stroke, and vertex info */
static struct Font { char name[NAME_LENGTH]; int first_char, num_chars; }
        font_table[NO_FONTS];
static struct Fontchar { int first_stroke, stroke_count, xcenter, xshift; }
        char_table[ NO_FONTS * MAX_HCHARS ];
static struct Stroke { int start, count; } stroke_table[ DATA_SIZE/4 ];
static unsigned char vertex_table[ DATA_SIZE/2 ][2];

/* Other constants */
#define END_FLAG 0
#define UP_FLAG_1 0
#define UP_FLAG_2 128

write_header(outptr)
FILE *outptr;
{
	static char header[] = "\
/* \n\
This file contains Hershey font coordinate information.  It is generated \n\
by make_hershey.c . \n\
*/ \n\
\n\
#define NO_FONTS %d \n\
#define FONT_NM_LENGTH %d \n\
#define FONT_SCALE %f \n\
#define COORD_BIAS %d \n\
\n\
static struct Font { char name[FONT_NM_LENGTH]; int first_char, num_chars; } \n\
        font_table[] = { \n\
"; /* end of header text */

	fprintf(outptr, header, NO_FONTS, NAME_LENGTH, FONT_SCALE, COORD_BIAS );
}

main()
{
	static unsigned char all_array[INDEX_SIZE + DATA_SIZE];
	static int indices_done[INDEX_SIZE][2];
	int bytes_got, j, k, l, m, index_in, new_index, 
	current_font, current_char=0, current_stroke=0, current_vertex=0,
	max_char, max_stroke, max_vertex, stroke_init= 1;
	FILE *inptr, *outptr;
	static char in_file[] = "hfonts.bin";
	static char out_file[] = "hershey.h";

	/* open the file */
	if (NULL == (inptr = fopen(in_file, "r"))) {
	    fprintf(stderr, "couldn't open %s for input !\n", in_file);
	    return(0);
	}

	/* load it all in */
	bytes_got = 0;
	/* do it the slow way */	
	while ((!feof(inptr)) && (bytes_got < (INDEX_SIZE + DATA_SIZE)))
	    all_array[bytes_got++] = getc(inptr);

	fprintf(stderr, "read %d bytes\n", bytes_got);	
	fclose(inptr);

	/* now create a new file and index */
	if (NULL == (outptr = fopen(out_file, "w"))) {
	    fprintf(stderr, "couldn't open %s for output !\n", out_file);
	    return(0);
	}

	/* Build the tables */
	for (current_font=0; current_font<NO_FONTS; current_font++) {

	    fprintf(stderr, "starting %s\n", fonts_wanted[current_font]);
	    strncpy( font_table[current_font].name, 
			fonts_wanted[current_font], NAME_LENGTH);
	    font_table[current_font].first_char= current_char;
	    font_table[current_font].num_chars= MAX_HCHARS;

	    /* Find font index j corresponding to desired font */
	    for (j=0; (j<NO_HFONTS) && 
		strcmp(fonts_wanted[current_font], h_names[j]); ++j);
	    if (j>=NO_HFONTS) {
		fprintf(stderr, "couldn't find %s\n", 
			fonts_wanted[current_font]);
		return(0);
	    } else fprintf(stderr, "using font %d\n", j);

	    /* For each character in the font... */
	    for (k=0; k<MAX_HCHARS; ++k) {
		if (index_in = h_ptrs[j][k]) {	/* character is present */
		    /* have we already done it ? */
		    for (l=0; 
			(l<current_char) && (indices_done[l][0] != index_in);
			++l);
		    if (l == current_char) {	/* new one */
			indices_done[l][0]= index_in;
			indices_done[l][1]= current_char;
			char_table[current_char].first_stroke= current_stroke;
			stroke_table[current_stroke].start= current_vertex;

			/* 
			Compute the location in all_array of the start
			of the character.
			*/
			new_index = (all_array[(index_in - 1) * 2] << 8) +
			    all_array[(index_in - 1) * 2 + 1];
			m = INDEX_SIZE + (new_index - 1) * 2;

			/* First two bytes add to give xshift */
			char_table[current_char].xcenter= 
			    COORD_BIAS - all_array[m++];
			char_table[current_char].xshift= 
			    char_table[current_char].xcenter
			    + all_array[m++] - COORD_BIAS;

			/* Loop over the strokes in the character */
			for ( ; ((all_array[m] != END_FLAG) || 
			    (all_array[m+1] != END_FLAG)); m += 2 ) {

			    if ((all_array[m] == UP_FLAG_1) &&
				(all_array[m+1] == UP_FLAG_2)) { /*stroke end*/
				stroke_table[current_stroke].count=
				    current_vertex -
				    stroke_table[current_stroke].start;
			        current_stroke++;
				stroke_init= 1; /* ready to begin stroke */
			    }
			    else { /* starting or continuing stroke */
				vertex_table[current_vertex][0]= 
				    all_array[m];
				vertex_table[current_vertex][1]= 
				    2*COORD_BIAS - all_array[m+1];
				if (stroke_init) { /* starting stroke */
				    stroke_table[current_stroke].start=
					current_vertex;
				    stroke_init= 0;
				}
				current_vertex++;
			    }
			}
			char_table[current_char].stroke_count= 
			    current_stroke - 
			    char_table[current_char].first_stroke + 1;
			stroke_table[current_stroke].count= current_vertex
			    - stroke_table[current_stroke].start;
			current_stroke++;
		    }
		    else {  /* have already set up this character */
			char_table[current_char].first_stroke= 
				char_table[indices_done[l][1]].first_stroke;
			char_table[current_char].stroke_count= 
				char_table[indices_done[l][1]].stroke_count;
			char_table[current_char].xcenter= 
				char_table[indices_done[l][1]].xcenter;
			char_table[current_char].xshift= 
				char_table[indices_done[l][1]].xshift;
		    };
		} 
		else { /* this char is not available */
			char_table[current_char].first_stroke= -1;
			char_table[current_char].stroke_count= -1;
			char_table[current_char].xcenter= 0;
			char_table[current_char].xshift= 0;
			}
		current_char++;
	    }
	    fprintf(stderr, "finished %s\n", fonts_wanted[current_font]);
	}

	/* Save the maximum character, stroke, and vertex values */
	max_char= current_char;
	max_stroke= current_stroke;
	max_vertex= current_vertex;

	/* Write header */
	write_header(outptr);

	/* Emit the font table */
	for ( current_font= 0; current_font<NO_FONTS-1; current_font++ )
		fprintf(outptr, "	{ \"%s\", %d, %d },\n", 
			font_table[current_font].name,
			font_table[current_font].first_char,
			font_table[current_font].num_chars );
	fprintf(outptr, "	{ \"%s\", %d, %d }\n", 
		font_table[current_font].name,
		font_table[current_font].first_char,
		font_table[current_font].num_chars );
	fprintf(outptr,"	};\n\n");

	/* Emit the character table */
	fprintf(outptr,
"static struct Fontchar { int first_stroke, stroke_count, xcenter, xshift;}\n");
	fprintf(outptr,"	char_table[]= {\n");
	for ( current_char= 0; current_char<max_char-1; current_char++ ) {
		if ( !(current_char % 3) ) fprintf(outptr,"	");
		fprintf(outptr,"{ %d, %d, %d, %d }, ",
			char_table[current_char].first_stroke,
			char_table[current_char].stroke_count,
			char_table[current_char].xcenter,
			char_table[current_char].xshift );
		if ( !((current_char+1) % 3) ) fprintf(outptr,"\n");
	};
	if ( !(current_char % 3) ) fprintf(outptr,"	");
	fprintf(outptr,"{ %d, %d, %d, %d }",
		char_table[current_char].first_stroke,
		char_table[current_char].stroke_count,
		char_table[current_char].xcenter,
		char_table[current_char].xshift );
	fprintf(outptr,"\n");
	fprintf(outptr,"	};\n\n");

	/* Emit the stroke table */
	fprintf(outptr,
	    "static struct Stroke { int start, count; } stroke_table[]= {\n");
	for ( current_stroke= 0; current_stroke<max_stroke-1; current_stroke++)
		{
		if ( !(current_stroke % 6) ) fprintf(outptr,"	");
		fprintf(outptr,"%d, %d, ",
			stroke_table[current_stroke].start,
			stroke_table[current_stroke].count);
		if ( !( (current_stroke + 1) % 6) ) fprintf(outptr,"\n");
		};
	if ( !(current_stroke % 6) ) fprintf(outptr,"	");
	fprintf(outptr,"%d, %d",
		stroke_table[current_stroke].start,
		stroke_table[current_stroke].count);
	fprintf(outptr,"\n");
	fprintf(outptr,"	};\n\n");

	/* Emit the vertex table */
	fprintf(outptr,"static unsigned char vertex_table[][2]= {\n");
	for (current_vertex=0; current_vertex<max_vertex-1; current_vertex++)
		{
		if ( !(current_vertex % 6) ) fprintf(outptr,"	");
		fprintf(outptr,"%d, %d, ",
			vertex_table[current_vertex][0],
			vertex_table[current_vertex][1]);
		if ( !( (current_vertex + 1) % 6) ) fprintf(outptr,"\n");
		};
	if ( !(current_vertex % 6) ) fprintf(outptr,"	");
	fprintf(outptr,"%d, %d",
		vertex_table[current_vertex][0],
		vertex_table[current_vertex][1]);
	fprintf(outptr,"\n");
	fprintf(outptr,"	};\n");

	fclose(outptr);
}
