/****************************************************************************
 * lexer.c
 * Author Chris Nuuja, Frank Wu
 * Copyright 1991, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "alisp.h"
#include "parse.h"

TYPES_OF_TOKEN TOKEN_TYPE;
KEY_TYPE SYMBOL_TYPE;
enum state_enum state;

static short	invalid_keyname, invalid_callout;

unsigned short curr_arg = 0;
int LINE_NUMBER;
char *CURRENT_FILE;
char token[tokensize+1];
char	linebuf[maxline];
char	*tokchar = token;
char	*toklim = token + tokensize;
int	CHAR_NUM=0;
/* This macro adds a character to the current token. */
#define addchar(thischar)                                                 \
{                                                                         \
	if (tokchar >= toklim) {fprintf(stderr,"Token too long:%s\n",token); exit(1);}; \
        *tokchar++= thischar;                                             \
	++CHAR_NUM;						          \
}

/* 
The following macros (those named prs_something) handle particular
actions of the parsing finite state machine.  The line about ;'s
serves to discard comments.
*/
#define	prs_begin(thischar) \
{ \
	switch (thischar) \
		{ \
		case EOF: \
			state= st_done; \
			TOKEN_TYPE= EOF_TK; \
			addchar('\0'); \
			break; \
                case (int)'\0': \
                        state = st_done; \
			fprintf(stderr,"Warning:FOUND a \0\n"); \
			addchar('\0'); \
                        break; \
		case (int)';': \
			fgets(linebuf,maxline-1,Infile(f_stack)); \
			LINE_NUMBER++;   \
			break; \
		case (int)'\n':   \
			LINE_NUMBER++;   \
			break;   \
		case (int)' ': case (int) '\t':     \
			break;    \
		case (int) '\r': \
			LINE_NUMBER++;   \
			break;   \
		case (int)':': \
			state= st_keyname; \
			invalid_keyname = 1; \
			addchar(thischar); \
			break; \
		case (int)'\'': \
			state =	st_done; \
			TOKEN_TYPE = QUOTEMACRO_TK; \
			addchar(thischar); \
			addchar('\0');  \
			break; \
		case (int)'~': \
			state= st_callout; \
			addchar(thischar); \
			invalid_callout	= 1; \
			break; \
		case (int)'"': \
			state= st_string; \
			break; \
		case (int)'(': \
			state= st_done; \
			addchar('(');  \
			addchar('\0'); \
			TOKEN_TYPE	= LEFT_PAREN_TK; \
			break; \
		case (int)')': \
			state= st_done; \
			addchar(')');  \
			addchar('\0'); \
			TOKEN_TYPE	= RIGHT_PAREN_TK; \
			break; \
		default: \
			if (strchr("+-",thischar)) \
				{ \
				state= st_sign; \
				addchar(thischar); \
				} \
			else if (strchr("0123456789.",thischar)) \
				{ \
				state= st_iorf; \
			     	if (thischar=='.') state= st_float; \
				addchar(thischar); \
				} \
			else    { \
				state= st_name; \
				addchar(thischar); \
			       	} \
 		} \
}

#define prs_sign(thischar) \
{ \
	if (strchr("0123456789",thischar)) \
		{ \
		state= st_iorf; \
		addchar(thischar); \
		} \
	else if (thischar=='.') \
		{ \
		state= st_float; \
		addchar(thischar); \
		} \
	else if	(white_space(thischar)) \
		{ \
		if (newline(thischar))  LINE_NUMBER++;  \
		state= st_done; \
		addchar('\0'); \
		TOKEN_TYPE= SYMBOL_TK; \
		if (!strcmp(token,"+"))  \
		  SYMBOL_TYPE = K_ADD;  \
		else   \
		  SYMBOL_TYPE = K_SUB;  \
		} \
        else if (thischar=='(') \
                { \
                state= st_done; \
                addchar('\0'); \
		TOKEN_TYPE= SYMBOL_TK; \
		if (!strcmp(token,"+"))  \
		  SYMBOL_TYPE = K_ADD;  \
		else   \
		  SYMBOL_TYPE = K_SUB;  \
		ungetc('(',Infile(f_stack)); \
		} \
	else if (thischar==')') \
		{ \
		state= st_done; \
		addchar('\0'); \
		TOKEN_TYPE= SYMBOL_TK; \
		if (!strcmp(token,"+"))  \
		  SYMBOL_TYPE = K_ADD;  \
		else   \
		  SYMBOL_TYPE = K_SUB;  \
		ungetc(')',Infile(f_stack)); \
		} \
	else    { \
		state = st_name; \
		addchar(thischar); \
		} \
}

#define prs_iorf(thischar) \
{ \
	if (strchr("0123456789",thischar)) \
		{ \
		addchar(thischar); \
		} \
	else if (thischar=='.') \
		{ \
		state= st_float; \
		addchar(thischar); \
		} \
	else if (strchr("eE",thischar)) \
		{ \
		state= st_exponent; \
		addchar(thischar); \
		} \
	else if (white_space(thischar)) \
		{ \
		if (newline(thischar))  LINE_NUMBER++;  \
		state= st_done; \
		addchar('\0'); \
		TOKEN_TYPE= INT_TK; \
		} \
        else if (thischar=='(') \
                { \
                state= st_done; \
                addchar('\0'); \
		TOKEN_TYPE= INT_TK; \
                ungetc('(',Infile(f_stack)); \
                } \
	else if (thischar==')') \
		{ \
		state= st_done; \
		addchar('\0'); \
		TOKEN_TYPE= INT_TK; \
		ungetc(')',Infile(f_stack)); \
		} \
	else	{ \
		state =	st_name; \
		addchar(thischar); \
		} \
}

#define prs_float(thischar)                                               \
{                                                                         \
	if (strchr("0123456789",thischar))                                 \
		{                                                         \
		addchar(thischar);                                        \
		}                                                         \
	else if (strchr("eE",thischar))                                    \
		{                                                         \
		state= st_exponent;                                       \
		addchar(thischar);                                        \
		}                                                         \
	else if	(white_space(thischar))					  \
		{                                                         \
		if (newline(thischar))  LINE_NUMBER++;  \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= FLOAT_TK;                                        \
		}                                                         \
        else if (thischar=='(')                                           \
                {                                                         \
                state= st_done;                                           \
                addchar('\0');                                            \
		TOKEN_TYPE= FLOAT_TK;					  \
                ungetc('(',Infile(f_stack));          \
                }                                                         \
	else if (thischar==')')                                           \
		{                                                         \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= FLOAT_TK;                                        \
		ungetc(')',Infile(f_stack));          \
		}                                                         \
	else	{							  \
		state =	st_name;					  \
		addchar(thischar);					  \
		}							  \
}

#define prs_exponent(thischar)                                            \
{                                                                         \
	if (strchr("0123456789+-",thischar))                               \
		{                                                         \
		state= st_exp2;                                           \
		addchar(thischar);                                        \
		}                                                         \
	else if	(white_space(thischar))					  \
		{                                                         \
		if (newline(thischar))  LINE_NUMBER++;  \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= FLOAT_TK;                                        \
		}                                                         \
        else if (thischar=='(')                                           \
                {                                                         \
                state= st_done;                                           \
                addchar('\0');                                            \
		TOKEN_TYPE= FLOAT_TK;					  \
                ungetc('(',Infile(f_stack));          \
                }                                                         \
	else if (thischar==')')                                           \
		{                                                         \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= FLOAT_TK;                                        \
		ungetc(')',Infile(f_stack));          \
		}                                                         \
	else	{							  \
		state =	st_name;					  \
		addchar(thischar);					  \
		}							  \
}

#define prs_exp2(thischar)                                                \
/* This	is like	prs_exponent, but no longer accepts a sign */			  \
{                                                                         \
	if (strchr("0123456789",thischar))                                 \
		{                                                         \
		addchar(thischar);                                        \
		}                                                         \
	else if	(white_space(thischar))					  \
		{                                                         \
		if (newline(thischar))  LINE_NUMBER++;  \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= FLOAT_TK;                                        \
		}                                                         \
        else if (thischar=='(')                                           \
                {                                                         \
                state= st_done;                                           \
                addchar('\0');                                            \
		TOKEN_TYPE= FLOAT_TK;					  \
                ungetc('(',Infile(f_stack));          \
                }                                                         \
	else if (thischar==')')                                           \
		{                                                         \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= FLOAT_TK;                                        \
		ungetc(')',Infile(f_stack));          \
		}                                                         \
	else	{							  \
		state =	st_name;					  \
		addchar(thischar);					  \
		}							  \
}

#define prs_string(thischar)                                              \
{                                                                         \
	if ((thischar=='"') || (thischar==EOF))                           \
		{                                                         \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= STRING_TK;                                       \
		}                                                         \
	else if (thischar=='\\')                                          \
		{                                                         \
		state= st_literal;                                        \
		}                                                         \
	else    {                                                         \
                addchar(thischar);                                        \
		}                                                         \
}

#define prs_literal(thischar)                                             \
{                                                                         \
	state= st_string;                                                 \
	if (thischar=='n')                                                \
		{                                                         \
		addchar('\n');                                            \
		}                                                         \
	else if (thischar==EOF)                                           \
		{                                                         \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= SYMBOL_TK;                                         \
		SYMBOL_TYPE = getkeytype(check_key());		          \
		}                                                         \
	else    {                                                         \
		addchar(thischar);                                        \
		}                                                         \
}
#define	prs_keyname(thischar)						  \
{                                                                         \
	if (white_space(thischar))					  \
 		{                                                         \
		if (newline(thischar))  LINE_NUMBER++;  \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= SYMBOL_TK;					  \
		SYMBOL_TYPE = K_SYMBOLKEY;					  \
		if(invalid_keyname)					  \
		   {							  \
		      SYMBOL_TYPE = getkeytype(check_key());		  \
		      invalid_keyname = 0;				  \
		   }							  \
		}                                                         \
        else if (thischar=='(')                                           \
                {                                                         \
                state= st_done;                                           \
                addchar('\0');                                            \
		TOKEN_TYPE= SYMBOL_TK;					  \
		SYMBOL_TYPE = K_SYMBOLKEY;					  \
                ungetc('(',Infile(f_stack));          \
		}							  \
	else if (thischar==')')                                           \
		{							  \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= SYMBOL_TK;					  \
		SYMBOL_TYPE = K_SYMBOLKEY;					  \
		ungetc(')',Infile(f_stack));          \
		}                                                         \
	else	{							  \
		invalid_keyname= 0;					  \
		addchar(thischar);                                        \
		}                                                         \
}
#define	prs_callout(thischar)						  \
{                                                                         \
	if (white_space(thischar))					  \
 		{                                                         \
		if (newline(thischar))  LINE_NUMBER++;  \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= SYMBOL_TK;					  \
		SYMBOL_TYPE = K_CALLOUT;				  \
		invalid_callout	= 0;					  \
		}							  \
        else if (thischar=='(')                                           \
                {                                                         \
                state= st_done;                                           \
                addchar('\0');                                            \
		TOKEN_TYPE= SYMBOL_TK;					  \
		SYMBOL_TYPE = K_CALLOUT;				  \
		if(invalid_callout)					  \
		   {							  \
		      SYMBOL_TYPE = getkeytype(check_key());		  \
		      invalid_callout = 0;				  \
		   }							  \
                ungetc('(',Infile(f_stack));          \
		}							  \
	else if (thischar==')')                                           \
		{                                                         \
		state= st_done;                                           \
		TOKEN_TYPE= SYMBOL_TK;					  \
		SYMBOL_TYPE = K_CALLOUT;				  \
		addchar('\0');                                            \
		if(invalid_callout)					  \
		   {							  \
		      SYMBOL_TYPE = getkeytype(check_key());	  \
		      invalid_callout = 0;				  \
		   }							  \
		ungetc(')',Infile(f_stack));          \
		}                                                         \
	else    {                                                         \
		invalid_callout= 0;					  \
		addchar(thischar);                                        \
		}                                                         \
}
#define prs_name(thischar)                                                \
{                                                                         \
	if (white_space(thischar))					  \
 		{                                                         \
		if (newline(thischar))  LINE_NUMBER++;  \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= SYMBOL_TK;                                         \
		SYMBOL_TYPE = getkeytype(check_key());			  \
		}                                                         \
        else if (thischar=='(')                                           \
                {                                                         \
                state= st_done;                                           \
                addchar('\0');                                            \
		TOKEN_TYPE= SYMBOL_TK;					  \
		SYMBOL_TYPE = getkeytype(check_key());			  \
                ungetc('(',Infile(f_stack));          \
		}							  \
	else if (thischar==')')                                           \
		{                                                         \
		state= st_done;                                           \
		addchar('\0');                                            \
		TOKEN_TYPE= SYMBOL_TK;                                         \
		SYMBOL_TYPE = getkeytype(check_key());			  \
		ungetc(')',Infile(f_stack));          \
		}                                                         \
	else if ( (thischar==':') && (!strncmp(token,"system",6)) )	  \
		{							  \
		state= st_system;					  \
		tokchar = token;					  \
		}							  \
	else    {                                                         \
		addchar(thischar);                                        \
		}                                                         \
}
#define prs_system(thischar)						  \
{									  \
	if (white_space(thischar))					  \
		{							  \
		if (newline(thischar))  LINE_NUMBER++;  \
		state= st_done;					  	  \
		addchar('\0');						  \
		TOKEN_TYPE= SYMBOL_TK;					  \
		SYMBOL_TYPE = K_SYSTEM;					  \
		}							  \
	if ( (thischar=='(') || (thischar==')') ) 			  \
		{							  \
		state= st_done;						  \
		addchar('\0');						  \
		TOKEN_TYPE= SYMBOL_TK;					  \
		SYMBOL_TYPE = K_SYSTEM;					  \
		ungetc(thischar,Infile(f_stack));	  \
		}                                                         \
	else								  \
		{						 	  \
		addchar(thischar);					  \
		}							  \
}

/*FUNCTION -- get_char()
 *get_char() will return a char from either
 *an global input file "infile", or a global string called
 *"defstruct_list" based on a flag "defstruct_flag"*/

/*WHEN USED?
 *get_char() is called in a while loop in get_token*/

/*WHAT's RETURNED?
 *get_char() will returned a single character in the form of a short.*/

static short get_char()
{    
    return( getc(Infile(f_stack)) );
}

/*FUNCTION -- get_token()
 *get_token() will assign the global char string "token"
 *to a string, and global enum type to the specific type of
 *the token*/

/*WHEN USED?
 *get_token() is called in the parser every time it wants a new token */

/*WHAT's RETURNED?
 *a 1 is returned when an EOF is encountered, otherwise
 *a zero is returned*/

void get_token()
{
    short thischar;

    
    CHAR_NUM=0;
    tokchar = token;		/*   resets the fillptr <tokchar> to the 
				     beginning of the buffer <token>        */
    state= st_begin;

    while (state != st_done)
    {
	thischar = get_char(); /*a special char returning function*/
	if ( isupper(thischar) && (state != st_string) )
		thischar= tolower(thischar);
	
	switch ((int)state){
	    case (int)st_begin:	    prs_begin(thischar); break;
	    case (int)st_sign:	    prs_sign(thischar); break;
	    case (int)st_iorf:	    prs_iorf(thischar); break;
	    case (int)st_float:	    prs_float(thischar); break;
	    case (int)st_exponent:  prs_exponent(thischar); break;
	    case (int)st_exp2:	    prs_exp2(thischar); break;
	    case (int)st_string:    prs_string(thischar); break;
	    case (int)st_literal:   prs_literal(thischar); break;
	    case (int)st_name:	    prs_name(thischar); break;
	    case (int)st_keyname:   prs_keyname(thischar); break;
	    case (int)st_callout:   prs_callout(thischar); break;
	    case (int)st_system:    prs_system(thischar); break;
	}
    }
}

void push_filestack(filename)
char *filename;
{
   G_FILES *stack_ptr;

   /* save current file name and line number */
   FileName(f_stack) = CURRENT_FILE;
   FileLine(f_stack) = LINE_NUMBER;

   /*  put new file on top of stack */
   stack_ptr = Alloc(G_FILES, 1);
   stack_ptr->nxt = f_stack;
   f_stack->print_flag = PRINT_FLAG;
   PRINT_FLAG=0;
   stack_ptr->print_flag = PRINT_FLAG;
   f_stack = stack_ptr;

   /* open file  and set CURRENT_FILE and LINE_NUMBER */
   if((Infile(f_stack) = fopen(filename, "r"))==NULL)
	{
   	fprintf(stderr,"Error:could not open :%s\n",filename);
    	exit(1);
	}
   CURRENT_FILE = (char *) malloc(strlen(filename)+1*sizeof(char));
   strcpy(CURRENT_FILE,filename);
   LINE_NUMBER = 1;
   fprintf(stderr,"Loading %s\n\n",filename);
}

void pop_filestack()
{
      if(Infile(f_stack) == stdin)
	{
	    fprintf(stderr,"ERROR: implementation error with file stack\n");
	    exit(1);
	}else{
	    if(f_stack->nxt)	
		{
	        if ((Infile(f_stack)!=NULL) && (Infile(f_stack)!=stdin)
		     && (fclose(Infile(f_stack))==EOF))
		  {
		  fprintf(stderr,"Error closing %s\n",CURRENT_FILE);
		  exit(1);
		  }
		Nxt(f_stack);
		CURRENT_FILE = FileName(f_stack);
		LINE_NUMBER = FileLine(f_stack);
		PRINT_FLAG = f_stack->print_flag;
		if (Infile(f_stack) == stdin)
			{
			fprintf(stderr,"\n");
			fprintf(stderr,"\n:>");
			}
		}
	    else 
		SESSION_END=1;
	}
}
