/****************************************************************************
 * dump_ren.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This 'renderer' dumps information about structures as they are defined.
*/
#include <stdio.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "ren.h"
#include "indent.h"

/* Needed external definitions */
extern char *malloc(), *realloc();

/* Symbols defined by ren_setup and used in parsing attribute lists */
static Symbol color_symbol, backcull_symbol, text_height_symbol,
              material_symbol;

static void dump_point( point )
Point point;
/* This routine dumps a point */
{
	ger_debug("dump_point: dumping point");
	ind_write("Point: (x, y, z)= (%f %f %f)",
		point_x( point ),
		point_y( point ),
		point_z( point ) );
	ind_eol();
}

static void dump_vector( vector )
Vector vector;
/* This routine dumps a vector */
{
	ger_debug("dump_vector: dumping vector");
	ind_write("Vector: (x, y, z)= (%f %f %f)",
		vector_x( vector ),
		vector_y( vector ),
		vector_z( vector ) );
	ind_eol();
}

static void dump_color( color )
Color color;
/* This routine dumps a color */
{
	ger_debug("dump_color: dumping color");
	ind_write("Color: (r, g, b, a)= ( %f %f %f %f )",
		color_red( color ),
		color_green( color ),
		color_blue( color ),
		color_alpha( color ) );
	ind_eol();
}

static void dump_material( material )
Material material;
/* This routine dumps a material */
{
	ger_debug("dump_material: dumping material");
	ind_write("Material: ka= %f, kd= %f, ks= %f, exp= %f,",
		  material_ka( material ),
		  material_kd( material ),
		  material_ks( material ),
		  material_exp( material ));
	ind_eol();
	ind_write("          reflect= %f, refract= %f,",
		  material_reflect( material ),
		  material_refract( material ));
	ind_eol();
	ind_write("          energy RGBA= (%f, %f, %f, %f)",
		  color_red( material_energy(material) ),
		  color_green( material_energy(material) ),
		  color_blue( material_energy(material) ),
		  color_alpha( material_energy(material) ));
	ind_eol();
}

char *ren_def_material(material)
Material material;
/* This routine fills the slot for a material definition routine */
{
  ger_debug("ren_def_material: doing nothing");
  return((char *)0);
}

void ren_free_material(thismat)
Material thismat;
/* This routine fills the slot for a material deletion routine */
{
  ger_debug("ren_free_material: doing nothing");
}

/* 
This routine should dump the renderer's internal definition
of an object. 
*/
void ren_dump(thisgob)
Gob thisgob;
{
	int thisid;

	thisid= gob_idnum(thisgob);
	ger_debug("ren_dump: dump requested for gob number %d",thisid);
	ind_write("Dump requested for gob number %d",thisid);
	ind_eol();
}

static void dump_vertex( vertex )
Vertex vertex;
/* 
This routine dumps vertices.
*/
{
 	ind_write("Vertex: (x, y, z)= ( %f %f %f )",
		vertex_x(vertex),
		vertex_y(vertex),
		vertex_z(vertex));
	ind_eol();
	ind_push();
	if ( !null( vertex_normal(vertex) ) ) 
		dump_vector( vertex_normal(vertex) );
	if ( !null( vertex_color(vertex) ) )
		dump_color( vertex_color(vertex) );
	ind_pop();
}

void ren_sphere()
/*
This routine dumps a sphere primitive object.
*/
{
	ger_debug("ren_sphere");
	ind_write("Sphere primitive");
	ind_eol();
}

void ren_cylinder()
/*
This routine dumps a cylinder primitive object.
*/
{
	ger_debug("ren_cylinder");
	ind_write("Cylinder primitive");
	ind_eol();
}

void ren_torus(bigradius, smallradius)
float bigradius, smallradius;
/*
This routine dumps a torus primitive object.
*/
{
	ger_debug("ren_torus");
	ind_write("Torus primitive: major and minor radii %f and %f",
		bigradius, smallradius);
	ind_eol();
}

void ren_text(txtpoint,uvec,vvec,txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
/*
This routine dumps a text primitive object.
*/
{
	ger_debug("ren_text");
	ind_write("Text primitive object:");
	ind_eol();
	ind_push();
	ind_write("location: "); dump_point(txtpoint);
	ind_write("u: "); dump_vector(uvec);
	ind_write("v: "); dump_vector(vvec);
	ind_write("text: <%s>",txtstring); ind_eol();
	ind_pop();
}

void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polyline primitive object.
*/
{
	ger_debug("ren_polyline");
	ind_write("Polyline primitive object: %d vertices", count);
	ind_eol();
	ind_push();
	if (count<2) {
		ind_write("***Error***: not enough vertices"); 
		ind_eol();
		}
	while (!null(vlist)) {
		dump_vertex( first_vertex( vlist ) );
		vlist= rest_of_vertices( vlist );
		};
	ind_pop();
}

void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polymarker primitive object.
*/
{
	ger_debug("ren_polymarker");
	ind_write("Polymarker primitive object: %d vertices", count);
	ind_eol();
	ind_push();
	if (count<1) {
		ind_write("***Error***: not enough vertices"); 
		ind_eol();
		}
	while (!null(vlist)) {
		dump_vertex( first_vertex( vlist ) );
		vlist= rest_of_vertices( vlist );
		};
	ind_pop();
}

void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polygon primitive object.
*/
{
	ger_debug("ren_polygon");
	ind_write("Polygon primitive object: %d vertices", count);
	ind_eol();
	ind_push();
	if (count<3) {
		ind_write("***Error***: not enough vertices"); 
		ind_eol();
		}
	while (!null(vlist)) {
		dump_vertex( first_vertex( vlist ) );
		vlist= rest_of_vertices( vlist );
		};
	ind_pop();
}
 
void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a triangle strip primitive object.
*/
{
 	ger_debug("ren_triangle");
	ind_write("Triangle strip primitive object: %d vertices", count);
	ind_eol();
	ind_push();
	if (count<3) {
		ind_write("***Error***: not enough vertices"); 
		ind_eol();
		}
	while (!null(vlist)) {
		dump_vertex( first_vertex( vlist ) );
		vlist= rest_of_vertices( vlist );
		};
	ind_pop();
	
}

void ren_mesh( vlist, vcount, flist, fcount )
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/*
This routine dumps a general mesh object from the args passed it.
*/
{
	Vertex_list thisfacet;
	int facet_vertices;

	ger_debug("ren_mesh");
	ind_write("Mesh primitive object: %d vertices, %d facets",
		vcount, fcount);
	ind_eol();
	ind_push();
	while (fcount--) {
		thisfacet= first_facet( flist );
		flist= rest_of_facets( flist );
		facet_vertices= length_list( thisfacet );
		ren_polygon( thisfacet, facet_vertices );
		};
	ind_pop();
}

void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a Bezier patch primitive object.
*/
{
 	ger_debug("ren_bezier");
	ind_write("Bezier patch primitive object: %d vertices", count);
	ind_eol();
	ind_push();
	if (count!=16) {
		ind_write("***Error***: should be 16 vertices."); 
		ind_eol();
		}
	while (!null(vlist)) {
		dump_vertex( first_vertex( vlist ) );
		vlist= rest_of_vertices( vlist );
		};
	ind_pop();
	
}

static void dump_trans( trans )
Transformation trans;
/* This routine dumps a transformation matrix */
{
	float *matrix;
	register float *m_copy;

	ger_debug("dump_trans: dumping transformation matrix");
	ind_write("Transformation matrix:");
	ind_eol();
	ind_push();

	matrix= array2d_to_c( trans );

	for (m_copy= matrix; m_copy<matrix+16; m_copy= m_copy+4) {
		ind_write( "( %f %f %f %f )", *m_copy, *(m_copy+1),
			  *(m_copy+2), *(m_copy+3) );
		ind_eol();
		};

	ind_pop();
	free( (char *)matrix );
}
 
static void dump_attr( attrlist )
Attribute_list attrlist;
/* This routine dumps an attribute list */
{
	Pair thispair;
 
	ger_debug("dump_attr:");
 
	ind_write("Attribute list:");
	ind_eol();

	ind_push();

	/* check for 'color attribute */
 	if ( !null( thispair= symbol_attr( color_symbol, attrlist ) ) )
		dump_color( pair_color( thispair ) );
	/* check for 'backcull attribute */
 	if ( !null( thispair= symbol_attr( backcull_symbol, attrlist ) ) ) {
		if ( pair_boolean( thispair ) ) {
			ind_write("Backcull: on"); ind_eol();
			}
		else {
			ind_write("Backcull: off"); ind_eol();
			}
		};
	/* check for 'text-height attribute */
 	if ( !null( thispair= symbol_attr( text_height_symbol, attrlist ) ) ) {
		ind_write("Text-height: %f", pair_float( thispair ) );
		ind_eol();
		};
	/* check for 'material attribute */
	if ( !null( thispair= symbol_attr( material_symbol, attrlist ) ) )
	        dump_material( pair_material( thispair ) );

	ind_pop();
}
 
static void dump_kids( kidlist )
Child_list kidlist;
/*  
This routine dumps the ID numbers of the gobs in the 'children' slot 
of the given gob.
*/
{
	int thisid;
	Gob thiskid;
 
	ger_debug("dump_kids: adding children");

	ind_write("Children:"); ind_eol();
	ind_push();
 	while ( !null(kidlist) )
		{
		thiskid= first_child( kidlist );
		kidlist= rest_of_children( kidlist );
	    	thisid= gob_idnum( thiskid );
		ind_write("gob %d", thisid);
		ind_eol();
		};
	ind_pop();
}

void ren_gob(current_gob, trans, attr, primitive, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
/*
This routine sees all gobs as they are defined.  It dumps information
about their internal structure.
*/
{
	ger_debug("ren_gob: Defining gob %d.", current_gob);
 
	ind_write("Gob %d:", current_gob); ind_eol();
	ind_push();
	if ( !null(trans) ) dump_trans(trans);
	if ( !null(attr) ) dump_attr(attr);
	if ( !null(primitive) ) eval_function( primitive_op(primitive) );
	else dump_kids( children );
	ind_pop();
}
 
void ren_light( location, lightcolor )
Point location;
Color lightcolor;
/*                                             
This routine dumps a light primitive object.
*/
{
	ger_debug("ren_light");

	ind_write("Light:"); ind_eol();
	ind_push();
	dump_point( location );
	dump_color( lightcolor );
	ind_pop();
}

void ren_ambient(lightcolor)
Color lightcolor;
/*                                             
This routine dumps an ambient light primitive object.
*/
{
	ger_debug("ren_ambient");

	ind_write("Ambient light:"); ind_eol();
	ind_push();
	if ( !null(lightcolor) )
		dump_color( lightcolor );
	else
		ind_write("No color given");
	ind_pop();
}

void ren_camera( lookat, lookfrom, lookup, fov, hither, yon, background )
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
/*
This routine dumps a camera.
*/
{
	ger_debug("ren_camera");

	ind_write("Camera:"); ind_eol();
	ind_push();
	ind_write("Lookat:   "); dump_point(lookat);
	ind_write("Lookfrom: "); dump_point(lookfrom);
	ind_write("Lookup:   "); dump_vector(lookup);
	ind_write("Fovea:    %f",fov); ind_eol();
	ind_write("Hither:   %f",hither); ind_eol();
	ind_write("Yon:      %f",yon); ind_eol();
	ind_write("Background: "); dump_color(background); ind_eol();
	ind_pop();
}

void ren_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine reports a request to render the given object.
 */
{
	int thisid; 

	thisid= gob_idnum( thisgob );

	ger_debug("render: rendering object given by gob %d", thisid);
	ind_write("Rendering gob %d", thisid); ind_eol();
	ind_push();
	dump_trans( thistrans );
	dump_attr( thisattrlist );
	ind_pop();
}

void ren_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/*
This routine reports a request to traverse the given object in search
of lights.
*/
{
	int thisid;

	thisid= gob_idnum( thisgob );
	ger_debug("ren_traverselights: setting lights to gob %d",thisid);
	ind_write("Traversing gob %d in search of lights", thisid);
	ind_eol();
	ind_push();
	dump_trans( thistrans );
	dump_attr( thisattrlist );
	ind_pop();
}

void ren_free(thisgob)
Gob thisgob;
/* 
This routine frees memory associated with the given gob.  Note that any 
future call referencing this gob or its children will be impossible unless 
it is somehow redefined first. 
*/
{
	int thisid;

	thisid= gob_idnum( thisgob );

	ger_debug("ren_free:  freeing gob %d",thisid);
	ind_write("Freeing gob %d", thisid); ind_eol();
}

void ren_setup(renderer,device,open_device,outfile,hints)
char *renderer, *device, *outfile;
int open_device;
Attribute_list hints;
/* This routine initializes the renderer, if it is not already initialized. */
{
	static int initialized=0; /* to hold initialization state */
 
	ger_debug("ren_setup: initializing renderer; renderer %s, device %s",
		renderer, device);

	/* 
	Ultimately renderer and device will be useful;  for the moment
	they are ignored.
	*/

 	/* 
	Generate some symbols to be used later in attribute list
	parsing.
	*/
	if (!initialized) {
		color_symbol= create_symbol("color");
		backcull_symbol= create_symbol("backcull");
		text_height_symbol= create_symbol("text-height");
		material_symbol= create_symbol("material");

 		ger_debug("ren_setup: initializing dumper");
		ind_setup();
		ind_write("Dump started"); ind_eol();

		initialized= 1;
	}
	else ger_error("ren_setup: called twice, this call ignored");

}

void ren_reset( hard )
int hard;
/* This routine resets the renderer */
{
	ger_debug("ren_reset: resetting renderer; hard= %d", hard);

        /* Hard resets require recreation of lisp-side variables */
        if (hard) {
		color_symbol= create_symbol("color");
		backcull_symbol= create_symbol("backcull");
		text_height_symbol= create_symbol("text-height");
		material_symbol= create_symbol("material");
	}
}

void ren_shutdown()
/* This routine shuts down the renderer */
{
	/* At the moment, this routine does nothing. */
	ger_debug("ren_shutdown: doing nothing");
}
