/****************************************************************************
 * control.c
 * Author Joel Welling and Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/* 
This module drives an animation controller, through the GPlot controller
interface.
*/

/* Several of these includes are needed only because of lvr_defines.h */
#include <stdio.h>
#include <time.h>
#include "ge_error.h"
#ifdef ultrix
#include <sys/types.h> 
#include <signal.h>
#endif
#ifdef sgi
#include <limits.h>
#endif
#include "defs.h" 	 /* the GPlot type definitions */
#include "alisp.h"       /* for access to global versions of argv and argc */
#include "lvr_defines.h" /* for use with lvr controller */

static struct one_opt opt[opt_size];	/* fake 'command line opts' */

static int (*ctrl[Delim_Size])();		/* controller functions */

static int initialized = 0;		/* non-zero if initialized */

static int pause_flag= 0;        /* non-zero to pause briefly each frame */

void ctrl_setup(controller,startframe,framecopies)
char *controller;
int startframe,framecopies;
/* 
This routine sets up the animation controller.  Its parameters are:
	controller	string which specifies which controller
	startframe	frame index at which to start animation
	framecopies	number of copies of each frame
 */
/* NOTE: Needs user name, description inputs; startfame for lvr??? */
{
        ger_debug(
	  "ctrl_setup: controller= <%s>, startframe= %d, framecopies= %d",
	  controller, startframe, framecopies );

	if (!initialized) {
		/* Set the appropriate 'command line options' */
		opt[ (int)copies ].set= 1;
		opt[ (int)copies ].val.i= framecopies;
		opt[ (int)start ].set= 1;
		opt[ (int)start ].val.i= startframe;

		/* 
		Call the controller setup routine (for whatever controller was
		requested).
		*/
		if (!strcmp(controller,"tty")) {
		  opt[ (int)tty ].set= 1;
		  opt[ (int)tty ].val.i= 1;
		  tty_ctrl(ctrl,opt);
		}
	        else if (!strcmp(controller,"lvr"))
		  {
		    pause_flag= 1; /* pause needed for scan converter */
		    opt[ (int)user ].set= 1;
		    strncpy(opt[(int)user].val.str,"P3D", MAXUSER);
		    
		    /* Steal input file name from command line */
		    opt[ (int)in_name ].set= 1;
		    strncpy(opt[(int)in_name].val.str,gargv[gargc-1], MAXCGM);
		    lvr_ctrl(ctrl,opt);
		  }	        
		else if (!strcmp(controller,"diaquest"))
#ifdef VMS
		  ger_fatal("DiaQuest controller not supported on VMS");
#else
		  dq_ctrl(ctrl,opt);
#endif
		else if (!strcmp(controller,"none"))
			; /* leave controller functions null */
		else ger_error("ctrl_setup: unknown controller %s requested",
			       controller);

		/* Now set up the controller. */
		if (ctrl[ (int)B_Mf ]) (*ctrl[ (int)B_Mf ])();

		initialized= !initialized;
		}
	else ger_error("ctrl_setup: called twice, this call ignored");
}

void ctrl_reset(hard,startframe,framecopies)
int hard,startframe,framecopies;
/*
This routine resets the starting frame and number of frames per image.
*/
{
        ger_debug("ctrl_reset: hard= %d, startframe= %d, framecopies= %d",
		  hard, startframe, framecopies);

	if (initialized) {

	  /* Hard resets require recreation of lisp symbols, but this doesn't
	   * use any.
	   */

	  /* Reset the framecopies 'command line option' */
	  opt[ (int)copies ].set= 1;
	  opt[ (int)copies ].val.i= framecopies;
	  
	  /* If a new startframe is given, restart at that frame */
	  if (startframe) {
	    if (ctrl[ (int)E_Mf ])(*ctrl[ (int)E_Mf ])();
	    opt[ (int)start ].set= 1;
	    opt[ (int)start ].val.i= startframe;
	    if (ctrl[ (int)B_Mf ]) (*ctrl[ (int)B_Mf ])();
	  }
	}
	else ger_error("ctrl_reset: called before setup, this call ignored");
}

#ifdef ultrix
static void null_sig_handler(sig, code, scp)
int sig, code;
struct sigcontext *scp; 
{
  /* This routine is needed to provide a non-terminating signal handler
   * for the pause routine used in ctrl_doframe().
   */
}
#endif

void ctrl_doframe()
/* This routine records nframes frames */
{
        ger_debug("ctrl_doframe:");

	/* We may need to pause, for example to let the scan converter
	 * output stabilize.  The following produces a 100 ms pause
	 * (at least on the supported machines).
	 */
	if (pause_flag) {
#ifdef ultrix
	  struct itimerval val;
	  struct sigvec sig, old_sig;
	  val.it_interval.tv_sec= 0;
	  val.it_interval.tv_usec= 0;
	  val.it_value.tv_sec= 0;
	  val.it_value.tv_usec= 100000; /* = .1 sec */
	  sig.sv_handler= null_sig_handler;
	  sig.sv_mask= 0;
	  sig.sv_flags= 0;
	  if ( sigvec(SIGALRM, &sig, &old_sig) )
	    perror("ctrl_doframe: sigvec1: ");
	  if ( !(setitimer( ITIMER_REAL, &val, 0 )) ) pause();
	  if ( sigvec(SIGALRM, &old_sig, 0) )
	    perror("ctrl_doframe: sigvec2: ");
#endif
#ifdef sgi
	  (void)sginap((long)(CLK_TCK/10)); /* wait .1 sec */
#endif
	}

	/* Record the frame(s) */
	if (ctrl[ (int)E_Pic ]) (*ctrl[ (int)E_Pic ])();
}

void ctrl_end()
/* This routine shuts down the animation controller */
{
        ger_debug("ctrl_end:");

	/* Shut down the controller */
	if (ctrl[ (int)E_Mf ])(*ctrl[ (int)E_Mf ])();
}
