/****************************************************************************
 * art_ren.c
 * Authors Joe Lappa and Joel Welling
 * Copyright 1991, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission to use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This renderer generates output compatible with the input of the ART ray
tracer (from the VORT package).
*/

#include <stdio.h>
#include <math.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "ren.h"
#include "indent.h"
#include "assist.h"

/* notes-
   -This implementation is tied to the calling behavior of the tui interface.
*/

/* Default values of parameters */
#define DEFAULT_RAYS_PER_PIXEL 4
#define DEFAULT_MAX_REFLECTION 6
#define DEFAULT_IGNORE_TEXT 0 /* text prims ignored if nonzero */
#define DEFAULT_X_RES 512  /* for output art commands (script mode only) */
#define DEFAULT_Y_RES 512  /* for output art commands (script mode only) */
#define ORTHO_STD_FOVEA 10.0 /* degrees, used for contracting eye dist. 
				in ortho projection */
#ifndef ART_COMMAND
#define ART_COMMAND "art"  /* used in generated output */
#endif

/* Values needed for creating shell script output */
static int x_res= DEFAULT_X_RES, y_res= DEFAULT_Y_RES;
static int script_flag= 0, frame_id= 0;

/* Needed external definitions */
#ifndef NOMALLOCDEF
extern char *malloc(), *realloc();
#endif

enum traversal_type { RENDER,LIGHTING,INVALID };
static enum traversal_type current_traversal = INVALID;

/* Symbols defined by ren_setup and used in parsing attribute lists */
static Symbol color_symbol, backcull_symbol, text_height_symbol;
static Symbol point_diameter_symbol, line_diameter_symbol, material_symbol;
static Symbol x_res_symbol, y_res_symbol, make_script_symbol;
static Symbol initial_frame_symbol, rays_per_pixel_symbol;
static Symbol max_reflection_symbol, ignore_text_symbol;
static Symbol text_stroke_width_symbol, text_thickness_symbol;
static Symbol shadows_symbol;

/* Current total ambient light intensity */
static float cur_amb_red, cur_amb_green, cur_amb_blue;

/* Globals used in communication while rendering text */
static float *current_text_trans= (float *)0;
static float current_text_normal[4];

/* Some other flags and parameters */
static int ignore_text_flag= DEFAULT_IGNORE_TEXT; 
static int rays_per_pixel= DEFAULT_RAYS_PER_PIXEL;
static int max_reflection= DEFAULT_MAX_REFLECTION;

static void dump_color_attr(color)
Color color;
{
	ind_write("colour %f,%f,%f ",
		color_red(color),
		color_green(color),
		color_blue(color));
	ind_eol();
	if(color_alpha(color) != 1.0) {
		ind_write("transparency %f",(1. - color_alpha(color)));
		ind_eol();
		}
}

static void dump_material_attr(material)
Material material;
{
  ger_debug("dump_material_attr:");
  ind_write("material %f, %f, %f, %f",
	    material_refract(material),
	    material_kd(material),
	    material_ks(material),
	    material_exp(material));
  ind_eol();
  ind_write("reflectance %f",material_reflect(material));
  ind_eol();
}

char *ren_def_material(material)
Material material;
/* This routine fills the slot for a material definition routine */
{
  ger_debug("ren_def_material: doing nothing");
  return((char *)0);
}

void ren_free_material(thismat)
Material thismat;
/* This routine fills the slot for a material deletion routine */
{
  ger_debug("ren_free_material: doing nothing");
}

/* 
This routine should dump the renderer's internal definition
of an object. 
*/
void ren_dump(thisgob)
Gob thisgob;
{
  ger_debug("ren_dump: doing nothing");
  fprintf(stderr,"GOB dumping is not implemented for this renderer.\n");
}

static void dump_triangle_vertex(vertex) 
Vertex vertex;
/* 
This routine is used to dump vertices from triangle strips
*/
{
  Color color;
  ind_write("( %f, %f, %f )",
	    vertex_x(vertex),
	    vertex_y(vertex),
	    vertex_z(vertex));
  if ( !null( color= vertex_color(vertex) ) )
    ind_write(", %f, %f, %f",
	      color_red(color),color_green(color),color_blue(color));
  ind_eol();
}

static void dump_polygon_vertex(vertex,count)
Vertex vertex;
/*
This routine is used to dump vertices from polygons
*/
{
  Vector vector;
  Color color;
  ind_write("vertex ( %f, %f, %f )",
	    vertex_x(vertex),vertex_y(vertex),vertex_z(vertex));
  if(count > 4) 
    ind_eol();
  else  {
    if ( !null( vector= vertex_normal(vertex) ) )
      ind_write(",( %f,%f,%f)",
		vector_x(vector),vector_y(vector),vector_z(vector));
    if ( !null( color= vertex_color(vertex) ) )
      ind_write(", %f, %f, %f",
		color_red(color),color_green(color),color_blue(color));
    ind_eol();
  }
}

void ren_sphere()
/*
This routine dumps a sphere primitive object.
*/
{
	ger_debug("ren_sphere");
	if (current_traversal != RENDER) return;
	ind_write("MySphere");
	ind_eol();
}

void ren_cylinder()
/*
This routine dumps a cylinder primitive object.
*/
{
	ger_debug("ren_cylinder");
	if (current_traversal != RENDER) return;
	ind_write("MyCylinder");
	ind_eol();
}

void ren_torus(bigradius, smallradius)
float bigradius, smallradius;
/*
This routine dumps a torus primitive object.
*/
{
	ger_debug("ren_torus");
	if (current_traversal != RENDER) return;
	ind_write("torus {");
	ind_eol();
	ind_push();
		ind_write("radius %f",bigradius);
		ind_eol();
		ind_write("radius %f",smallradius);
		ind_eol();
		ind_write("center (0.0,0.0,0.0)");
		ind_eol();
	ind_pop();
	ind_write("}"); ind_eol();
}

void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polyline primitive object.
*/
{
  Vertex vertex1, vertex2;
  float radius;
  Color color;

  ger_debug("Polyline primitive object: %d vertices", count);

  if (current_traversal != RENDER) return;
  
  if (count<2) {
    ger_error("ren_polyline: %d is not enough vertices; call ignored.",
	      count);
    return;
  }
  
  radius= 0.5*ast_float_attribute(line_diameter_symbol);
  while ( !null(rest_of_vertices(vlist)) ) {
    vertex1= first_vertex(vlist);
    vlist= rest_of_vertices( vlist );
    vertex2= first_vertex(vlist);

    /* Put in a sphere to miter the joint */
    ind_write("sphere {"); ind_eol();
    ind_push();
    ind_write("center ( %f, %f, %f )",
	      vertex_x(vertex1),vertex_y(vertex1),vertex_z(vertex1));
    ind_eol();
    ind_write("radius %f",radius); ind_eol();
    if ( !null( vertex_color(vertex1) ) ) {
      color= vertex_color(vertex1);
      ind_write("colour %f, %f, %f",
		color_red(color), color_green(color), color_blue(color));
      ind_eol();
    }
    ind_pop();
    ind_write("}"); ind_eol();

    /* Put in a cylinder to draw the line */
    ind_write("cylinder {"); ind_eol();
    ind_push();
    ind_write("center ( %f, %f, %f )",
	      vertex_x(vertex1),vertex_y(vertex1),vertex_z(vertex1));
    ind_eol();
    ind_write("center ( %f, %f, %f )",
	      vertex_x(vertex2),vertex_y(vertex2),vertex_z(vertex2));
    ind_eol();
    ind_write("radius %f",radius); ind_eol();
    if ( !null( vertex_color(vertex1) ) ) {
      color= vertex_color(vertex1);
      ind_write("colour %f, %f, %f",
		color_red(color), color_green(color), color_blue(color));
      ind_eol();
    }
    ind_pop();
    ind_write("}"); ind_eol();
  }

  /* We still need the closing sphere */
  ind_write("sphere {"); ind_eol();
  ind_push();
  ind_write("center ( %f, %f, %f )",
	    vertex_x(vertex2),vertex_y(vertex2),vertex_z(vertex2));
  ind_eol();
  ind_write("radius %f",radius); ind_eol();
  if ( !null( vertex_color(vertex2) ) ) {
    color= vertex_color(vertex2);
    ind_write("colour %f, %f, %f",
	      color_red(color), color_green(color), color_blue(color));
    ind_eol();
  }
  ind_pop();
  ind_write("}"); ind_eol();
}

void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polymarker primitive object.
*/
{
  Vertex vertex;
  float radius;
  Color color;

  ger_debug("Polymarker primitive object: %d vertices", count);

  if (current_traversal != RENDER) return;
  
  if (count<1) {
    ger_error("ren_polymarker: %d is not enough vertices; call ignored.",
	      count);
    return;
  }
  
  radius= 0.5*ast_float_attribute(point_diameter_symbol);
  while (!null(vlist)) {
    vertex= first_vertex(vlist);
    ind_write("sphere {"); ind_eol();
    ind_push();
    ind_write("center ( %f, %f, %f )",
	      vertex_x(vertex),vertex_y(vertex),vertex_z(vertex));
    ind_eol();
    ind_write("radius %f",radius); ind_eol();
    if ( !null( vertex_color(vertex) ) ) {
      color= vertex_color(vertex);
      ind_write("colour %f %f %f",
		color_red(color), color_green(color), color_blue(color));
      ind_eol();
    }
    ind_pop();
    ind_write("}"); ind_eol();
    vlist= rest_of_vertices( vlist );
  }
}


void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a polygon primitive object.
*/
{
  int thisid;
  Color color;	
  
  ger_debug("ren_polygon");

  if (current_traversal != RENDER) return;

  ind_write("polygon {"); ind_eol();
  ind_push();
  if (count<3) {
    ger_error("ren_polygon: %d is not enough vertices; call ignored.",
	      count);
    return;
  }
  while (!null(vlist)) {
    dump_polygon_vertex( first_vertex( vlist ) ,count);
    vlist= rest_of_vertices( vlist );
  }
  
  /* Close the polygon */
  ind_pop();
  ind_write("}"); ind_eol();
}

void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a triangle strip primitive object.
*/
{
  ger_debug("ren_triangle:");

  if (current_traversal != RENDER) return;

  if (count<3) {
    ger_error("ren_polygon: %d is not enough vertices; call ignored.",
	      count);
    return;
  }
  ind_write("geometry {"); ind_eol();
  ind_push();
  
  ind_write("phongshading on");
  ind_eol();
  
  ind_write("strip {"); ind_eol();
  ind_push();
  while (!null(vlist)) {
    dump_triangle_vertex( first_vertex( vlist ) );
    vlist= rest_of_vertices( vlist );
  }
  
  ind_pop();
  ind_write("}"); ind_eol();
  ind_pop();
  ind_write("}"); ind_eol();
}

void ren_mesh( vlist, vcount, flist, fcount )
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/*
This routine dumps a general mesh object from the args passed it.
*/
{
        Vertex_list thisfacet;
        int facet_vertices;

        ger_debug("ren_mesh");

	if (current_traversal != RENDER) return;

	ast_mesh_normals(vlist,vcount,flist,fcount);
        ind_write("composite {"); ind_eol();
        ind_push();
        while (fcount--) {
                thisfacet= first_facet( flist );
                flist= rest_of_facets( flist );
                facet_vertices= list_length( thisfacet );
                ren_polygon( thisfacet, facet_vertices );
                }
        ind_pop();
	ind_write("}"); ind_eol();
}

void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
/*
This routine dumps a Bezier patch primitive object.
*/
{
 	ger_debug("ren_bezier");

	if (current_traversal != RENDER) return;

	ast_bezier(vlist,ren_mesh);
}

static void dump_c_trans( matrix )
float *matrix;
/* This routine writes a transformation from a float array. */
{
  ger_debug("dump_c_trans: dumping transformation matrix");

  ind_write("transform {");
  ind_eol();
  ind_push();
  ind_write("%f,%f,%f,",*(matrix+0),*(matrix+4),*(matrix+8));
  ind_eol();
  ind_write("%f,%f,%f,",*(matrix+1),*(matrix+5),*(matrix+9));
  ind_eol();
  ind_write("%f,%f,%f,",*(matrix+2),*(matrix+6),*(matrix+10));
  ind_eol();
  ind_write("%f,%f,%f",*(matrix+3),*(matrix+7),*(matrix+11));
  ind_eol();
  ind_pop();
  ind_write("}");
  ind_eol();
}

static void dump_trans( trans )
Transformation trans;
/* This routine dumps a transformation matrix */
{
  float *matrix;
  register float *m_copy;
  
  ger_debug("dump_trans: converting transformation matrix");
  
  matrix= array2d_to_c( trans );
  dump_c_trans(matrix);
  free( (char *)matrix );
}

static void print_matrix(string,matrix)
char *string;
float *matrix;
/* This is a debugging tool */
{
  int i,j;

  fprintf(stderr,"%s:\n",string);
  for (j=0; j<4; j++) {
    for (i=0; i<4; i++) fprintf(stderr,"%f ",matrix[i+4*j]);
    fprintf(stderr,"\n");
  }
}

static void make_located_cylinder( pt1, diameter, depth )
float *pt1;
float diameter, depth;
/* This routine draws a cylinder of the given size at the given point,
 * for purposes of starting or terminating a text stroke. */
{
  float back[4];

  back[0]= pt1[0] - depth*current_text_normal[0];
  back[1]= pt1[1] - depth*current_text_normal[1];
  back[2]= pt1[2] - depth*current_text_normal[2];
  ind_write("cylinder {"); ind_eol();
  ind_push();
  ind_write("center ( %f, %f, %f )", pt1[0], pt1[1], pt1[2]); 
  ind_eol();
  ind_write("center ( %f, %f, %f )",back[0], back[1], back[2]);
  ind_eol();
  ind_write("radius %f",diameter/2.0); ind_eol();
  ind_pop();
  ind_write("}"); ind_eol();
}

static void make_aligned_block( pt1, pt2, height, depth )
float *pt1, *pt2;
float height, depth;
/* This routine draws a block of the given size between the given points,
 * for purposes of doing one stroke in a text string.
 */
{
  float norm;
  float corner1[4], corner2[4], sep[4], yvec[4];
  float *rot1, *rot2, *roty, *trans, *temp, *matrix;

  /* Calculate coordinates */
  sep[0]= pt2[0] - pt1[0];
  sep[1]= pt2[1] - pt1[1];
  sep[2]= pt2[2] - pt1[2];
  sep[3]= 1.0;
  norm= sqrt( sep[0]*sep[0] + sep[1]*sep[1] + sep[2]*sep[2] );

  /* Corners of axis-aligned box */
  corner1[0]= norm;
  corner1[1]= 0.0;
  corner1[2]= height/2.0;
  corner1[3]= 1.0;
  corner2[0]= 0.0;
  corner2[1]= -depth;
  corner2[2]= -height/2.0;
  corner2[3]= 1.0;
  yvec[0]= 0.0;
  yvec[1]= 1.0;
  yvec[2]= 0.0;
  yvec[3]= 1.0;

  /* Rotate the x axis into the alignment direction of the block.  We
   * have to watch out for the degenerate case where the alignment and
   * the x vector are anti-parallel.  If they are we simply rotate
   * about the y axis.  In all other cases the aligning rotation
   * is unique and we can't get into trouble.
   */
  if ( (sep[0] < 0.0) && (sep[1] == 0.0) && (sep[2] == 0.0) )
    rot1= make_rotate_c( 180.0, 0.0, 1.0, 0.0 );
  else
    rot1= make_aligning_rotation( 1.0, 0.0, 0.0, sep[0], sep[1], sep[2] );

  /* Rotate the y axis into the text normal direction.  We have to
   * watch out for the degenerate case where the text normal and the
   * rotated y vector are anti-parallel.  If they are we simply rotate
   * about the alignment axis.  In all other cases the aligning rotation
   * is unique and we can't get into trouble.
   */
  roty= matrix_vector_c( rot1, yvec );
  if ( (roty[0] == -current_text_normal[0])
      && (roty[1] == -current_text_normal[1])
      && (roty[2] == -current_text_normal[2]) )
    rot2= make_rotate_c( 180.0, sep[0], sep[1], sep[2] );
  else rot2= make_aligning_rotation( roty[0], roty[1], roty[2],
				current_text_normal[0], current_text_normal[1],
				current_text_normal[2] );

  /* Translate the box to the appropriate position */
  trans= make_translate_c( pt1[0], pt1[1], pt1[2] );

  /* Final matrix is the product. */
  temp= matrix_mult_c( rot2, rot1 );
  matrix= matrix_mult_c( trans, temp );

  /* Write the box */
  ind_write("composite {"); ind_eol();
  ind_push();
  dump_c_trans( matrix );
  ind_write("box {"); ind_eol();
  ind_push();
  ind_write("vertex ( %f, %f, %f )", 
	    corner1[0], corner1[1], corner1[2]); 
  ind_eol();
  ind_write("vertex ( %f, %f, %f )", 
	    corner2[0], corner2[1], corner2[2]); 
  ind_eol();
  ind_pop();
  ind_write("}"); ind_eol();
  ind_pop();
  ind_write("}"); ind_eol();

  /* Clean up */
  free( (char *)rot1 );
  free( (char *)rot2 );
  free( (char *)roty );
  free( (char *)trans );
  free( (char *)temp );
  free( (char *)matrix );
}

static void text_pline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine is used by ast_text to draw line segments.
*/
{
  float *matrix;
  float height, depth, textheight;
  float pt1[4], pt2[4];
  Vertex vertex1, vertex2;

  ger_debug("text_pline:");

  textheight= ast_float_attribute(text_height_symbol);
  height= textheight * ast_float_attribute(text_stroke_width_symbol);
  depth= textheight * ast_float_attribute(text_thickness_symbol);
  
  matrix= current_text_trans;

  /* Write the wrapper and the current transform */
  ind_write("composite {");
  ind_eol();
  ind_push();
  dump_c_trans(matrix);

  /* Generate the polyline of text */
  while ( !null(rest_of_vertices(vlist)) ) {
    vertex1= first_vertex(vlist);
    vlist= rest_of_vertices( vlist );
    vertex2= first_vertex(vlist);

    pt1[0]= vertex_x(vertex1);
    pt1[1]= vertex_y(vertex1);
    pt1[2]= vertex_z(vertex1);
    pt1[3]= 1.0;
    pt2[0]= vertex_x(vertex2);
    pt2[1]= vertex_y(vertex2);
    pt2[2]= vertex_z(vertex2);
    pt2[3]= 1.0;

    /* Put in a cylinder to miter the joint */
    make_located_cylinder(pt1, height, depth);

    /* Put in a block to draw the line */
    make_aligned_block(pt1, pt2, height, depth);
  }

  /* We still need the cylinder sphere */
  make_located_cylinder(pt2, height, depth);

  /* Close the wrapper */
  ind_pop();
  ind_write("}");
  ind_eol();
}

static void text_trans(trans)
Transformation trans;
/* This function is used by ast_text to do transformations between letters. */
{
  float *transmatrix, *product;

  /* Premultiply the current text transform by the given transform,
   * and reset the current transform to the result.
   */
  transmatrix= array2d_to_c( trans );
  if (!current_text_trans)
    ger_fatal("text_trans: algorithm error; current_text_trans is null!");
  product= matrix_mult_c( transmatrix, current_text_trans );
  free( (char *)transmatrix );
  free( (char *)current_text_trans );
  current_text_trans= product;
}

void ren_text(txtpoint,uvec,vvec,txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
/*
This routine dumps a text primitive object.
*/
{
  static Transformation identity_trans;
  float norm;

  ger_debug("ren_text");
  
  if (current_traversal != RENDER) return;

  if (!ignore_text_flag) {
    /* Initialize the transformation matrix */
    if (current_text_trans) free( (char *)current_text_trans );
    else { /* First call to this routine */
      identity_trans= Make_Identity();
      grab(identity_trans);
    }
    current_text_trans= array2d_to_c( identity_trans ); /* allocs space */
    
    /* Calculate the text normal direction */
    current_text_normal[0]= 
      vector_y(uvec) * vector_z(vvec) - vector_z(uvec) * vector_y(vvec);
    current_text_normal[1]= 
      vector_z(uvec) * vector_x(vvec) - vector_x(uvec) * vector_z(vvec);
    current_text_normal[2]= 
      vector_x(uvec) * vector_y(vvec) - vector_y(uvec) * vector_x(vvec);
    norm= sqrt( current_text_normal[0]*current_text_normal[0]
	       + current_text_normal[1]*current_text_normal[1]
	       + current_text_normal[2]*current_text_normal[2] );
    if (norm==0.0) {
      ger_error("ren_text: u and v vectors align; prim ignored.");
      return;
    }
    current_text_normal[0]= current_text_normal[0]/norm;
    current_text_normal[1]= current_text_normal[1]/norm;
    current_text_normal[2]= current_text_normal[2]/norm;
    current_text_normal[3]= 1.0;
    
    /* Add the text string, with identifying comments. */
    ind_write("/* Text string <%s> */",txtstring); ind_eol();
    ast_text(txtpoint, uvec, vvec, txtstring, text_trans, text_pline);
    ind_write("/* End of text string <%s> */",txtstring); ind_eol();
  }
  else {
    ind_write("/* Test string <%s> ignored */",txtstring);
    ind_eol();
  }
}

static void dump_attr( attrlist )
Attribute_list attrlist;
/* This routine dumps an attribute list */
{
  Pair thispair;
  
  ger_debug("dump_attr:");
  
  /* check for 'color attribute */
  if ( !null( thispair= symbol_attr( color_symbol, attrlist ) ) )
    dump_color_attr( pair_color( thispair ) );
  /* check for 'material attribute */
  if ( !null( thispair= symbol_attr( material_symbol, attrlist ) ) )
    dump_material_attr( pair_material( thispair ) );
  /* check for 'backcull' attribute */
  if ( !null( thispair= symbol_attr( backcull_symbol, attrlist ) ) ) {
    if ( pair_boolean(thispair) ) ind_write("backfacing on");
    else ind_write("backfacing off");
    ind_eol();
  }
  /* check for 'shadows' attribute */
  if ( !null( thispair= symbol_attr( shadows_symbol, attrlist ) ) ) {
    if ( pair_boolean(thispair) ) ind_write("shadows on");
    else ind_write("shadows off");
    ind_eol();
  }
}

void ren_gob(current_gob, trans, attr, primitive, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
/*
This routine sees all gobs as they are defined.  It dumps information
about their internal structure.
*/
{
  ger_debug("ren_gob: doing nothing");
}
 
void ren_light( location, lightcolor )
Point location;
Color lightcolor;
/*                                             
This routine constructs a light object, first adding a translation
and a color if necessary.  All new objects are added to the already
open structure.
*/
{
  float lightpos[4], *transformed_pos;

  ger_debug("ren_light");
  
  if (current_traversal != LIGHTING) return;

  /* Set the location, and transform to world coordinates */
  lightpos[0]= point_x( location );
  lightpos[1]= point_y( location );
  lightpos[2]= point_z( location );
  lightpos[3]= 1.0;
  transformed_pos= matrix_vector_c( ast_get_matrix(), lightpos ); 

  ind_write("light {"); ind_eol();
  ind_push();
  if (!max_reflection) {
    /* max_reflection==0 implies ray casting */
    ind_write("shadows off");
    ind_eol();
  }
  ind_write("colour %f, %f, %f",
	    color_red( lightcolor ),color_green( lightcolor ),
	    color_blue( lightcolor ));
  ind_eol();
  ind_write("location (%f, %f, %f)", 
	    transformed_pos[0], transformed_pos[1], transformed_pos[2]);
  ind_eol();
  ind_pop();
  ind_write("}");
  ind_eol();

  free( (char *)transformed_pos );
}

void ren_ambient(lightcolor)
Color lightcolor;
/*                                             
This routine dumps an ambient light primitive object.
*/
{
  ger_debug("ren_ambient:");

  if (current_traversal != LIGHTING) return;

  if ( !null(lightcolor) ) {
    cur_amb_red += color_red(lightcolor);
    cur_amb_green += color_green(lightcolor);
    cur_amb_blue += color_blue(lightcolor);
  }
}

static void write_frame_header()
/* This routine writes the stuff needed at the beginning of each
 * frame.
 */
{
  char filename[32];

  ger_debug("write_frame_header:");

  if (script_flag) {
    ind_write("# This is a csh script that runs the art ray tracer.");
    ind_eol();
    ind_write("echo Starting frame %d",frame_id); 
    ind_eol();
    sprintf(filename,"frame_%04d.pix",frame_id++);
    ind_write("%s - %d %d -o %s -n << End_Of_Scene",
	      ART_COMMAND, x_res, y_res, filename);
    ind_eol();
  }

  ind_write("raysperpixel %d",rays_per_pixel);
  ind_eol();
  ind_write("maxhitlevel %d",max_reflection);
  ind_eol();
  ind_eol();

  ind_write("composite MyCylinder {");
  ind_eol();
  ind_push();
  ind_write("cylinder {");
  ind_eol();
  ind_push();
  ind_write("radius 1");
  ind_eol();
  ind_write("center (0,0,0)");
  ind_eol();
  ind_write("center (0,0,1)");
  ind_eol();
  ind_pop();
  ind_write("}");
  ind_eol();
  ind_pop();
  ind_write("}");
  ind_eol();
  
  ind_write("composite MySphere {");
  ind_eol();
  ind_push();
  ind_write("sphere {");
  ind_eol();
  ind_push();
  ind_write("radius 1");
  ind_eol();
  ind_write("center (0,0,0)");
  ind_eol();
  ind_pop();
  ind_write("}");
  ind_eol();
  ind_pop();
  ind_write("}");
  ind_eol();
  ind_eol();
}

static void write_frame_end()
/* This writes the stuff that comes at the end of a frame */
{
  ger_debug("write_frame_end:");

  ind_reset();  /* We'd better be at the beginning of the line for this */

  if (script_flag) {
    ind_write("End_Of_Scene");
    ind_eol();
  }
}

static void calc_ortho_screen( width, height, lookfrom, lookat, 
			      fov, x_res, y_res )
float *width, *height;
Point lookfrom, lookat;
float fov;
int x_res, y_res;
/* This routine calculates equivalent orthographic projection screen
 * dimensions for a given fovea and view distance.
 */
{
    float dx, dy, dz, range, min_dim;

    ger_debug("calc_ortho_screen:");

    dx= point_x(lookfrom) - point_x(lookat);
    dy= point_y(lookfrom) - point_y(lookat);
    dz= point_z(lookfrom) - point_z(lookat);
    range= sqrt( dx*dx + dy*dy + dz*dz );

    if (fov==0.0) fov= 0.5; /* to avoid overflow */
    min_dim= 2.0*range*tan( DegtoRad*fov/2.0 );
    *width= *height= min_dim;
}

void ren_camera( lookat, lookfrom, lookup, fov, hither, yon, background )
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
/*
This routine dumps a camera.
*/
{
  float width, height, aspect_ratio;
  int min_res;

  ger_debug("ren_camera:");

  write_frame_header();

  ind_write("up (%f, %f, %f)",
	    vector_x(lookup),vector_y(lookup),vector_z(lookup)); 
  ind_eol();

  if (x_res < y_res) {
    min_res= x_res;
    aspect_ratio= (float)x_res/(float)y_res;
  }
  else {
    min_res= y_res;
    aspect_ratio= (float)y_res/(float)x_res;
  }

  /* If the fovea is less than 1 degree, we go over to orthographic
   * projection.
   */
  if (fov>=1.0) {
    /* Perspective projection.  Unfortunately the 'fieldofview' sets 
     * the widest angle visible on the screen, not the narrowest, so we must
     * do a little calculation to handle raster aspect ratios other
     * than 1.0.
     */
    float vx, vy, vz, range, half_width, corrected_fov;
    vx= point_x(lookat) - point_x(lookfrom);
    vy= point_y(lookat) - point_y(lookfrom);
    vz= point_z(lookat) - point_z(lookfrom);
    range= sqrt( vx*vx + vy*vy + vz*vz );
    half_width= range*tan( DegtoRad*fov/2.0 )/aspect_ratio;
    corrected_fov= 2.0 * RadtoDeg * atan( half_width/range );
    ind_write("lookat(%f, %f, %f, ",
	      point_x(lookfrom),point_y(lookfrom),point_z(lookfrom));
    ind_write("%f, %f, %f, 0.0)",
	      point_x(lookat),point_y(lookat),point_z(lookat)); 
    ind_eol();
    ind_write("screensize 0.0, 0.0");
    ind_eol();
    ind_write("fieldofview %f",corrected_fov); 
    ind_eol();
  }
  else {
    float vx, vy, vz, scale;
    /* Contract the view distance to something reasonable */
    vx= point_x(lookat) - point_x(lookfrom);
    vy= point_y(lookat) - point_y(lookfrom);
    vz= point_z(lookat) - point_z(lookfrom);
    scale= tan( DegtoRad * fov ) / tan( DegtoRad * ORTHO_STD_FOVEA );
    ind_write("lookat(%f, %f, %f, ",
	      point_x(lookat) - scale*vx,
	      point_y(lookat) - scale*vy,
	      point_z(lookat) - scale*vz);
    ind_write("%f, %f, %f, 0.0)",
              point_x(lookat), point_y(lookat), point_z(lookat)); 
    ind_eol();
    calc_ortho_screen( &width, &height, lookfrom, lookat, fov, x_res, y_res );
    ind_write("projection orthographic");
    ind_eol();
    ind_write("screensize %f, %f", width, height);
    ind_eol();
  }

  ind_write("background %f, %f, %f", color_red( background ),
	    color_green( background ),color_blue( background ));
  ind_eol();
  ind_eol();
}

static void fast_render(thisgob)
Gob thisgob;
/* This routine renders a gob, by handling its transformation and attributes
 * and either triggering execution of its primitive or rendering its
 * children.  It is used to traverse both the lighting and geometry
 * gobs.
 */
{
  int thisid;
  Attribute_list newattr;
  Transformation newtrans;
  Primitive newprim;
  Child_list kidlist; 

  thisid= gob_idnum( thisgob );
  ger_debug("fast-render: rendering object given by gob %d", thisid);

  /* Check for and if necessary handle new attributes */
  if ( !null( newattr=gob_attr(thisgob) ) )
    ast_push_attributes( newattr );

  ind_write("composite {"); ind_eol();
  ind_push();

  if(!null(newtrans= gob_trans(thisgob))) dump_trans(newtrans);
  if ( !null(newattr) ) dump_attr(newattr);

  /* Either execute the primitive, or render the children */
  if ( !null( newprim= gob_primitive(thisgob) ) ) 
    eval_function( primitive_op(newprim) );
  else {
    kidlist= gob_children( thisgob );
    while ( !null(kidlist) ) {
      fast_render( first_child( kidlist ) );
      kidlist= rest_of_children( kidlist );  
    }
  }

  ind_pop();
  ind_write("}"); ind_eol(); 

  /* Pop attributes and transformations, and clean up. */
  if ( !null(newattr) ) ast_pop_attributes( newattr );

}

void ren_render(thisgob, thistrans, thisattr)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattr;
/* 
This routine dumps a gob, by dumping any attributes or transforms it may
carry and by evaluating its primop (if present) or alternatively by
recursively dumping its children.
 */
{
  int thisid; 
  
  thisid= gob_idnum( thisgob );
  
  ger_debug("render: rendering object given by gob %d", thisid);
  
  if( !null(thisattr)) ast_push_attributes(thisattr);
  
  ind_write("composite {"); ind_eol();
  ind_push();
  current_traversal = RENDER ;
  if(!null(thistrans)) dump_trans(thistrans);
  if ( !null(thisattr) ) dump_attr(thisattr);
  fast_render(thisgob);
  ind_pop();
  ind_write("}"); ind_eol();
  
  if( !null(thisattr)) ast_pop_attributes(thisattr);
  current_traversal= INVALID;

  write_frame_end();
}

static void fast_traverse(thisgob)
Gob thisgob;
/* This routine traverses a gob DAG, by handling its transformation 
 * and attributes and either triggering execution of its primitive 
 * or rendering its children.  It is used to traverse the lighting DAG.
 */
{
  int thisid;
  Attribute_list newattr;
  Transformation newtrans;
  float *transmatrix;
  Primitive newprim;
  Child_list kidlist;

  thisid= gob_idnum( thisgob );
  ger_debug("fast-traverse: traversing object given by gob %d", thisid);

  /* If there is a transformation, add it to the transformation stack */
  if ( !null( newtrans=gob_trans(thisgob) ) ) {
    transmatrix= array2d_to_c( newtrans );
    ast_push_matrix(transmatrix);
  }
  /* Check for and if necessary handle new attributes */
  if ( !null( newattr=gob_attr(thisgob) ) )
    ast_push_attributes( newattr );

  /* Either execute the primitive, or render the children */
  if ( !null( newprim= gob_primitive(thisgob) ) ) {
    eval_function( primitive_op(newprim) );
  }
  else {
    kidlist= gob_children( thisgob );
    while ( !null(kidlist) ) {
      fast_traverse( first_child( kidlist ) );
      kidlist= rest_of_children( kidlist );
    }
  }

  /* Pop attributes and transformations, and clean up. */
  if ( !null(newattr) ) ast_pop_attributes( newattr );
  if ( !null(newtrans) ) {
    ast_pop_matrix();
    free( (char *)transmatrix );
    }
}

void ren_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* This routine adds the given object to the lights environment. */
{
  int thisid;
  float *transmatrix;

  thisid= gob_idnum( thisgob );
  ger_debug("ren_traverselights: setting lights to gob %d",thisid);
 
  /* Set up for traversal */
  current_traversal= LIGHTING;
  cur_amb_red= cur_amb_green= cur_amb_blue= 0.0;
  
  /* If there is a transformation, add it to the transformation stack */
  if ( !null( thistrans ) ) {
    transmatrix= array2d_to_c( thistrans );
    ast_push_matrix(transmatrix);
  }
  
  /* Check for and if necessary handle new attributes */
  if ( !null( thisattrlist ) ) ast_push_attributes( thisattrlist );
  
  fast_traverse( thisgob );

  /* Add ambient lighting */
  ind_write("ambient %f, %f, %f", cur_amb_red, cur_amb_green, cur_amb_blue);
  ind_eol();
  
  /* Pop attributes and transformations, and clean up. */
  if ( !null(thisattrlist) ) ast_pop_attributes( thisattrlist );
  if ( !null(thistrans) ) {
    ast_pop_matrix();
    free( (char *)transmatrix );
  }

  current_traversal= INVALID;
}

void ren_free(thisgob)
Gob thisgob;
/* 
This routine frees memory associated with the given gob.  Note that any 
future call referencing this gob or its children will be impossible unless 
it is somehow redefined first. 
*/
{
	int thisid;

	thisid= gob_idnum( thisgob );

	ger_debug("ren_free: gob %d; doing nothing",thisid);
}

static void parse_hints(hints)
Attribute_list hints;
/* This routine handles hint list parsing */
{
  Pair thispair;
  
  ger_debug("parse_hints:");

  if ( !null( thispair=symbol_attr(make_script_symbol,hints) ) ) {
    if ( script_flag= !null(pair_symbol( thispair )) ) 
      ger_debug("Generating shell script output");
  }
  if ( !null( thispair=symbol_attr(initial_frame_symbol,hints) ) ) {
    frame_id= pair_int(thispair);
    ger_debug("Initial frame id %d", frame_id);
  }
  if ( !null( thispair=symbol_attr(x_res_symbol,hints) ) ) {
    x_res= pair_int(thispair);
    if (x_res<1) x_res= 1;
    ger_debug("X resolution %d", x_res);
  }
  if ( !null( thispair=symbol_attr(y_res_symbol,hints) ) ) {
    y_res= pair_int(thispair);
    if (y_res<1) x_res= 1;
    ger_debug("Y resolution %d", y_res);
  }
  if ( !null( thispair=symbol_attr(ignore_text_symbol,hints) ) ) {
    if ( ignore_text_flag= !null(pair_symbol( thispair )) ) 
      ger_debug("Text primitives will be ignored");
  }
  if ( !null( thispair=symbol_attr(rays_per_pixel_symbol,hints) ) ) {
    rays_per_pixel= pair_int(thispair);
    if (rays_per_pixel<1) rays_per_pixel= 1;
    ger_debug("Rays per pixel will be %d", rays_per_pixel);
  }
  if ( !null( thispair=symbol_attr(max_reflection_symbol,hints) ) ) {
    max_reflection= pair_int(thispair);
    if (max_reflection<0) max_reflection= 0;
    ger_debug("Maximum reflection depth will be %d", max_reflection);
  }
}

void ren_setup(renderer,device,open_device,outfile,hints)
char *renderer, *device, *outfile;
int open_device;
Attribute_list hints;
/* This routine initializes the renderer, if it is not already initialized. */
{
  static int initialized=0; /* to hold initialization state */

  ger_debug("ren_setup: initializing renderer; renderer %s, device %s",
	    renderer, device);
  
  /* 
    Ultimately renderer and device will be useful;  for the moment
    they are ignored.
   */
  
  /* Initialize needed assist modules */
  ast_init_matrix();
  
  /* 
    Generate some symbols to be used later in attribute list
    parsing.
   */
  if (!initialized) {
    ger_debug("ren_setup: initializing renderer");

    /* Create needed symbols */
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    point_diameter_symbol= create_symbol("point-diameter");
    line_diameter_symbol= create_symbol("line-diameter");
    material_symbol= create_symbol("material");
    x_res_symbol= create_symbol("x-resolution");
    y_res_symbol= create_symbol("y-resolution");
    make_script_symbol= create_symbol("make-shellscript");
    initial_frame_symbol= create_symbol("initial-frame-id");
    rays_per_pixel_symbol= create_symbol("rays-per-pixel");
    max_reflection_symbol= create_symbol("max-reflection-depth");
    ignore_text_symbol= create_symbol("ignore-text");
    text_stroke_width_symbol= create_symbol("text-stroke-width-fraction");
    text_thickness_symbol= create_symbol("text-thickness-fraction");
    shadows_symbol= create_symbol("shadows");

    parse_hints(hints);

    ind_file_setup(outfile);
    
    initialized= 1;
  }
  else ger_error("ren_setup: called twice, this call ignored");
}

void ren_reset( hard )
int hard;
/* This routine resets the renderer */
{
  ger_debug("ren_reset: resetting renderer; hard= %d", hard);
  
  /* Hard resets require recreation of lisp-side variables */
  if (hard) {
    color_symbol= create_symbol("color");
    backcull_symbol= create_symbol("backcull");
    text_height_symbol= create_symbol("text-height");
    point_diameter_symbol= create_symbol("point-diameter");
    line_diameter_symbol= create_symbol("line-diameter");
    material_symbol= create_symbol("material");
    x_res_symbol= create_symbol("x-resolution");
    y_res_symbol= create_symbol("y-resolution");
    make_script_symbol= create_symbol("make-shellscript");
    initial_frame_symbol= create_symbol("initial_frame_id");
    rays_per_pixel_symbol= create_symbol("rays-per-pixel");
    max_reflection_symbol= create_symbol("max-reflection-depth");
    ignore_text_symbol= create_symbol("ignore-text");
    text_stroke_width_symbol= create_symbol("text-stroke-width-fraction");
    text_thickness_symbol= create_symbol("text-thickness-fraction");    
    shadows_symbol= create_symbol("shadows");

    current_text_trans= (float *)0; /* to trigger recreation of transform */
  }
}

void ren_shutdown()
/* This routine shuts down the renderer */
{
  ger_debug("ren_shutdown: doing nothing");
}

