/****************************************************************************
 * alisp_util.c
 * Author Joel Welling and Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

 /* 	this file defines various general purpose utility functions */

#include  <stdio.h>
#include  <ctype.h>

#include "alisp.h"
#define EOS '\0'

/* 
   Generic error printer.  Returns NIL
*/
NODE *handle_error(err_string)
char *err_string;
{

	fprintf(stderr," ERROR :%s \n",err_string);
	return(NIL);
}

/*
   returns an integer value between 0 and PRIME-1 to be used as <astring>'s
   index into the symbol table
*/
int hash_string(astring)
char *astring;
{
	char *p;
	unsigned h=0,g;

	for (p=astring;*p != EOS;p=p+1)
		{
		h = (h << 4) + (*p);
		if (g = h&0xf0000000)
			{
			h = h ^ (g >> 24);
			h = h ^ g;
			}
		}
	return(h%PRIME);
}

/* 
   returns the number of cons nodes in <list>.  Assumes a nil terminamted list
*/
int list_length(list)
NODE *list;
{
	int i;
	for (i=0;!null(list);i++)
		{
		list = cdr(list);
		if (!listp(list)) 
			{
			fprintf(stderr,
				"Error: non-nil terminated internal list\n");
			exit(0);
			}
		}
	return(i);
}

/*
   This predicate returns 1 if <exp> is a constant like a string, number, or
   boolean, 0 otherwise
*/
int is_const(exp)
NODE *exp;
{
	int result;
	
	if ( atom(exp) && ( stringp(exp) || fixp(exp) || null(exp) || 
	   floatp(exp) ) )
		result = (int) TRUE;
	else
		result = 0;

	return(result);
}

/*
    This function creates a new NODE X of type LIST with car(X) being
    <arg1> and cdr(X) being <arg2>.
*/
NODE *cons(arg1, arg2)
NODE *arg1, *arg2;
{
	NODE *cons_cell;


	cons_cell = new_node(N_LIST);
	incr_ref(arg1);
	incr_ref(arg2);
	rplaca(cons_cell, arg1);
	rplacd(cons_cell,arg2);

	return(cons_cell);
}

/*
    This function returns the cons cell from the list of cons cells <list> 
    whose car value is eq to <var>.  Commonly, <list> is the environment
    and <var> is a variable whose binding is desired.  It returns NIL 
    if <var> does not occur in <list>.  Note that alterations to the binding 
    returned will alter <list> by side effect. 
    Assumes a NIL terminated list, each of who's car is a cons cell.
*/
NODE *assoc(var, list)
NODE *var, *list;
{
	NODE *list_check;

	for (list_check=list;!null(list_check);list_check=cdr(list_check))
		{
		if (car(car(list_check)) == var)
			return(car(list_check));
		}
	return(NIL);
}

/*
    This function takes a list of unevaluated arguments <arg_list> and 
    returns an evaluated list. This is used before apply_prim or apply_lambda 
    is called.  This does not assume a nil terminated list, but perhaps that
    would be reasonable (look into it some time).
*/
NODE *eval_list(arg_lst)
NODE *arg_lst;
{
	NODE *currlist,*end_list,*result,*new_elem;
	NODE *cons();

	if (null(arg_lst))
		{
		result = NIL;
		return(result);
		}
	else if (!consp(arg_lst))
		{
		result = eval_ilisp(arg_lst);
		result = cons(result,NIL);
		return(result);
		}
	else
		result = new_node(N_LIST);

	end_list =  result;
	for (currlist=arg_lst; !(not_list(currlist)); currlist=cdr(currlist))
		{
		new_elem = eval_ilisp(car(currlist));
		if (not_list(cdr(currlist)))
			{
			rplaca(end_list,new_elem);   
			incr_ref(car(end_list));
			}
		else
			{
			rplaca(end_list,new_elem);
			incr_ref(car(end_list));

			rplacd(end_list,new_node(N_LIST));
			incr_ref(cdr(end_list));

			end_list = cdr(end_list);

			}
		}
	new_elem = eval_ilisp(currlist);
	rplacd(end_list,new_elem);
	incr_ref(cdr(end_list));
	return(result);
}


/* 
    This function returns a new list which copies every element of <arg1> and
    whose cdr points to <arg2>.  The result is apparently the concatination
    of arg1 and arg2. (This is the standard implementation).
*/
NODE *append(arg1, arg2)
NODE *arg1, *arg2;
{
	NODE *walk_arg1,*walk_app,*next_cell,*app_cell;

	if (!listp(arg1) || !listp(arg2))
		return(NIL);
	if (null(arg1))
		{
		if (null(arg2))   
			app_cell = NIL;
		else
			{
			app_cell = arg2;
			}
		return(app_cell);
		}
	walk_arg1 = arg1;
	app_cell = new_node(N_LIST);
	walk_app = app_cell;
	rplaca(app_cell,car(walk_arg1));
	incr_ref(car(app_cell));
	walk_arg1 = cdr(walk_arg1);
	while (!null(walk_arg1))
		{	
		next_cell = new_node(N_LIST);
		rplaca(next_cell,car(walk_arg1));
		incr_ref(car(next_cell));
		rplacd(walk_app,next_cell);
		incr_ref(cdr(walk_app));
		walk_app = next_cell;
		walk_arg1 = cdr(walk_arg1);
		}	
	rplacd(walk_app,arg2);
	incr_ref(cdr(walk_app)); 
	return(app_cell);
}


/*
   This function assumes that <arg1> is a NIL terminated list.  It replaces the
   terminating NIL of <arg1> with a <arg2>, thus concatinating the two lists.
   Note that, unlike append or cons, nothing is returned, but <arg1> is
   altered by side-effect to include <arg2>
*/
void join(arg1,arg2)
NODE *arg1,*arg2;
{
	NODE *walk_arg1,*cons(),*old;

	if (null(arg1))
		fprintf(stderr,"join null \n");

	for (walk_arg1=arg1;!not_list(cdr(walk_arg1));walk_arg1=cdr(walk_arg1));

	if (null(cdr(walk_arg1)))
		rplacd(walk_arg1,arg2);	
	else
		{
		fprintf(stderr,"suprise\n");
		old = cdr(walk_arg1);
		rplacd(walk_arg1,cons(cdr(walk_arg1),arg2));	
		decr_ref(arg2);
		decr_ref(old);
		incr_ref(cdr(walk_arg1));
		}

}

/*
   The opposite of join(), this function removes <arg2> from the list <arg1>,
   leaving <arg1> in the state it was in before join() was applied to it.  
   It is possible to apply this to a list that was not created by join(),
   but that is not the intent
*/
void unjoin(arg1,arg2)
NODE *arg1,*arg2;
{
	NODE *walk_arg1;

	for (walk_arg1=arg1;cdr(walk_arg1)!=arg2;walk_arg1=cdr(walk_arg1));
	rplacd(walk_arg1,NIL);

}

/*
   This creates a FUNARG node from the input parameters.  It is used for 
   turning a LAMBDA node into a FUNARG node.  Once a function is defined
   by 'defun', the name of the function is bound to the funarg of that 
   lambda expression in the environment.
*/
NODE *make_funarg(body,variables,kvars)
NODE *body,*variables,*kvars;
{
	NODE *funarg_node;


	return(NIL);
}


