;/****************************************************************************
; * P3D definitions file
; * Author Joel Welling and Chris Nuuja
; * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
; *
; * Permission use, copy, and modify this software and its documentation
; * without fee for personal use or use within your organization is hereby
; * granted, provided that the above copyright notice is preserved in all
; * copies and that that copyright and this permission notice appear in
; * supporting documentation.  Permission to redistribute this software to
; * other organizations or individuals is not granted;  that must be
; * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
; * University make any representations about the suitability of this
; * software for any purpose.  It is provided "as is" without express or
; * implied warranty.
; *****************************************************************************/

;Skip everything if this file has already been loaded once.
(if (boundp 'p3d-version) (pprint "p3d.lsp already loaded") (progn

;Various ID symbols
(setq p3d-version 2.2)
(setq p3d-id-string "PSC P3D 2.2")
(setq p3d-copyright 
 "Copyright 1992, Pittsburgh Supercomputing Center, Carnegie Mellon University")
(setq p3d-framenumber 0)

;Primitive structs
;Point in 3-space - this is simulated through call-outs now;  see func_defs.lsp
;(defstruct point 
;Coordinates of a point in 3-space 
;	(:constructor make-point (x y z))
;	(x 0.0 )
;	(y 0.0 )
;	(z 0.0 ))
 
;Vector - this is simulated through call-outs now; see func_defs.lsp
;(defstruct vector 
;	(:constructor make-vector (x y z))
;	(x 0.0 )
;	(y 0.0 )
;	(z 0.0 ))
 
;Color - this is simulated through call-outs now; see func_defs.lsp
;(defstruct color 
;	(:constructor make-color (r g b &optional a))
;	(r 0.8 ) 
;	(g 0.8 )
;	(b 0.8 )
;	(a 1.0 ))
 
;Vertex - this is simulated through call-outs now; see func_defs.lsp
;Vertex of geometrical primitive, possibly with color or normal
;(defstruct vertex
;Vertex structure
;	(:constructor make-vertex (x y z &optional clr &optional normal))
;	(:include point)
;	(x 0.0 )
;	(y 0.0 )
;	(z 0.0 )
;	(clr nil)
;	(normal nil)
;	(index 0))
 
;cameras
(defstruct camera
;The camera, specifies viewpoint
	(lookfrom origin)   	 ; intended type point
	(lookat origin)     	 ; type point
	(up y-vec)	    	 ; type vector
	(fovea 56.0)  		 ; type float
	(hither -0.01) 		 ; type float
	(yon -100.0)   	         ; type float
	(background black))      ; type color
               
;materials (really a user data object, but looks like a structure)
(defun def-material (&key (ka 0.8) (kd 0.8) (ks 0.3) (exp 30.0) 
			  (reflect 0.1) (refract 1.0) 
			  (energy (make-color :r 0.0 :g 0.0 :b 0.0 :a 1.0)))
  (make-material ka kd ks exp reflect refract energy))

;------------------------------------------------------------------------------
;Primitive structure and related defs
(defstruct primitive
; Primitives have a function that creates it and an idnum that identifies it
        primnum			; internally-generated id number
	operation)		; the operation that defines this instance

;Node structure and related defs
(defstruct gob
; Geometrical object;  holds a node 
	idnum				;internally-generated id number
	(parent-count 0)                ;internally-generated, num. of parents
	(primitive nil)			;must be a 'primitive' structure or nil
	(attr nil)			;attributes
	(transform nil)			;transformation
        (children nil)  		;list of decendants
        (hold nil) )                    ;T if gob is held
 
(setq current-gob-id 0)
(setq current-prim-id 0)

(defun increment-gob-id ()
	(setq current-gob-id (+ current-gob-id 1)))
	
(defun increment-prim-id ()
	(setq current-prim-id (+ current-prim-id 1)))
 
(defun def-gob (&rest arglist)
        (let ((newgob (apply 'make-gob 
			     (append arglist 
				     '(:idnum) (list (increment-gob-id))))))
	  (do* ((kidlist (gob-children newgob) (cdr kidlist))
	       (thiskid (car kidlist) (car kidlist)))
	      ((null kidlist) T)
	      (setf (aref thiskid 2) ;; alisp specific!
		    (+ (gob-parent-count thiskid) 1)))
	  (do-gob newgob)))

(defun def-prim-instance (function)
	(make-primitive :primnum (increment-prim-id) :operation function))

(defun free-gob (thisgob) ;does nothing if it has parents or is held
  (if (and (not (gob-hold thisgob)) (eq (gob-parent-count thisgob) 0))
    (progn 
       (do* ((kidlist (gob-children thisgob) (cdr kidlist))
             (thiskid (car kidlist) (car kidlist)))
            ((null kidlist) T)
            (setf (aref thiskid 2) ;; alisp specific!
	      (- (gob-parent-count thiskid) 1))
            (free-gob thiskid))
       (do-free-gob thisgob))
    t))

(defun hold-gob (thisgob) (setf (gob-hold thisgob) T))

(defun unhold-gob (thisgob) (setf (gob-hold thisgob) NIL))

;-----------------------------------------------------------------------------
;matrix ops
(defun compose-transforms (&rest translist) 
	(if (null (cdr translist))
		(car translist)
	     	(compose-transform (car translist) 
			(apply 'compose-transforms (cdr translist)))))

;------------------------------------------------------------------------------
;Primitive objects
(defun sphere () (def-gob :primitive (def-prim-instance '(do-sphere) )))
;draws in unit cube
 
(defun cylinder () (def-gob :primitive (def-prim-instance '(do-cylinder) )))
;draws in unit cube
       
(defun torus (bigradius smallradius)
	(def-gob :primitive (def-prim-instance 
			  (list 'do-torus bigradius smallradius))))
 
(defun polyline (&rest params) 
	(def-gob :primitive (def-prim-instance (cons 'do-polyline params))))
 
(defun polymarker (&rest params) 
	(def-gob :primitive (def-prim-instance (cons 'do-polymarker params))))
 
(defun polygon (&rest params) 
	(def-gob :primitive (def-prim-instance (cons 'do-polygon params))))
 
(defun triangle (&rest params) 
	(def-gob :primitive (def-prim-instance (cons 'do-triangle params))))

(defun mesh (&rest params)
	(def-gob :primitive 
	     (def-prim-instance 
		(list 'do-mesh 
			(list 'quote (car params))
			(list 'quote (cadr params))))))

(defun bezier (&rest params) 
	(def-gob :primitive (def-prim-instance (cons 'do-bezier params))))
 
(defun text (txtpoint uvec vvec txtstring)
	(def-gob :primitive (def-prim-instance 
		   	  (list 'do-text txtpoint uvec vvec txtstring))))
;------------------------------------------------------------------------------
;Light sources
(defun light (location lightcolor)
	(def-gob :primitive (def-prim-instance 
			  (list 'do-light location lightcolor))))

(defun ambient (lightcolor)
	(def-gob :primitive (def-prim-instance (list 'do-ambient lightcolor))))

;------------------------------------------------------------------------------
; Load the implementation-specific definitions here
(load 'func_defs.lsp)

;------------------------------------------------------------------------------
;Some basis vectors, colors, etc.
(setq origin (make-point :x 0.0 :y 0.0 :z 0.0))
 
(setq x-vec (make-vector :x 1.0 :y 0.0 :z 0.0))
(setq y-vec (make-vector :x 0.0 :y 1.0 :z 0.0))
(setq z-vec (make-vector :x 0.0 :y 0.0 :z 1.0))
 
(setq white (make-color :r 1.0 :g 1.0 :b 1.0))
(setq black (make-color :r 0.0 :g 0.0 :b 0.0))
(setq red (make-color :r 1.0 :g 0.0 :b 0.0))
(setq green (make-color :r 0.0 :g 1.0 :b 0.0))
(setq blue (make-color :r 0.0 :g 0.0 :b 1.0))
(setq yellow (make-color :r 1.0 :g 1.0 :b 0.0))
(setq cyan (make-color :r 0.0 :g 1.0 :b 1.0))
(setq magenta (make-color :r 1.0 :g 0.0 :b 1.0))
 
(setq null-transform (make-identity))

;------------------------------------------------------------------------------
;Function to render a view
(defun snap (object lights acamera)
        (setq p3d-framenumber (+ p3d-framenumber 1))
	(set-camera acamera)
	(traverselights lights (make-identity) p3d-default-attributes)
	(render object (make-identity) p3d-default-attributes))
 
;------------------------------------------------------------------------------
; Load the global and local default definition files
(load 'p3d_global.lsp)
(load 'p3d_dflts.lsp)

;------------------------------------------------------------------------------
; Initialize the system
(p3d-setup p3d-renderer p3d-device p3d-controller p3d-startframe 
	p3d-framecopies p3d-outfile p3d-hints)

;------------------------------------------------------------------------------
; Set up the predefined materials (can only happen after system initialzation)
(setq default-material (def-material))
(setq dull-material 
      (def-material :ka 0.9 :kd 0.9 :ks 0.1 :exp 5.0 :reflect 0.1))
(setq shiny-material 
      (def-material :ka 0.5 :kd 0.5 :ks 0.5 :exp 50.0 :reflect 0.3))
(setq metallic-material 
      (def-material :ka 0.1 :kd 0.1 :ks 0.9 :exp 100.0 :reflect 0.7))
(setq matte-material 
      (def-material :ka 1.0 :kd 1.0 :ks 0.0 :exp 0.0 :reflect 0.0))
(setq aluminum-material
      (def-material :ka 0.25 :kd 0.25 :ks 0.75 :exp 6.0 :reflect 0.75))
 
;------------------------------------------------------------------------------
; Evaluate the (previously quoted) default attributes

(setq p3d-default-attributes (eval p3d-default-attributes))

;------------------------------------------------------------------------------
;End of if clause at the very beginning.
))

