/***************************************************************************/
/* FUNCTION: int img_quant()
/*
/* DESCRIPTION:
/*    Quantizes a 24-bit image into an 8-bit image with a user specifed
/*    bit per component distribution.
/*
/* USAGE:
/*    error_ret = img_quant(buf24,buf8,xsiz,ysiz,bits1,bits2,bits3);
/*
/* ARGUMENTS:
/*    unsigned char  *buf24     (in) : input 24-bit image array
/*    unsigned char  *buf8     (out) : output 8-bit image array
/*    int            xsiz       (in) : X dimension of image
/*    int            ysiz       (in) : Y dimension of image
/*    int            bits1      (in) : bits per component1
/*    int            bits2      (in) : bits per component2
/*    int            bits3      (in) : bits per component3
/*
/* RETURNS: (int)
/*    0 : function call completed successfully
/*    1 : illegal value for bits1,bits2,bits3
/*    2 : bits1+bits2+bits3 exceeds quantization bits available
/*
/* COMMENTS:
/*    The 24-bit image input array must be in pixel interleaved format.
/*
/* EXAMPLE:
/*    As an example, to compress a 24-bit image into an 8-bit image with 
/*    3 bits for Red, 3-bits for Green, and 2-bits for Blue, you would use
/*    err_ret = img_quant(buf24,buf8,xsiz,ysiz,3,3,2).  To create the
/*    corresponding color map for the quantisized image, use the
/*    subroutine make_rgbpal().
/*
/* SEE ALSO:
/*    make_rgbpal()
/*
/* INFO:
/*    Author : Ray Idaszak
/*    Date   : May 8, 1989
/*    email  : rayi@ncsa.uiuc.edu
/***************************************************************************/
#include "rtep.h"

#define MAX_BITS 8

int img_quant(buf24,buf8,xsiz,ysiz,bits1,bits2,bits3)
unsigned char *buf24,*buf8;
int xsiz,ysiz,bits1,bits2,bits3;
{
	int err_ret;
	int num_pixels;
	int tbits;
	int ir,ig,ib;
	int rmask,gmask,bmask;
	int rshft,gshft,bshft;
	unsigned char *buf_ptr;
	int i;

	err_ret = 0;

	if ((bits1 < 0) || (bits2 < 0) || (bits3 < 0)){
		err_ret = 1;
		goto error;
	}

	tbits = bits1 + bits2 + bits3;
	if (tbits > MAX_BITS){
		err_ret = 2;
		goto error;
	}

	rmask = 0;
	gmask = 0;
	bmask = 0;

	if (bits1){
		rmask = 1 << (tbits - 1);
		for (i=0; i<(bits1-1); i++)
			rmask = (rmask >> 1) | rmask;
	}
	rshft = 0;

	if (bits2){
		gmask = 1 << (tbits - 1);
		for (i=0; i<(bits2-1); i++)
			gmask = (gmask >> 1) | gmask;
	}
	gshft = bits1;

	if (bits3){
		bmask = 1 << (tbits - 1);
		for (i=0; i<(bits3-1); i++)
			bmask = (bmask >> 1) | bmask;
	}
	bshft = bits1 + bits2;

	num_pixels = xsiz * ysiz;
	buf_ptr = &buf24[0];
	for(i=0; i<num_pixels; i++){
		ir = ((*buf_ptr++) & rmask);	
		ig = ((*buf_ptr++) & gmask) >> gshft;	
		ib = ((*buf_ptr++) & bmask) >> bshft;	
		buf8[i] = (unsigned char)(ir | ig | ib);
	}

error:
#if RTE_PRERR
	err_msg("img_quant",err_ret);
#endif
	return(err_ret);
}

/***************************************************************************/
/* FUNCTION: int img_quant2()
/*
/* DESCRIPTION:
/*    Quantizes a 24-bit image into an 8-bit image using a 
/*    sophisticated algorithm with excellent results.
/*
/* USAGE:
/*    error_ret = img_quant2(buf24,buf8,xsiz,ysiz,pal);
/*
/* ARGUMENTS:
/*    unsigned char  *buf24     (in) : input 24-bit image array
/*    unsigned char  *buf8     (out) : output 8-bit image array
/*    int            xsiz       (in) : X dimension of image
/*    int            ysiz       (in) : Y dimension of image
/*    unsigned char  *pal      (out) : palette for 8-bit image
/*				       (HDF 8-bit format)
/*
/* RETURNS: (int)
/*    0 : function call completed successfully
/*
/* COMMENTS:
/*    The 24-bit image input array must be in pixel interleaved format
/*    e.g. the first 3 pixels would be ordered as RGBRGBRGB.....
/*    for the first 9 bytes in the file.
/*
/* SEE ALSO:
/*
/* INFO:
/*    Author : NCSA
/*    Date   : May 9, 1990
/*    email  : rayi@ncsa.uiuc.edu
/***************************************************************************/

#define PALBUFSIZE 256

img_quant2(dat24,dat8,xres,yres,pal)
unsigned char *dat24,*dat8;
long xres,yres;
unsigned char *pal;
{
	register int ct,xct,yct;
	register int rres,rd,rr,rn,rct;
	register int gres,gd,gr,gn,gct;
	register int bres,bd,br,bn,bct;
	register int coff,cres;
	register unsigned int *idat[2];
	register unsigned int *cp,*np;
	register unsigned char *dip,*dop,*rp,*gp,*bp;
	unsigned char cdat[PALBUFSIZE * 3],*p;

	if ((idat[0] = (unsigned int *)malloc(6*xres*sizeof(unsigned int))) == NULL){
		printf("error: Memory allocation fault\n");
		return -1;
	}

	cres = 256;
	idat[1] = idat[0] + (3 * xres);

	rres = 6;
	gres = 7;
	bres = 6;
	coff = 2;

	rr = gr = br = 255;
	rn = rres - 1;
	gn = gres - 1;
	bn = bres - 1;

	rp = cdat + coff;
	gp = rp + cres;
	bp = gp + cres;

	for (rct=0; rct<rres; rct++){
		for (gct=0; gct<gres; gct++){
			for (bct=0; bct<bres; bct++){
				*rp++ = (unsigned char)(rr * rct / rn);
				*gp++ = (unsigned char)(gr * gct / gn);
				*bp++ = (unsigned char)(br * bct / bn);
			}
		}
	}

	rp = cdat;
	gp = rp + cres;
	bp = gp + cres;
	cp = idat[0];
	np = idat[1];
	dip = dat24;
	dop = dat8;

	for (xct=3*xres; --xct>=0; )
		*cp++ = *dip++;

	for (yct=0; yct<(yres-1); yct++){
		np = idat[(yct+1)%2];
		for (xct=3*xres; --xct>=0; )
			*np++ = *dip++;

		cp = idat[yct%2];
		np = idat[(yct+1)%2];

		if ((rct = (cp[0] * rn / rr)) > rn) rct = rn;
		if ((gct = (cp[1] * gn / gr)) > gn) gct = gn;
		if ((bct = (cp[2] * bn / br)) > bn) bct = bn;

		*dop++ = ct = (rct * gres + gct) * bres + bct + coff;

		rd = cp[0] - rp[ct];
		gd = cp[1] - gp[ct];
		bd = cp[2] - bp[ct];

		cp += 3;
		np += 3;

		cp[0]  += (rd * 7) >> 4;
		cp[1]  += (gd * 7) >> 4;
		cp[2]  += (bd * 7) >> 4;
		np[-3] += (rd * 5) >> 4;
		np[-2] += (gd * 5) >> 4;
		np[-1] += (bd * 5) >> 4;
		np[0]  += rd >> 4;
		np[1]  += gd >> 4;
		np[2]  += bd >> 4;

		for (xct=2; xct<xres; xct++){
			if ((rct = (cp[0] * rn / rr)) > rn) rct = rn;
			if ((gct = (cp[1] * gn / gr)) > gn) gct = gn;
			if ((bct = (cp[2] * bn / br)) > bn) bct = bn;

			*dop++ = ct = (rct * gres + gct) * bres + bct + coff;

			rd = cp[0] - rp[ct];
			gd = cp[1] - gp[ct];
			bd = cp[2] - bp[ct];

			cp += 3;
			np += 3;

			cp[0]  += (rd * 7) >> 4;
			cp[1]  += (gd * 7) >> 4;
			cp[2]  += (bd * 7) >> 4;
			np[-6] += (rd * 3) >> 4;
			np[-5] += (gd * 3) >> 4;
			np[-4] += (bd * 3) >> 4;
			np[-3] += (rd * 5) >> 4;
			np[-2] += (gd * 5) >> 4;
			np[-1] += (bd * 5) >> 4;
			np[0]  += rd >> 4;
			np[1]  += gd >> 4;
			np[2]  += bd >> 4;
		}

		if ((rct = (cp[0] * rn / rr)) > rn) rct = rn;
		if ((gct = (cp[1] * gn / gr)) > gn) gct = gn;
		if ((bct = (cp[2] * bn / br)) > bn) bct = bn;

		*dop++ = ct = (rct * gres + gct) * bres + bct + coff;

		rd = cp[0] - rp[ct];
		gd = cp[1] - gp[ct];
		bd = cp[2] - bp[ct];

		cp += 3;
		np += 3;

		np[-6] += (rd * 3) >> 4;
		np[-5] += (gd * 3) >> 4;
		np[-4] += (bd * 3) >> 4;
		np[-3] += (rd * 5) >> 4;
		np[-2] += (gd * 5) >> 4;
		np[-1] += (bd * 5) >> 4;
	}

	cp = idat[yct%2];

	if ((rct = (cp[0] * rn / rr)) > rn) rct = rn;
	if ((gct = (cp[1] * gn / gr)) > gn) gct = gn;
	if ((bct = (cp[2] * bn / br)) > bn) bct = bn;

	*dop++ = ct = (rct * gres + gct) * bres + bct + coff;

	rd = cp[0] - rp[ct];
	gd = cp[1] - gp[ct];
	bd = cp[2] - bp[ct];

	cp += 3;

	cp[0]  += (rd * 7) >> 4;
	cp[1]  += (gd * 7) >> 4;
	cp[2]  += (bd * 7) >> 4;

	for (xct=2; xct<xres; xct++){
		if ((rct = (cp[0] * rn / rr)) > rn) rct = rn;
		if ((gct = (cp[1] * gn / gr)) > gn) gct = gn;
		if ((bct = (cp[2] * bn / br)) > bn) bct = bn;

		*dop++ = ct = (rct * gres + gct) * bres + bct + coff;

		rd = cp[0] - rp[ct];
		gd = cp[1] - gp[ct];
		bd = cp[2] - bp[ct];

		cp += 3;

		cp[0]  += (rd * 7) >> 4;
		cp[1]  += (gd * 7) >> 4;
		cp[2]  += (bd * 7) >> 4;
	}

	if ((rct = (cp[0] * rn / rr)) > rn) rct = rn;
	if ((gct = (cp[1] * gn / gr)) > gn) gct = gn;
	if ((bct = (cp[2] * bn / br)) > bn) bct = bn;

	*dop++ = (rct * gres + gct) * bres + bct + coff;

	free(idat[0]);

	p = pal;
	for(xct=0; xct<256; xct++){
		*p++ = cdat[xct];
		*p++ = cdat[xct + PALBUFSIZE];
		*p++ = cdat[xct + (PALBUFSIZE + PALBUFSIZE)];
	}

	return(0);
}

