/*
 * Copyright (C) 1992, Board of Trustees of the University of Illinois.
 *
 * Permission is granted to copy and distribute source with out fee.
 * Commercialization of this product requires prior licensing
 * from the National Center for Supercomputing Applications of the
 * University of Illinois.  Commercialization includes the integration of this 
 * code in part or whole into a product for resale.  Free distribution of 
 * unmodified source and use of NCSA software is not considered 
 * commercialization.
 *
 */


/*
 * Functions to create and manage a hash table of pixmaps allocated when
 * an area is doodled on.  We  don't free areas if doodles are erased.
 */
#include <stdio.h>
#include <X11/Intrinsic.h>
#include "viewer.h"
#include "dhash.h"

#define	HASHFUNC(x, y)		(((y) << 6) ^ (x)) & 0x3fff

#define DBUCKETS	16384


extern Display *myDpy;


/*
 * Allocate an array of hash pointers and zero it.
 */
Dptr *
InitDHash()
{
	int i;
	Dptr *dptr;

	dptr = (Dptr *)MALLOC(DBUCKETS * sizeof(Dptr));
	if (dptr == NULL)
	{
		fprintf(stderr, "Cannot allocate memory for doodle hash table\n");
		exit(1);
	}
	for (i=0; i<DBUCKETS; i++)
	{
		dptr[i] = NULL;
	}
	return(dptr);
}


/*
 * Free all allocated hash table entries, find out deepest bucket.
 */
void
ClearDHash(dptr)
	Dptr *dptr;
{
	int i;
	int count, depth, tmpd;
	Dptr ptr1;
	Dptr ptr2;

	count = 0;
	depth = 0;
	for (i=0; i<DBUCKETS; i++)
	{
		ptr1 = dptr[i];
		tmpd = 0;
		while(ptr1 != NULL)
		{
			ptr2 = ptr1;
			ptr1 = ptr1->next;
			if (ptr2->pmap != (Pixmap)0)
			{
				XFreePixmap(myDpy, ptr2->pmap);
			}
			if (ptr2->clipmask != (Pixmap)0)
			{
				XFreePixmap(myDpy, ptr2->clipmask);
			}
			XtFree((char *)ptr2);
			count++;
			tmpd++;
		}
		if (tmpd > depth)
			depth = tmpd;
		dptr[i] = NULL;
	}
}


/*
 * Free all allocated hash table entries, find out deepest bucket.
 */
void
FreeDHash(dptr)
	Dptr *dptr;
{
	ClearDHash(dptr);
	XtFree((char *)dptr);
}


/*
 * Add a pixmap to the hash table
 */
void
AddDHash(dptr, x, y, pmap, clipmask)
	Dptr *dptr;
	int x, y;
	Pixmap pmap;
	Pixmap clipmask;
{
	int indx;
	Dptr ptr;

	indx = HASHFUNC(x, y);

	ptr = (Dptr)MALLOC(sizeof(Dhash));
	ptr->x = x;
	ptr->y = y;
	ptr->pmap = pmap;
	ptr->clipmask = clipmask;
	ptr->next = dptr[indx];
	dptr[indx] = ptr;
}


/*
 * Look up a pixmap in the hash table, and return the pixmap if found,
 * return (Pixmap)0 if the pixmap isn't in the hash table.
 */
Pixmap
FindDHash(dptr, x, y, clipptr)
	Dptr *dptr;
	int x, y;
	Pixmap *clipptr;
{
	int indx;
	Dptr ptr;

	indx = HASHFUNC(x, y);
	ptr = dptr[indx];
	while(ptr != NULL)
	{
		if ((ptr->x == x)&&
		    (ptr->y == y))
		{
			*clipptr = ptr->clipmask;
			return(ptr->pmap);
		}
		ptr = ptr->next;
	}
	*clipptr = (Pixmap)0;
	return((Pixmap)0);
}

