/*
  File: X11ShowWFF.c
  Authors: K.R. Sloan,
           Philip J. Schneider,
           Mike Schmidt,
           Jamie Painter, 
           J. Kercheval
  Last Modified: 17 January 1992
  Purpose: Display a WFF image file on an X11 display

           This is a retrenchment from the baroque recent past.  
           In  a radical change - this version supports the
           MINIMAL, ACTUALLY USED features.  We will add features
           as needed - and ONLY as needed.

           So...it's primitive because we WANT it to be primitive.
           And..it's specific, with several special cases where it
                might be possible to be more general - because we
                WANT it that way.
           
           ColorSystems supported are: 
                RGBA
                RGB
                I

           Displays supported are:
               24-bit Full Color
                8-bit PseudoColor
                4-bit monochrome
                1-bit monochrome

           If you need something else, please ASK! 
 */

#include <math.h>
#include <errno.h>
#include <sys/file.h>
#include <signal.h>
#include <stdio.h>
#include <wff.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xutil.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Form.h>
#include <X11/StringDefs.h>
#include <X11/Box.h>
#include <X11/Command.h>


extern int VERBOSE;
extern char *RoutineName;
extern Widget toplevel;


static Display *display;
static Screen  *screen;
static Window  win;
static Visual  *visual;
static Pixmap  pixmap;
static XImage  *image;
static int     ImageFormat;
static GC      gc;
static Widget  ask;
static int     width, height, depth;
static int     byte_order;

static XStandardColormap cmap;
static unsigned long pixels[256];
static XColor colors[256];

static int mycolormap;
static int ncolors;
static int grayscale;
static int NEGATIVE;


static void
FatalError(s)
 {
  fprintf(stderr,"%s: FatalError(%s)\n", RoutineName,s);
  exit(-1);
 }

/*
  User controls
 */

static void
term_handler( )
 {
  fflush(stdout);
  if (mycolormap) 
   {
    XGrabServer(display); 
    XDeleteProperty(display, DefaultRootWindow(display), 
		    grayscale ? XA_RGB_GRAY_MAP : XA_RGB_DEFAULT_MAP);
    XUngrabServer(display); 
   }
  mycolormap = 0;
  XSync(display,False);
  exit(0);
 }

static void
ExitButton(widget, clientData, callData)
 Widget widget;
 caddr_t clientData,callData;
 {
  XtMapWidget(ask);
  XWarpPointer(XtDisplay(ask),(Window) None,XtWindow(ask),0,0,0,0,10,35);
 }

static void
Cancel(widget, clientData, callData)
 Widget widget;
 caddr_t clientData,callData;
 {
  XtUnmapWidget(ask);
 }

static void
Quit(widget,clientData,callData)
 Widget widget;
 caddr_t clientData,callData;
 { 
  XtUnmapWidget(ask);
  XtDestroyWidget(toplevel);
  term_handler();
 }

/*
  Display device 
 */

int
InitializeColormap(grayscale)
 int *grayscale;
 {
  int i,r,g,b;

  /* Get the appropriate standard colormap */

  mycolormap = 0;
  cmap.colormap = 0;

  XGetStandardColormap(display, DefaultRootWindow(display),
                       &cmap, 
                       *grayscale ? XA_RGB_GRAY_MAP : XA_RGB_DEFAULT_MAP);

  /* If the standard map is defined, use it. */

  if ((Colormap)0 != cmap.colormap)
   {
    int i; 

    switch (depth)
     {
      default: FatalError("InitializeColormap: unsupported depth");

      case 8: ncolors = (*grayscale ? cmap.red_max+1
                                    : 1 + (cmap.red_max   * cmap.red_mult)
                                        + (cmap.green_max * cmap.green_mult)
                                        + (cmap.blue_max  * cmap.blue_mult));
              break;
     }

    for(i=0;i<ncolors;i++) colors[i].pixel = cmap.base_pixel + i; 

    XQueryColors(display,cmap.colormap,colors,ncolors);
    XSync(display,False);
   }

  /* If the standard map is not defined, define it.  See StandardMaps.c */ 

  if ((Colormap)0 == cmap.colormap) 
   {
    Colormap cm;
    Status cmstatus;
    int oncolors;

    switch(depth)
     {
      default: FatalError("InitializeColormap: unsupported depth");

      case 8: ncolors =   (*grayscale) ? 256 : 216;
              break;
     }

    cm = DefaultColormapOfScreen(screen);

    /* try to get up to ncolors */

    oncolors = ncolors;
    do 
     {
      cmstatus = XAllocColorCells(display, cm,
                                  TRUE,                   /* contiguous     */
                                  (unsigned long *)0, 0,  /* no plane masks */
                                  pixels, ncolors);       /* pixel values   */
      if ((Status)0 == cmstatus) 
       {
        if (*grayscale) ncolors--; /* try for fewer for grayscale */
        else break;               /* insist on all 216 for color */
       } 
     } while ((ncolors > 1) && ((Status)0 == cmstatus));

    if ((Status)0 == cmstatus) 
     {
      ncolors = oncolors;
      fprintf(stderr, "%s: Unable to allocate colormap cells.\n",RoutineName);
      fprintf(stderr, "%s: Hope you like this colormap.\n",RoutineName);
      pixels[0] = 3;
     }

    cmap.colormap = cm;
    cmap.base_pixel = pixels[0];

    /* Load the color map */

    if (*grayscale) 
     {
      for(i=0; i<ncolors; i++) 
       {
        colors[i].pixel = pixels[i];
        colors[i].red = colors[i].green = colors[i].blue
                      = (i * 65535) / (ncolors-1);
        colors[i].flags = DoRed | DoGreen | DoBlue;
       }
      cmap.red_max  = ncolors-1; cmap.green_max  = 0; cmap.blue_max  = 0;
      cmap.red_mult = 1;         cmap.green_mult = 0; cmap.blue_mult = 0;
     }
    else 
     {
      i = 0;
      for(b=0; b<6; b++)
       for(g=0; g<6; g++)
        for(r=0; r<6; r++) 
         {
          colors[i].pixel = pixels[i];
          colors[i].red   = r * 13107; /* 65535 / 5; */
          colors[i].green = g * 13107;
          colors[i].blue  = b * 13107;
          colors[i++].flags = DoRed | DoGreen | DoBlue;
         }
      cmap.red_max  = 5; cmap.green_max  = 5; cmap.blue_max  =  5;
      cmap.red_mult = 1; cmap.green_mult = 6; cmap.blue_mult = 36;
     } 

    /* Make this the standard map, for as long as we're around... */

    XStoreColors(display, cm, colors, ncolors);
    XInstallColormap(display, cm);

    XChangeProperty(display, DefaultRootWindow(display), 
                    *grayscale ? XA_RGB_GRAY_MAP : XA_RGB_DEFAULT_MAP,
                    XA_RGB_COLOR_MAP, 32, PropModeReplace,
                    (char *) &cmap, 
                    sizeof(cmap)/sizeof(long));

    mycolormap = -1;
  }
  XSetWindowColormap(display, win, cmap.colormap);
 }

static void
PackScanLine (ScanLineIn, ScanLineOut,
              width, BitsPerBand, BandsToSkip, KnownColorSystem, grayscale )
 unsigned short *ScanLineIn;
 unsigned char *ScanLineOut;
 int width, BitsPerBand, BandsToSkip, KnownColorSystem, grayscale;
 {
  int shift, A;
  int cr, cg, cb;
  register int x, I, r,g,b;
  unsigned long base_pixel;
  unsigned short *pin;
  unsigned char  *pout;
  long red_mult, green_mult, blue_mult;
  int r_B, g_B, b_B;

  cr = 77; cg = 151; cb = 28;         /* Luminance coeffs times 256 */
  shift = 16 - BitsPerBand;
  A = (1 << BitsPerBand) - 1;
  base_pixel = cmap.base_pixel;
  red_mult   = cmap.red_mult;
  green_mult = cmap.green_mult;
  blue_mult  = cmap.blue_mult;
  r_B = cmap.red_max  +1;
  g_B = cmap.green_max+1;
  b_B = cmap.blue_max +1;

  switch (KnownColorSystem)
   {
    case 1: /* RGB */
    case 2: /* RGBA */
     {
      if (grayscale) 
       {
        pin = ScanLineIn; pout = ScanLineOut;
        for(x=0; x<width; x++)
         {
          r = *pin++; g = *pin++; b = *pin++; pin += BandsToSkip;
          I = ((r*cr + g*cg + b*cb) + 127) >> 8;
          I *= ncolors;  I /= A; if (I >= ncolors) I = ncolors-1;
          if (NEGATIVE) I = ncolors-1-I;
          switch(depth)
           {
            case  1: *pout++ = (unsigned char)I;                break;
            case  8: *pout++ = (unsigned char)(base_pixel + I); break;
            case 24: *pout++ = (unsigned char)(base_pixel + I);
                     *pout++ = (unsigned char)(base_pixel + I);
                     *pout++ = (unsigned char)(base_pixel + I); break;
           }
         }
       }
      else 
       { 
        pin = ScanLineIn; pout = ScanLineOut;
        if (24 == depth) 
         {
          for(x=0; x<width; x++)
           {
            r = *pin++; g = *pin++; b = *pin++; pin += BandsToSkip;
	    if (byte_order == LSBFirst)
	     { *pout++ = b; *pout++ = g; *pout++ = r; *pout++ = 0; }
	    else
	     { *pout++ = 0; *pout++ = r; *pout++ = g; *pout++ = b; }
	   }
         }
        else
         {
          for (x=0;x<width;x++)
           {
            r = *pin++; g = *pin++; b = *pin++; pin += BandsToSkip;
            r *= r_B;  r /= A;   if (r >= r_B) r = r_B-1;
            g *= g_B;  g /= A;   if (g >= g_B) g = g_B-1;
            b *= b_B;  b /= A;   if (b >= b_B) b = b_B-1;
            *pout++ = (unsigned char) (  base_pixel 
                                       + (r * red_mult)
                                       + (g * green_mult)
                                       + (b * blue_mult) );
	   }
         }
       }      
      break;
     }

    case 3: /* I */
     {
      pin = ScanLineIn; pout = ScanLineOut;     
      for (x=0;x<width;x++)
       {
        I = *pin++; pin += BandsToSkip;
        I *= ncolors;  I /= A; if (I >= ncolors) I = ncolors-1;
        if (NEGATIVE) I = ncolors-1-I;
        switch(depth)
         {
          case  1: *pout++ = (unsigned char)I;                break;
          case  8: *pout++ = (unsigned char)(base_pixel + I); break;
          case 24: *pout++ = (unsigned char)(base_pixel + I);
                   *pout++ = (unsigned char)(base_pixel + I);
                   *pout++ = (unsigned char)(base_pixel + I); break;
         }
       }      
      break;
     }
   }

  switch (depth)
   {
    case 1: {
             unsigned char *p;
             register int r;

             p = pout = ScanLineOut;
             for(x=0; x<width; x+=8)
              {
               r =            *p++;
               r = (r << 1) | *p++;
               r = (r << 1) | *p++;
               r = (r << 1) | *p++;
               r = (r << 1) | *p++;
               r = (r << 1) | *p++;
               r = (r << 1) | *p++;
               r = (r << 1) | *p++;

               *pout++ = (unsigned char)r;
              }
	    }
            break;
    case 4: {
             unsigned char *p;
             register int r;

             p = pout = ScanLineOut;
             for(x=0; x<width; x+=2)
              {
               r =            *p++; 
               r = (r << 4) | *p++;
               *pout++ = (unsigned char)r;
              }
            }
            break;

    default: break; 
   }
 } 

static void
DisplayScanLine (ScreenScanLinePtr,y,Left,Right)
 int *ScreenScanLinePtr,y,Left,Right;
 {
  if ((y   <   0    ) || (height <= y)) return; /* off the screen */

  if (Left <   0    ) { ScreenScanLinePtr -= Left; Left = 0; }
  if (width <= Right)  Right = width-1;

  XPutImage(display,pixmap,gc,image,0,0,Left,height-1-y,Right-Left+1,1);
  XPutImage(display,win,   gc,image,0,0,Left,height-1-y,Right-Left+1,1);
 }


static void
InitializeDevice (useroot, grayscale, ScanLineOut)
 int useroot, *grayscale;
 unsigned char *ScanLineOut;
 {
  Widget form, button, tmp;
  Window root;
  Arg arglist[10];
  XtAppContext apContext;
  XEvent event;
  unsigned int n, x, y, border, nwidth, nheight;

  if (!useroot)  
   {
    n=0;
    XtSetArg(arglist[n],XtNdefaultDistance,0); n++;
    form = XtCreateManagedWidget("form",formWidgetClass,toplevel,arglist,n);

    n=0;
    XtSetArg(arglist[n],XtNmappedWhenManaged,False); n++;
    XtSetArg(arglist[n],XtNdefaultDistance,0); n++;
    ask = XtCreateManagedWidget("Ask",boxWidgetClass,form,arglist,n);
    tmp = XtCreateManagedWidget("Continue",commandWidgetClass,ask,(Arg *)0,0);
    XtAddCallback(tmp,XtNcallback,Cancel,NULL);
    tmp = XtCreateManagedWidget("Quit",commandWidgetClass,ask,(Arg *)0,0);
    XtAddCallback(tmp,XtNcallback,Quit,NULL);

    n=0;
    XtSetArg(arglist[n],XtNlabel,""); n++;
    XtSetArg(arglist[n],XtNdefaultDistance,0); n++;
    XtSetArg(arglist[n],XtNborderWidth,0); n++;
    XtSetArg(arglist[n],XtNhighlightThickness,0); n++;
    XtSetArg(arglist[n],XtNwidth,width); n++;
    XtSetArg(arglist[n],XtNheight,height); n++;
    button = XtCreateManagedWidget ( "",commandWidgetClass,form,arglist,n);
    XtAddCallback(button,XtNcallback,ExitButton,(caddr_t) NULL);

    XtRealizeWidget(toplevel);

    apContext = XtWidgetToApplicationContext(toplevel);

    for(n=0; n < 2; )
     {
      XtAppNextEvent(apContext,&event);
      if (event.type == MapNotify) n++;
     }
   }

  display = XtDisplay(toplevel);

  if (useroot) win = DefaultRootWindow( display );
  else         win = XtWindow(button);
  if ((Window)0 == win) FatalError("Unable to open X window");

  screen = XtScreen(toplevel);
  visual = DefaultVisualOfScreen(screen);
  gc     = DefaultGCOfScreen(screen);
    
  XSync(display, False);

  XGetGeometry(display,win,&root,&x,&y,&nwidth,&nheight,&border,&depth);

  if (!useroot) { width  = nwidth; height = nheight; }

  /* Make the image structure for passing scanlines to the server */

  switch(depth)
   {
    default: FatalError("Unsupported display depth"); 

    case 1: ImageFormat = XYPixmap; break;

    case 8: ImageFormat = ZPixmap; break;
   }

  image = XCreateImage( display, visual,  depth, ImageFormat,
                        0,             /* offset */
                        ScanLineOut,   /* data   */
                        width, 1,      /* dimensions */
                        8,             /* bitmap_pad */
                        0              /* bytes_per_line-you figure it out */
                       );

  image->bitmap_unit = 8;	/* Units are bytes to avoid swapping issues */
  image->bitmap_bit_order = MSBFirst;

  /* Make a pixmap for updating the display */
  pixmap = XCreatePixmap(display, win, width, height, depth);
   
  if ((Pixmap)0 == pixmap) FatalError("No memory for Pixmap");

  /* Set up the colormap */

  switch(depth)
   {
    default: FatalError("Unsupported display depth"); 

    case  1: *grayscale = -1;
             ncolors = 2; 
             mycolormap = 0;            
             cmap.colormap = 0;
             cmap.base_pixel = 0;
             cmap.red_max = ncolors-1;
             cmap.red_mult = 1;
             NEGATIVE = (  (WhitePixel(display, DefaultScreen(display)))
                         < (BlackPixel(display, DefaultScreen(display))));
             break;

    case  4: *grayscale = -1;
             ncolors = 16;
             mycolormap = 0;
             cmap.colormap = 0;
             cmap.base_pixel = 0;
             cmap.red_max = ncolors-1;
             cmap.red_mult = 1;  
             NEGATIVE = 0;
             break;

    case  8: XGrabServer(display);  
             InitializeColormap(grayscale);
             XUngrabServer(display); 
             NEGATIVE = 0;
             break;

    case 24: mycolormap = 0;
             ncolors = 1 << 24;
             cmap.colormap = 0;
             cmap.base_pixel = 0;
             cmap.red_max = cmap.green_max = cmap.blue_max = (1 << 8) - 1;
             cmap.red_mult =   1 << 16;
             cmap.green_mult = 1 <<  8;
             cmap.blue_mult =  1 <<  0;
             NEGATIVE = 0;
             break;
   }
 }


int
X11ShowWFF(FileDescriptor, useroot)
 FILE *FileDescriptor;
 int useroot;
 {
  FrameBufferType *FrameBuffer;
  int Left, Bottom, Top, Right;
  int BitsPerBand, BandsPerPixel, BandsToSkip;
  char WhatBands[10], Name[NameLength], Value[ValueLength];
  int KnownColorSystem;
  int ScanLength, ScanCount, ScanOffset;
  int WFFx, WFFy, value, r;
  int WFFl,WFFr,WFFt,WFFb;
  int BandsToDisplay;
  unsigned char  *ScanLineOut;
  unsigned short *ScanLineIn;
  unsigned short *ScanStart;
  unsigned short *p;
  double AspectRatio;

  FrameBuffer = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FrameBuffer)) 
   FatalError("Opening FrameBuffer");
  if (FAILURE == PassImageIn(FileDescriptor, FrameBuffer))
   FatalError("Reading header");
  if (FAILURE == GetBounds(FrameBuffer, &Bottom, &Left, &Top, &Right)) 
   FatalError("Getting Bounds");
  if (FAILURE == GetColorSystem(FrameBuffer, WhatBands, &BitsPerBand))
   FatalError("Getting Color System");
  if (FAILURE == GetDescriptor(FrameBuffer,"BitsPerBand",Value))
   FatalError("Getting BitsPerBand");   
  BitsPerBand = atoi(Value);
  if (FAILURE == GetDescriptor(FrameBuffer,"AspectRatio",Value))
    AspectRatio = 1.0;
  else
    AspectRatio = atof(Value);

  BandsPerPixel = FrameBuffer->BandsPerPixel;

  if      (0 == strcmp(WhatBands,"RGB")) 
   { KnownColorSystem = 1; BandsToDisplay = 3;}
  else if (0 == strcmp(WhatBands,"RGBA"))
   { KnownColorSystem = 2; BandsToDisplay = 3;}
  else if (0 == strcmp(WhatBands,"I"))
   { KnownColorSystem = 3; BandsToDisplay = 1;}
  else 
   {
    fprintf(stderr,"%s: %s is unknown ColorSystem\n", RoutineName, WhatBands);
    if (BandsPerPixel < 3) 
     {
      KnownColorSystem = 3; BandsToDisplay = 1;
      fprintf(stderr,"%s: using ColorSystem = I\n", RoutineName);
     }
    else
     {
      KnownColorSystem = 1; BandsToDisplay = 3;
      fprintf(stderr,"%s: using ColorSystem = RGB\n", RoutineName);
     }
   }
  BandsToSkip  = BandsPerPixel - BandsToDisplay;

  ScanLength   = Right-Left+1;
  ScanCount    = Top-Bottom+1;
  ScanOffset   = 0;

  width = ScanLength;
  height = ScanCount;
  grayscale =  (3 == KnownColorSystem);

  ScanLineOut = (unsigned char *)calloc(width,sizeof(unsigned char));
  if (!ScanLineOut) FatalError("No memory for ScanLineOut");

  ScanLineIn  = (unsigned short *)calloc(ScanLength*BandsPerPixel,
                                         sizeof(unsigned short));
  if (!ScanLineIn) FatalError("No memory for ScanLineIn");

  InitializeDevice(useroot, &grayscale, ScanLineOut);

  if (ScanLength > width)
   {
    ScanOffset = (ScanLength-width)/2;
    ScanLength = width;
   }
  ScanStart = ScanLineIn + (ScanOffset*BandsToDisplay);

  for (WFFy=Bottom; WFFy<=Top; WFFy++) 
   {
    if (FAILURE == NextNPixelsIn( FrameBuffer, Right-Left+1, ScanLineIn)) 
     {
      (void)free(ScanLineOut); (void)free(ScanLineIn);
      break;
     }

    PackScanLine (ScanStart, ScanLineOut, 
                  ScanLength, BitsPerBand, BandsToSkip, KnownColorSystem, 
		  grayscale);
    
    DisplayScanLine(ScanLineOut,WFFy-Bottom,0,ScanLength);
   }

  (void)free(ScanLineIn); (void)free(ScanLineOut);
  (void)CloseFB(&FrameBuffer);

  signal (SIGINT, term_handler);

  /* Set the pixmap as the window background pixmap.  This is done so
  ** the server will patch it on on expose events automagically
  */
  XSetWindowBackgroundPixmap(display, win, pixmap);
  XClearWindow(display, win);	/* Force it to update */
  
  if (!useroot) XtMainLoop();   /* ... 'neath the streets of Boston */
  term_handler();               /* have a nice day */
 }





