/*
  File: Spheres.c
  Authors: Jinling Lee (Bill),
           K.R. Sloan   
  Last Modified:  21 March 1989
  Purpose: Draw GRID^2 shaded spheres, with different surface properties
           Note - all measurements are in meters.  
 */

#include <stdio.h>
#include <math.h>
#include "3D.h"
#include "Lighting.h"

int VERBOSE = 0;
double atof();

#define GRID (5)

static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"Usage is\n\t%s [-v][-e x y z][-o x y z][-u x y z][-f f]\n",
                 RoutineName);
 }

void DrawSphere(Center, Radius, Precision, OpticalAxis)
 PointType3D Center;
 double Radius;
 int Precision;
 VectorType3D OpticalAxis;
 {
  int i, j;
  PointType3D Pa, Pb, Pc, Pd, Points[3];
  VectorType3D Na, Nb, Nc, Nd, A, B, Normal;
  sgpColorType Ca, Cb, Cc, Cd, Colors[3];
  double alpha, beta, e;
  
  /* 
    Divide the sphere into triangles.
    The sphere is centered at the origin, so position on the sphere
    is "the same as" the surface normal.

   */
  e = M_PI/Precision;
  for(i=0; i<Precision; i++)
   {
    for(j=0; j<2*Precision; j++)
     {
      alpha = i*e;
      beta  = j*e;

      /* normals */

      Na.dx = sin(alpha)*cos(beta); 
      Na.dy = sin(alpha)*sin(beta);
      Na.dz = cos(alpha);

      Nb.dx = sin(alpha+e)*cos(beta);
      Nb.dy = sin(alpha+e)*sin(beta);
      Nb.dz = cos(alpha+e);

      Nc.dx = sin(alpha+e)*cos(beta+e);
      Nc.dy = sin(alpha+e)*sin(beta+e);
      Nc.dz = cos(alpha+e);

      Nd.dx = sin(alpha)*cos(beta+e);
      Nd.dy = sin(alpha)*sin(beta+e);
      Nd.dz = cos(alpha);

      /* points */

      Pa.x = Center.x + Radius*Na.dx;
      Pa.y = Center.y + Radius*Na.dy;
      Pa.z = Center.z + Radius*Na.dz;
     
      Pb.x = Center.x + Radius*Nb.dx;
      Pb.y = Center.y + Radius*Nb.dy;
      Pb.z = Center.z + Radius*Nb.dz;

      Pc.x = Center.x + Radius*Nc.dx;
      Pc.y = Center.y + Radius*Nc.dy;
      Pc.z = Center.z + Radius*Nc.dz;

      Pd.x = Center.x + Radius*Nd.dx;
      Pd.y = Center.y + Radius*Nd.dy;
      Pd.z = Center.z + Radius*Nd.dz;

      /* colors */

      Ca = ColorAtPoint(Pa, Na);
      Cb = ColorAtPoint(Pb, Nb);
      Cc = ColorAtPoint(Pc, Nc);
      Cd = ColorAtPoint(Pd, Nd);

      /* draw the triangles */

      Points[0] = Pa;  Colors[0]= Ca;
      Points[1] = Pb;  Colors[1]= Cb;
      Points[2] = Pc;  Colors[2]= Cc;

      /* check for self-shadowing */
      A.dx = Pb.x-Pa.x; A.dy = Pb.y-Pa.y; A.dz = Pb.z-Pa.z;
      B.dx = Pc.x-Pa.x; B.dy = Pc.y-Pa.y; B.dz = Pc.z-Pa.z;
      Normal = Cross3D(A,B);
      if (Dot3D(Normal, OpticalAxis) < 0.0) ShadeTriangle3D(Points, Colors);

      Points[0] = Pa;  Colors[0]= Ca;
      Points[1] = Pc;  Colors[1]= Cc;
      Points[2] = Pd;  Colors[2]= Cd;

      A.dx = Pc.x-Pa.x; A.dy = Pc.y-Pa.y; A.dz = Pc.z-Pa.z;
      B.dx = Pd.x-Pa.x; B.dy = Pd.y-Pa.y; B.dz = Pd.z-Pa.z;
      Normal = Cross3D(A,B);
      if (Dot3D(Normal, OpticalAxis) < 0.0) ShadeTriangle3D(Points, Colors);
    }
  }
 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  static int ArgsParsed = 0;
  static PointType3D  EyePoint    = {  3.0,  3.0,  3.0 };
  static VectorType3D OpticalAxis = { -1.0, -1.0, -1.0 };
  static VectorType3D Up          = {  0.0,  1.0,  0.0 };
  static double FocalLength = 0.0040;
  static sgpRectangleType Window   = {-0.0015, -0.0012, 0.0015, 0.0012};
  static sgpRectangleType Viewport = {-0.125,   0.00,   1.125,  1.00};  
  static double L = 10.0;      /* half width of light cube */
  double Exposure;
  PointType3D LightPosition;
  sgpColorType Color;
  sgpColorType Ambient, Diffuse, Specular, Sky;
  double Specularity;
  PointType3D Center;
  static double Radius;
  int Precision;
  int i,j;
  sgpRectangleType subViewport;

  RoutineName = argv[ArgsParsed++];

  while (ArgsParsed < argc)
   {
    if ('-' == argv[ArgsParsed][0])
     {
      switch (argv[ArgsParsed++][1])
       {
        case 'e':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         EyePoint.x = atof(argv[ArgsParsed++]);
         EyePoint.y = atof(argv[ArgsParsed++]);
         EyePoint.z = atof(argv[ArgsParsed++]);
         break;
        case 'o':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         OpticalAxis.dx = atof(argv[ArgsParsed++]);
         OpticalAxis.dy = atof(argv[ArgsParsed++]);
         OpticalAxis.dz = atof(argv[ArgsParsed++]);
         break;
        case 'u':
         if ((argc-ArgsParsed)<3) { usage(); exit (-1); }
         Up.dx = atof(argv[ArgsParsed++]);
         Up.dy = atof(argv[ArgsParsed++]);
         Up.dz = atof(argv[ArgsParsed++]);
         break;
        case 'f':
         if ((argc-ArgsParsed)<1) { usage(); exit (-1); }
         FocalLength = atof(argv[ArgsParsed++]);
         break;
        case 'v':
         VERBOSE = 1;
         break;  
        case 'h': 
        default:
         usage(); exit(-1);
       }
     }
    else { usage(); exit (-1); }
   }   

  while (ArgsParsed < argc) { usage(); exit (-1); }

  sgpInit(ColorDisplay);
  sgpSetWindow(Window);
  sgpSetViewport(Viewport);

  Camera3D(EyePoint, OpticalAxis, Up);
  Lens3D(FocalLength);                  SetHither(FocalLength);

  Exposure = 1.0 / 2.0;
  SetExposure(Exposure);

  Ambient.r = 0.3; Ambient.g = 0.3; Ambient.b = 0.6;
  Sky.r = Exposure*Ambient.r;
  Sky.g = Exposure*Ambient.g;
  Sky.b = Exposure*Ambient.b;  
  SetAmbient(Ambient); sgpColor(Sky); sgpClearScreen();

  LightPosition.x=  L; LightPosition.y= -L; LightPosition.z= -L;
  Color.r = 1.0; Color.g = 0.0; Color.b = 0.0;
  AddLightSource(LightPosition, Color);

  LightPosition.x= -L; LightPosition.y=  L; LightPosition.z= -L;
  Color.r = 0.0; Color.g = 1.0; Color.b = 0.0;
  AddLightSource(LightPosition, Color);

  LightPosition.x= -L; LightPosition.y= -L; LightPosition.z=  L;
  Color.r = 0.0; Color.g = 0.0; Color.b = 1.0;
  AddLightSource(LightPosition, Color);

  LightPosition.x=  L; LightPosition.y=  L; LightPosition.z= -L;
  Color.r = 1.0; Color.g = 1.0; Color.b = 0.0;
  AddLightSource(LightPosition, Color);

  LightPosition.x=  L; LightPosition.y= -L; LightPosition.z=  L;
  Color.r = 1.0; Color.g = 0.0; Color.b = 1.0;
  AddLightSource(LightPosition, Color);

  LightPosition.x= -L; LightPosition.y=  L; LightPosition.z=  L;
  Color.r = 0.0; Color.g = 1.0; Color.b = 1.0;
  AddLightSource(LightPosition, Color);

  LightPosition.x = L; LightPosition.y = L; LightPosition.z=  L;
  Color.r = 1.0; Color.g = 1.0; Color.b = 1.0;
  AddLightSource(LightPosition, Color);

  Center.x = 0.0; Center.y = 0.0; Center.z = 0.0;
  Radius = 1.0;
  Precision = 20;

  for(j=0;j<GRID;j++)
   {
    subViewport.Bottom  = Viewport.Bottom
                         +(double)   j   *(Viewport.Top-Viewport.Bottom)
                                         /(double)GRID;
    subViewport.Top     = Viewport.Bottom
                         +(double)(j+1.0)*(Viewport.Top-Viewport.Bottom)
                                         /(double)GRID;
    Diffuse.r = (double) (j+1.0) / (double)(GRID+1.0);
    Diffuse.g = Diffuse.r;
    Diffuse.b = Diffuse.r;
    Specular.r = 1.0 - Diffuse.r;
    Specular.g = 1.0 - Diffuse.g;
    Specular.b = 1.0 - Diffuse.b;
    for(i=0;i<GRID;i++)
     {
      subViewport.Left  = Viewport.Left
                         +(double)   i   *(Viewport.Right-Viewport.Left)
                                         /(double)GRID;
      subViewport.Right = Viewport.Left
                         +(double)(i+1.0)*(Viewport.Right-Viewport.Left)
                                         /(double)GRID;
      sgpSetViewport(subViewport);      
      Specularity = (i+1.0) * 16.0 / (GRID+1.0);
      SetSurfaceReflectance(Diffuse, Specular, Specularity);
      if (VERBOSE)
       fprintf(stderr,"%s: drawing sphere %d,%d\n",RoutineName,i,j);
      DrawSphere(Center, Radius, Precision, OpticalAxis);
     }    
   } 

  sgpQuit();
  exit(0);
}
