/*
--------------------------------------------------------------------- 
---		 EPIC (Efficient Pyramid Image Coder)             ---
---	 Designed by Eero P. Simoncelli and Edward H. Adelson     ---
---		    Written by Eero P. Simoncelli                 ---
---  Developed at the Vision Science Group, The Media Laboratory  ---
---	Copyright 1989, Massachusetts Institute of Technology     ---
---			 All rights reserved.                     ---
---------------------------------------------------------------------

Permission to use, copy, or modify this software and its documentation
for educational and research purposes only and without fee is hereby
granted, provided that this copyright notice appear on all copies and
supporting documentation.  For any other uses of this software, in
original or modified form, including but not limited to distribution
in whole or in part, specific prior permission must be obtained from
M.I.T. and the authors.  These programs shall not be used, rewritten,
or adapted as the basis of a commercial software or hardware product
without first obtaining appropriate licenses from M.I.T.  M.I.T. makes
no representations about the suitability of this software for any
purpose.  It is provided "as is" without express or implied warranty.

---------------------------------------------------------------------
*/

#include <math.h>
#include "epic.h"

/*
======================================================================
collapse_pyr() -- collapse (reconstruct) a QMF-style pyramid using a
3-tap (1-2-1) filter.  Assumes the pyramid was built with correct
inverse filters, and reflected edge treatment (reflect1).  Assumes
storage order of images: LN, HN, VN, DN, HN-1, VN-1, .... D1.  
WARNING: The contents of the pyramid are destroyed!  Both dimensions 
of the image must be divisible by 2^num_levels (the routine
does not check this).
=======================================================================
*/

collapse_pyr(pyr, result, x_size, y_size, num_levels)
  int *pyr, x_size, y_size, num_levels;
  register int *result;
  {
  int size = x_size;    /* For now, only square images! */
  register int val, rpos, row_rpos;
  register int *im;
  register int rsize, isize, ipos;
  register int ccount, lcount;
  register int r2size;
  int level;

  for (level = num_levels-1; level >= 0; level--)
      {
      rsize = size >> level;	/* divide by 2^level */
      r2size = rsize << 1;      /* mul by 2 */
      isize = rsize >> 1;

      for (rpos=0; rpos<rsize*rsize; rpos++) result[rpos] = 0;

      im = pyr;			/***** lowpass  image ******/
      for (ipos=isize+1, rpos=r2size+2, lcount=1;
	   lcount < isize;  ipos++, rpos+=rsize+2, lcount++)
	for (ccount = 1;  ccount < isize;  ipos++, rpos+=2, ccount++)
	    {
	    val = im[ipos];
	    result[rpos]                  += val;
	    result[rpos-1]                += (val >>= 1); 
	    result[rpos+1]                += val;
	    result[row_rpos = rpos-rsize] += val;
	    result[row_rpos-1]            += (val >>= 1); 
	    result[row_rpos+1]            += val;
	    result[row_rpos = rpos+rsize] += (val << 1);
	    result[row_rpos-1]            += val;
	    result[row_rpos+1]            += val;
	    }
      for (ipos=1, rpos=2, ccount=1; /* top, low */
	   ccount<isize; ipos++, rpos+=2, ccount++)
	  {
	  val = im[ipos];
	  result[rpos]        += val;
	  result[rpos-1]      += (val >>= 1);
	  result[rpos+1]      += val;
	  result[row_rpos = rpos+rsize]    += val;
	  result[row_rpos-1]  += (val >>= 1);
	  result[row_rpos+1]  += val;
	  }
      for (ipos=isize, rpos=r2size, ccount=1; /* left, low */
	   ccount<isize; ipos+=isize, rpos+=r2size, ccount++)
	  {
	  val = im[ipos];
	  result[rpos]        += val;
	  result[rpos+1]      += (val >>= 1);
	  result[row_rpos=rpos-rsize]  += val;
	  result[row_rpos+1] += (val >> 1);
          result[row_rpos=rpos+rsize]  += val;
	  result[row_rpos+1] += (val >> 1);
	  }
      for (ipos=isize*(isize-1)+1, rpos=rsize*(rsize-1)+2, ccount=1;
	   ccount<isize; ipos++, rpos+=2, ccount++)
	  {
	  val = im[ipos] >> 1;
	  result[rpos]   += val;
	  result[rpos-1] += (val >>= 1);
	  result[rpos+1] += val;
	  }
      for (ipos=(isize<<1)-1, rpos=r2size+rsize-1, ccount=1; /* right, low */
	   ccount<isize; ipos+=isize, rpos+=r2size, ccount++)
	  {
	  val = im[ipos] >> 1;
	  result[rpos] += val;
	  result[rpos-rsize] += (val >>= 1);
	  result[rpos+rsize] += val;
	  }
      val = im[0];
      result[0] += val;  
      result[1] += (val >>= 1); 
      result[rsize] += val;
      result[rsize+1] += val >> 1;
      val = im[isize-1] >> 1;
      result[rsize-1] += val;  
      result[r2size-1] += val >> 1;
      val = im[isize*(isize-1)] >> 1;
      result[rsize*(rsize-1)] += val;  
      result[rsize*(rsize-1)+1] += val>>1;
      result[rsize*rsize-1] += im[isize*isize-1] >> 2;   

      im = pyr+(2*isize*isize);	/****** vertical image ******/
      transpose_int_image(im, isize);   /* vertical image is transposed! */
      for (ipos=isize, rpos=(2*rsize)+1, lcount = 1;
	   lcount < isize;  ipos+=1, rpos+=rsize+2, lcount++)
	for (ccount = 1;  ccount < isize;  ipos++, rpos+=2, ccount++)
	    {
	    val = im[ipos];
	    result[rpos]       += val;
	    result[rpos-1]     -= (val >>= 1); /* mul by 1/2 */
	    result[rpos+1]     -= val;
	    row_rpos = rpos-rsize;
	    result[row_rpos]   += val;
	    result[row_rpos-1] -= (val >>= 1); /* mul by 1/2 */
	    result[row_rpos+1] -= val;
	    row_rpos = rpos+rsize;
	    result[row_rpos]   += (val << 1);
	    result[row_rpos-1] -= val;
	    result[row_rpos+1] -= val;
	    }
      for (ipos=0, rpos=1, ccount=1; /* top, vertical */
	   ccount<isize; ipos++, rpos+=2, ccount++)
	  {
	  val = im[ipos];
	  result[rpos]        += val;
	  result[rpos-1]      -= (val >>= 1);
	  result[rpos+1]      -= val;
	  result[row_rpos = rpos+rsize]    += val;
	  result[row_rpos-1]  -= (val >>= 1);
	  result[row_rpos+1]  -= val;
	  }
      for (ipos=(isize<<1)-1, rpos=r2size+rsize-1, ccount=1; /* right, vertical */
	   ccount<isize; ipos+=isize, rpos+=r2size, ccount++)
	  {
	  val = im[ipos];
	  result[rpos]        += val;
	  result[rpos-1]      -= (val >>= 1);
	  result[row_rpos=rpos-rsize]  += val;
	  result[row_rpos-1] -= (val >> 1);
          result[row_rpos=rpos+rsize]  += val;
	  result[row_rpos-1] -= (val >> 1);
	  }
      for (ipos=isize*(isize-1), rpos=rsize*(rsize-1)+1, ccount=1; 
	   ccount<isize; ipos++, rpos+=2, ccount++)
	  {
	  val = im[ipos] >> 1;
	  result[rpos]   += val;
	  result[rpos-1] -= (val >>= 1);
	  result[rpos+1] -= val;
	  }
      for (ipos=isize, rpos=r2size, ccount=1; /* left, vertical */ 
	   ccount<isize; ipos+=isize, rpos+=r2size, ccount++)
	  {
	  val = - (im[ipos] >> 1);
	  result[rpos] += val;
	  result[rpos-rsize] += (val >>= 1);
	  result[rpos+rsize] += val;
	  }
      val = - (im[0] >> 1);
      result[0] += val;  
      result[rsize] += val >> 1;
      val = im[isize-1];
      result[rsize-1] += val;  
      result[rsize-2] -= (val >>= 1);
      result[r2size-1] += val;
      result[r2size-2] -= val >> 1;
      result[rsize*(rsize-1)] -= im[isize*(isize-1)] >> 2;
      val = im[isize*isize-1] >> 1;   
      result[rsize*rsize-1] += val;
      result[rsize*rsize-2] -= val >> 1;
      
      im = pyr+(isize*isize);	/***** horizontal  image *****/  
      for (ipos=1, rpos=rsize+2, lcount = 1;
	   lcount < isize;  ipos+=1, rpos+=rsize+2, lcount++)
	for (ccount = 1;  ccount < isize;  ipos++, rpos+=2, ccount++)
	    {
	    val = im[ipos];
	    result[rpos]       += val;
	    result[rpos-1]     += (val >>= 1); /* mul by 1/2 */
	    result[rpos+1]     += val;
	    row_rpos = rpos-rsize;
	    result[row_rpos]   -= val;
	    result[row_rpos-1] -= (val >>= 1); /* mul by 1/2 */
	    result[row_rpos+1] -= val;
	    row_rpos = rpos+rsize;
	    result[row_rpos]   -= (val << 1);
	    result[row_rpos-1] -= val;
	    result[row_rpos+1] -= val;
	    }
      for (ipos=1, rpos=2, ccount=1; /* top, horizontal */
	   ccount<isize; ipos++, rpos+=2, ccount++)
	  {
	  val = - (im[ipos] >> 1);
	  result[rpos]        += val;
	  result[rpos-1]      += (val >>= 1);
	  result[rpos+1]      += val;
	  }
      for (ipos=0, rpos=rsize, ccount=1; /* left, horizontal */
	   ccount<isize; ipos+=isize, rpos+=r2size, ccount++)
	  {
	  val = im[ipos];
	  result[rpos]        += val;
	  result[rpos+1]      += (val >>= 1);
	  result[row_rpos=rpos-rsize]  -= val;
	  result[row_rpos+1] -= (val >> 1);
          result[row_rpos=rpos+rsize]  -= val;
	  result[row_rpos+1] -= (val >> 1);
	  }
      for (ipos=isize*(isize-1)+1, rpos=rsize*(rsize-1)+2, ccount=1;
	   ccount<isize; ipos++, rpos+=2, ccount++)
	  {
	  val = im[ipos];
	  result[rpos]   += val;
	  result[rpos-1] += (val >>= 1);
	  result[rpos+1] += val;
	  result[row_rpos=rpos-rsize] -= val;
	  result[row_rpos+1] -= (val >>= 1);
	  result[row_rpos-1] -= val;
	  }
      for (ipos=isize-1, rpos=r2size-1, ccount=1; /* right, horizontal */
	   ccount<isize; ipos+=isize, rpos+=r2size, ccount++)
	  {
	  val = im[ipos] >> 1;
	  result[rpos] += val;
	  result[rpos-rsize] -= (val >>= 1);
	  result[rpos+rsize] -= val;
	  }
      val = - (im[0] >> 1);
      result[0] += val;  
      result[1] += val >> 1; 
      result[rsize-1] -= im[isize-1] >> 2;
      val = im[isize*(isize-1)];
      result[rsize*(rsize-1)] += val;  
      result[rsize*(rsize-1)+1] += (val >>= 1);
      result[rsize*(rsize-2)] -= val;
      result[rsize*(rsize-2)+1] -= val >> 1;
      val = im[isize*isize-1] >> 1;
      result[rsize*rsize-1] += val;
      result[rsize*(rsize-1)-1] -= val >> 1; 

      im = pyr+(3*isize*isize);	/*****  diagonal  image ******/
      for (ipos=0, rpos=rsize+1, lcount = 1;
	   lcount < isize;  ipos+=1, rpos+=rsize+2, lcount++)
	for (ccount = 1;  ccount < isize;  ipos++, rpos+=2, ccount++)
	    {
	    val = im[ipos];
	    result[rpos]       += val;
	    result[rpos-1]     -= (val >>= 1); /* mul by 1/2 */
	    result[rpos+1]     -= val;
	    result[row_rpos=rpos-rsize]   -= val;
	    result[row_rpos-1] += (val >>= 1); /* mul by 1/2 */
	    result[row_rpos+1] += val;
	    result[row_rpos=rpos+rsize]   -= (val << 1);
	    result[row_rpos-1] += val;
	    result[row_rpos+1] += val;
	    }
      for (ipos=0, rpos=1, ccount=1; /* top, diag */
	   ccount<isize; ipos++, rpos+=2, ccount++)
	  {
	  val = im[ipos] >> 1;
	  result[rpos]        -= val;
	  result[rpos-1]      += (val >>= 1);
	  result[rpos+1]      += val;
	  }
      for (ipos=0, rpos=rsize, ccount=1; /* left, diag */
	   ccount<isize; ipos+=isize, rpos+=r2size, ccount++)
	  {
	  val = im[ipos] >> 1;
	  result[rpos]        -= val;
	  result[rpos+rsize]  += (val >>= 1);
	  result[rpos-rsize]  += val;
	  }
      for (ipos=isize*(isize-1), rpos=rsize*(rsize-1)+1, ccount=1;
	   ccount<isize; ipos++, rpos+=2, ccount++)
	  {
	  val = im[ipos];
	  result[rpos]   += val;
	  result[rpos-1] -= (val >>= 1);
	  result[rpos+1] -= val;
	  result[row_rpos=rpos-rsize] -= val;
	  result[row_rpos+1] += (val >>= 1);
	  result[row_rpos-1] += val;
	  }
      for (ipos=isize-1, rpos=r2size-1, ccount=1; /* right diag */
	   ccount<isize; ipos+=isize, rpos+=r2size, ccount++)
	  {
	  val = im[ipos];
	  result[rpos] += val;
	  result[rpos-1] -= (val >>= 1);
	  result[row_rpos=rpos-rsize] -= val;
	  result[row_rpos-1] += val >> 1;
	  result[row_rpos=rpos+rsize] -= val;
	  result[row_rpos-1] += val >> 1;
	  }
      val = im[0] >> 2;
      result[0] += val;  
      val = im[isize-1] >> 1;
      result[rsize-1] -= val;
      result[rsize-2] += val >> 1;
      val = im[isize*(isize-1)] >> 1;
      result[rsize*(rsize-1)] -= val;  
      result[rsize*(rsize-2)] += val >> 1;
      val = im[isize*isize-1];
      result[row_rpos=rsize*rsize-1] += val;
      result[row_rpos-1] -= (val >>= 1); 
      result[row_rpos-rsize] -= val;
      result[row_rpos-rsize-1] += val >> 1;

      if (level > 0)		/* copy result into pyramid */
	for(im=pyr, rpos=0; rpos<rsize*rsize; rpos++)
	  im[rpos] = result[rpos];
      }				/* end for each level */
  }
  
/* In-place transpose of an integer image */
transpose_int_image(im, isize)
  register int *im;
  int isize;
  {
  register int x, y;
  register int x_ydim, y_xdim;
  register int temp;

  /* first, do minimal square upper corner */
  for (y = y_xdim = 0;  y < isize;  y++, y_xdim += isize)
    for (x = x_ydim = 0;  x <= y;  x++, x_ydim += isize)
	{
	temp = im[y_xdim+x];
	im[y_xdim+x] = im[x_ydim+y];
	im[x_ydim+y] = temp;
	}
  }
