/**********************************************************************
 *
 * txt2db.cpp -- 
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the 
 * University of Waikato, New Zealand.
 *
 * Copyright (C) 1999  The New Zealand Digital Library Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **********************************************************************/


#ifdef __WIN32__
#include "autoconf.h"
#include "systems.h"
#include "gdbmconst.h"
#include "gdbm.h"

#else
#include <gdbm.h>
#endif

#include "gsdlconf.h"
#include "text_t.h"
#include <stdlib.h>

#if defined(GSDL_USE_OBJECTSPACE)
#  include <ospace\std\iostream>
#elif defined(GSDL_USE_IOS_H)
#  include <iostream.h>
#else
#  include <iostream>
#endif


void print_usage (char *program_name) {
  cerr << "usage: " << program_name << " [options] database-name\n\n";
  cerr << "options:\n";
  cerr << " -append        append to existing database\n\n";
}


int main (int argc, char *argv[]) {
  int block_size = 0;
  GDBM_FILE dbf;
  char c;
  text_t key;
  text_t value;
  text_t tmp;
  int num_dashes = 0;

  // sanity check
  if (argc != 2 && argc != 3) {
    print_usage (argv[0]);
    exit (0);
  }
  
  char *dbname;
  int append = 0;
  if (argc == 3) {
    if (strcmp (argv[1], "-append") == 0) {
      append = 1;
      dbname = argv[2];
    } else {
      cerr << argv[1] << " is not a valid option.\n\n";
      print_usage (argv[0]);
      exit (0);
    }
  } else dbname = argv[1];

  
  // open the database
  // note that GDBM_FAST is obsolete on newer versions of gdbm
  int read_write = GDBM_NEWDB | GDBM_FAST;
  if (append) read_write = GDBM_WRCREAT | GDBM_FAST;
  
#ifdef __WIN32__
  dbf = gdbm_open (dbname, block_size, read_write, 00664, NULL, 1);
#else
  dbf = gdbm_open  (dbname, block_size, read_write, 00664, NULL);
#endif
  if (dbf == NULL) {
    cerr << "couldn't create " << dbname << "\n";
    exit (0);
  }
  
  cin.get(c);
  while (!cin.eof()) {
    num_dashes = 0;
    key = "";
    value = "";
    
    // look for [key]\n
    while (!cin.eof() && c != '[') cin.get(c);
    if (!cin.eof()) cin.get(c); // skip [
    while (!cin.eof() && c != ']') {
      key.push_back ((unsigned char)c);
      cin.get(c);
    }
    if (!cin.eof()) cin.get(c); // skip ]
    while (!cin.eof() && (c == '\n' || c == 'r')) cin.get(c);
    
    // look for 70 dashes
    tmp = "";
    while (!cin.eof() && (num_dashes < 70)) {
      if (c == '\n' || c == '\r') {
	tmp.push_back ((unsigned char)c);
	num_dashes = 0;
	
      } else if (c == '-') {
	tmp.push_back ((unsigned char)c);
	num_dashes++;
	
      } else {
	value += tmp;
	value.push_back ((unsigned char)c);
	tmp = "";
	num_dashes = 0;
      }
      cin.get(c);
    }
    
    // if the key is not an empty string store this key-value pair
    if (!key.empty()) {
      // convert key to a datum datatype
      datum key_data;
      key_data.dptr = key.getcstr();
      if (key_data.dptr == NULL) {
	cerr << "NULL key_data.dptr\n";
	exit (0);
      }
      key_data.dsize = strlen(key_data.dptr);
      
      // convert value to a datum datatype
      datum value_data;
      value_data.dptr = value.getcstr();
      if (value_data.dptr == NULL) {
	cerr << "NULL value_data.dptr\n";
	exit (0);
      }
      value_data.dsize = strlen(value_data.dptr);
      
      // store the value
      if (gdbm_store (dbf, key_data, value_data, GDBM_REPLACE) < 0) {
	cerr << "gdbm_store returned an error\n";
	exit (0);
      }
      
      delete key_data.dptr;
      delete value_data.dptr;
    }
  }
  
  gdbm_close (dbf);
  
  return 0;
}
