/**********************************************************************
 *
 * recptconfig.cpp -- 
 * Copyright (C) 1999  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

#include "recptconfig.h"
#include "fileutil.h"
#include "cfgread.h"
#include "cnfgator.h"


#if defined(GSDL_USE_OBJECTSPACE)
#  include <ospace/std/iostream>
#  include <ospace/std/fstream>
#elif defined(GSDL_USE_IOS_H)
#  include <iostream.h>
#  include <fstream.h>
#else
#  include <iostream>
#  include <fstream>
#endif

class __site_configuration : public configurable {
private:
  text_t *gsdlhome;
  text_t *httpdomain;
  text_t *httpprefix;
  int    *maxrequests;
public:
  __site_configuration (text_t *_gsdlhome, int *_maxrequests) {
    gsdlhome = _gsdlhome;
    httpdomain = NULL;
    httpprefix = NULL;
    maxrequests = _maxrequests;
  }

  __site_configuration (text_t *_gsdlhome, text_t *_httpdomain,
                        text_t *_httpprefix) {
    gsdlhome    = _gsdlhome;
    httpdomain = _httpdomain;
    httpprefix = _httpprefix;
    maxrequests = NULL;
  }

  inline void configure (const text_t &key, const text_tarray &cfgline) {
    if (key == "gsdlhome") {
      *gsdlhome = cfgline[0];
    }

    if (key == "httpprefix" &&
	httpprefix != NULL) {
      *httpprefix = cfgline[0];
    }

    if (key == "httpdomain" &&
	httpdomain != NULL) {
      *httpdomain = cfgline[0];
    }

    if (key == "maxrequests" &&
	maxrequests != NULL) {
      *maxrequests = cfgline[0].getint();
      if ((*maxrequests) < 1) {
	*maxrequests = 1;
      }
    }
  }
};


// reads site configuration file returning true on success
// also sets gsdlhome and maxrequests 
// gsdlsite.cfg should be in same directory as library executable
bool site_cfg_read (receptionist &recpt, text_t &gsdlhome, int &maxrequests) {
  // initialise configurator, etc.
  __site_configuration sitecfg (&gsdlhome, &maxrequests);
  configurator         gsdlconfigurator(&recpt);

  gsdlconfigurator.add_configurable(&sitecfg);

  // blank the gsdl home text
  gsdlhome.clear();

  if (gsdlconfigurator.configure("gsdlsite.cfg")) {
    return true;
  }
  return false;
}

// this version grabs gsdlhome, httpdomain and httpprefix, 
// returns false if it can't find all of them
bool site_cfg_read (text_t &gsdlhome, text_t &httpdomain, 
		    text_t &httpprefix) {
  // get gsdlhome etc
  __site_configuration sitecfg(&gsdlhome, &httpdomain, &httpprefix);
  configurator gsdlconfigurator(&sitecfg);

  gsdlhome.clear();
  httpdomain.clear();
  httpprefix.clear();

  if (gsdlconfigurator.configure("gsdlsite.cfg") &&
      !gsdlhome.empty() && !httpdomain.empty() && !httpprefix.empty()) {
    return true;
  }

  return true;

  return false;
}

// main_cfg_read reads both main.cfg and collect.cfg. It attempts
// to read main.cfg first so values in collect.cfg override those
// set earlier by main.cfg
bool main_cfg_read (receptionist &recpt, const text_t &gsdlhome,
		     const text_t &collection) {

  // read in the main configuration file
  bool rv = false;

  text_t key, filename;
  text_tarray cfgline;
  filename = filename_cat (gsdlhome, "etc");
  filename = filename_cat (filename, "main.cfg");
  if (file_exists (filename)) {
    rv = recpt.read_configfile(filename);
  }
  
  // Look for collect.cfg in GSDLHOME/collect/collection-name/etc directory 
  // (if this is for a particular collection), and then GSDLHOME/etc.
  if (!collection.empty()) {
    filename = filename_cat (gsdlhome, "collect");
    filename = filename_cat (filename, collection);
    filename = filename_cat (filename, "etc");
    filename = filename_cat (filename, "collect.cfg");
    if (!file_exists (filename)) {
      filename = filename_cat (gsdlhome, "etc");
      filename = filename_cat (filename, "collect.cfg");
      if (!file_exists (filename)) filename.clear();
    }

    if (!filename.empty()) {
      rv |= recpt.read_configfile(filename);
    }
  }
  return rv;
}
