/**************************************************************************
 *
 * IndexData.cpp -- Information needed for querying
 * Copyright (C) 1999 Rodger McNab
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: IndexData.cpp,v 1.1 2000/01/14 02:17:39 sjboddie Exp $
 *
 **************************************************************************/

#include "IndexData.h"
#include "string.h"
#include "mg_files.h"

IndexData::IndexData () {
  basePath[0] = '\0';
  filename[0] = '\0';

  dictFile = NULL;

  stem1File = NULL;
  stem2File = NULL;
  stem3File = NULL;

  invfFile = NULL;

  approxWeightsFile = NULL;
  exactWeightsFile = NULL;

  curLevelNum = 0;
}

IndexData::~IndexData () {
  UnloadData ();
}

bool IndexData::LoadData (const char *_basePath, const char *_filename) {
  if (_filename[0] == '\0') return false;
  
  // make sure this data has not already been loaded
  if (strcmp (_basePath, basePath) == 0 &&
      strcmp (_filename, filename) == 0) return true;
  
  // make sure there is nothing else loaded
  UnloadData ();

  strcpy (basePath, _basePath);
  strcpy (filename, _filename);

  set_basepath (basePath);

  // load in the level information
  FILE *ivfLevelFile;
  ivfLevelFile = open_file (filename, INVF_LEVEL_SUFFIX, "rb",
			    MAGIC_INVF_LEVELS, MG_ABORT);
  levels.Read (ivfLevelFile);
  fclose (ivfLevelFile);

  // blocked dictionary
  dictFile = open_file (filename, INVF_DICT_BLOCKED_SUFFIX, "rb",
			MAGIC_STEM, MG_ABORT);
  if (!bdh.Read (dictFile)) { UnloadData (); return false; }

  fseek (dictFile, bdh.wblk_idx_start, SEEK_SET);
  if (!ReadBlockIdx (dictFile, biWords)) { UnloadData (); return false; }

  fseek (dictFile, bdh.tblk_idx_start, SEEK_SET);
  if (!ReadBlockIdx (dictFile, biTags)) { UnloadData (); return false; }

  // blocked stem index 1
  stem1File = open_file (filename, INVF_DICT_BLOCKED_1_SUFFIX,
			 "rb", MAGIC_STEM_1, MG_ABORT);
  if (!sih1.Read (stem1File)) { UnloadData (); return false; }

  fseek (stem1File, sih1.block_idx_start, SEEK_SET);
  if (!ReadBlockIdx (stem1File, sii1)) { UnloadData (); return false; }
  
  // blocked stem index 2
  stem2File = open_file (filename, INVF_DICT_BLOCKED_2_SUFFIX,
			 "rb", MAGIC_STEM_2, MG_ABORT);
  if (!sih2.Read (stem2File)) { UnloadData (); return false; }

  fseek (stem2File, sih2.block_idx_start, SEEK_SET);
  if (!ReadBlockIdx (stem2File, sii2)) { UnloadData (); return false; }
  
  // blocked stem index 3
  stem3File = open_file (filename, INVF_DICT_BLOCKED_3_SUFFIX,
			 "rb", MAGIC_STEM_3, MG_ABORT);
  if (!sih3.Read (stem3File)) { UnloadData (); return false; }

  fseek (stem3File, sih3.block_idx_start, SEEK_SET);
  if (!ReadBlockIdx (stem3File, sii3)) { UnloadData (); return false; }

  // inverted file
  invfFile = open_file (filename, INVF_SUFFIX, "rb",
			MAGIC_INVF, MG_ABORT);
  ifh.Read (invfFile);

  // weights
  approxWeightsFile = open_file (filename, APPROX_WEIGHTS_SUFFIX, "rb",
				 MAGIC_WGHT_APPROX, MG_ABORT);
  exactWeightsFile = open_file (filename, WEIGHTS_SUFFIX, "rb",
				MAGIC_WGHT, MG_ABORT);

  return true;
}

bool IndexData::UnloadData () {
  basePath[0] = '\0';
  filename[0] = '\0';

  if (dictFile != NULL) {
    fclose (dictFile); dictFile = NULL;
  }

  if (stem1File != NULL) {
    fclose (stem1File); stem1File = NULL;
  }
  if (stem2File != NULL) {
    fclose (stem2File); stem2File = NULL;
  }
  if (stem3File != NULL) {
    fclose (stem3File); stem3File = NULL;
  }

  if (invfFile != NULL) {
    fclose (invfFile); invfFile = NULL;
  }

  if (approxWeightsFile != NULL) {
    fclose (approxWeightsFile); approxWeightsFile = NULL;
  }
  if (exactWeightsFile != NULL) {
    fclose (exactWeightsFile); exactWeightsFile = NULL;
  }

  UnloadLevel();
  
  return true;
}


bool IndexData::LoadLevel (const UCArray &level) {
  // make sure this level is not already loaded
  if (level == curLevel) return true;

  // unload any levels currently loaded
  UnloadLevel();

  // make sure the required files are open
  if (dictFile == NULL || invfFile == NULL ||
      approxWeightsFile == NULL || exactWeightsFile == NULL)
    return false;
  
  // read in the information from the dictionary
  block_dict_el tagEl;
  unsigned long tagElNum;
  if (!SearchBlockDictEl (dictFile, biTags, bdh.entries_per_tblk,
			  bdh.tag_dict_size, level, tagEl, tagElNum))
    return false;

  // read in the level conversion information
  if (!levelConverter.Read (invfFile, tagEl.invf_ptr,
			    bdh.num_frags, tagEl.frag_occur))
    return false;

  // read in the approximate weights
  if (!weightData.Read (approxWeightsFile,
			levels.levelInfo[level].approxWeightsDiskPtr,
			levels.levelInfo[level].numEntries))
    return false;
  
  // find the level number
  curLevelNum = 0;
  IvfLevelInfoMap::const_iterator levelHere, levelEnd;
  for (levelHere=levels.levelInfo.begin(), levelEnd=levels.levelInfo.end();
       levelHere!=levelEnd && (*levelHere).first != level; levelHere++)
    curLevelNum++;

  // make sure we found the level
  if (levelHere == levelEnd) return false;

  curLevel = level;

  return true;
}

bool IndexData::UnloadLevel () {
  curLevel.erase (curLevel.begin(), curLevel.end());
  curLevelNum = 0;
  
  weightData.Free ();
    
  return true;
}

