###########################################################################
#
# AZCompactList.pm --
# A component of the Greenstone digital library software
# from the New Zealand Digital Library Project at the 
# University of Waikato, New Zealand.
#
# Copyright (C) 1999 New Zealand Digital Library Project
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###########################################################################

# classifier plugin for sorting alphabetically
# options are:
#
# metadata=Metaname -- all documents with Metaname metadata
#                      will be included in list, list will be sorted
#                      by this element.
# buttonname=Title  -- (optional) the title field for this classification.
#                      if not included title field will be Metaname. 
# mingroup=Num      -- (optional) the smallest value that will cause
#                      a group in the hierarchy to form.
# minnesting=Num    -- (optional) the smallest value that will cause a 
#		       list to converted into nested list
# mincompact=Num    -- (optional) used in compact list
# maxcompact=Num    -- (optional) used in compact list
# doclevel=top|section -- (optional) level to process document at.
# onlyfirst         -- (optional) control whether all or only first
#                      metadata value used from array of metadata
package AZCompactList;

use BasClas;
use sorttools;

sub BEGIN {
    @ISA = ('BasClas');
}

sub print_usage {
    print STDERR "
  usage: classify AZCompactList [options]
  options:
   -metadata     Metadata field used for classification
   -buttonname   Title to use on web pages

   -mingroup
   -minnesting
   -mincompact
   -maxcompact
   -doclevel
   -onlyfirst
   -recopt
";
}

sub new {
    my $class = shift (@_);
    my $self = new BasClas($class, @_);

    my ($metaname, $title);
    my $mingroup = 2;
    my $minnesting = 20;
    my $mincompact = 10;
    my $maxcompact = 30;
    my $doclevel = "top";
    my $onlyfirst = 0;
    my $recopt   = undef;

     if (!parsargv::parse(\@_, 
			 q^metadata/.*/^, \$metaname,
			 q^buttonname/.*/^, \$title,

			 q^mingroup/.*/2^, \$mingroup,
			 q^minnesting/.*/20^, \$minnesting,
			 q^mincompact/.*/10^, \$mincompact,
			 q^maxcompact/.*/30^, \$maxcompact,
			 q^doclevel/.*/top^, \$doclevel,
			 q^onlyfirst/.*/0^, \$onlyfirst,
			 q^recopt/.*/-1^, \$recopt,

			 "allow_extra_options")) {
	
	print STDERR "\nIncorrect options passed to $class, check your collect.cfg file\n";
	&print_usage();
	die "\n";
    }

# These are the old-fashioned optins to AZCompactList.
# They should be deleted once we're sure the parsearg version is working. 
#    foreach $option (@options) {
#	if ($option =~ /^metadata=(.*)$/i) {
#	    $metaname = $1;
#	} elsif ($option =~ /^title=(.*)$/i) {
#	    $title = $1;
#	} elsif ($option =~ /^mingroup(size)?=(.*)$/i) {
#	    $mingroup = $2;
#	} elsif ($option =~ /^minnesting=(.*)$/i) {
#	    $minnesting = $1;
#	} elsif ($option =~ /^mincompact=(.*)$/i) {
#	    $mincompact = $1;
#	} elsif ($option =~ /^maxcompact=(.*)$/i) {
#	    $maxcompact = $1;
#	} elsif ($option =~ /^doclevel=(.*)$/i) {
#	    $doclevel = $1;
#	} elsif ($option =~ /^onlyfirst$/i) {
#	    $onlyfirst = 1;
#	} elsif ($option =~ /^recopt$/i) {
#	    $recopt = "on";
#	}
#    }

    if (!defined $metaname) {
	my $outhandle = $self->{'outhandle'};
	print $outhandle "AZCompactList used with no metadata name to classify by\n";
	die "\n";
    }

    $title = $metaname unless ($title);

    $self->{'list'} = {};
    $self->{'listmetavalue'} = {};
    $self->{'reclassify'} = {};
    $self->{'reclassifylist'} = {};
    $self->{'metaname'} = $metaname;
    $self->{'title'} = $title;
    $self->{'mingroup'} = $mingroup;
    $self->{'minnesting'} = $minnesting;
    $self->{'mincompact'} = $mincompact;
    $self->{'maxcompact'} = $maxcompact;
    $self->{'doclevel'} = $doclevel;

    if ($onlyfirst != 0) {
	$onlyfirst = 1;
    }
    $self->{'onlyfirst'} = $onlyfirst;
  
    if ($recopt == -1) {
	$recopt = undef; 
    } else {
	$recopt = "on";
    }
    $self->{'recopt'} = $recopt;

    return bless $self, $class;
}

sub init 
{
    my $self = shift (@_);
    
    $self->{'list'} = {};
    $self->{'listmetavalue'} = {};
    $self->{'reclassify'} = {};
    $self->{'reclassifylist'} = {};
}

$tmp = 0;

sub classify 
{
    my $self = shift (@_);
    my ($doc_obj) = @_;

    my $doc_OID = $doc_obj->get_OID();

    my @sectionlist = ();
    my $topsection = $doc_obj->get_top_section();

    my $metaname = $self->{'metaname'};
    my $outhandle = $self->{'outhandle'};

    $metaname =~ s/(\/.*)//; # grab first name in n1/n2/n3 list

    if ($self->{'doclevel'} =~ /^top(level)?/i)
    {
	push(@sectionlist,$topsection);
    }
    else
    {
	my $thissection = $doc_obj->get_next_section($topsection);
	while (defined $thissection) 
	{
	    push(@sectionlist,$thissection);
	    $thissection = $doc_obj->get_next_section ($thissection);
	}
    }

    my $thissection;
    foreach $thissection (@sectionlist)
    {
	my $full_doc_OID 
	    = ($thissection ne "") ? "$doc_OID.$thissection" : $doc_OID;

	if (defined $self->{'list'}->{$full_doc_OID}) 
	{
	    print $outhandle "WARNING: AZCompactList::classify called multiple times for $full_doc_OID\n";
	} 
	$self->{'list'}->{$full_doc_OID} = [];	
	$self->{'listmetavalue'}->{$full_doc_OID} = [];

	my $metavalues = $doc_obj->get_metadata($thissection,$metaname);
	my $metavalue;
	foreach $metavalue (@$metavalues) 
	{
###	    print STDERR "$metaname :\tmetavalue = $metavalue\n";
	    # if this document doesn't contain the metadata element we're
	    # sorting by we won't include it in this classification
	    if (defined $metavalue && $metavalue =~ /\w/) 
	    {
		my $formatted_metavalue = $metavalue;
		if ($self->{'metaname'} =~ m/^Creator(:.*)?$/)
		{
		    &sorttools::format_string_name_english (\$formatted_metavalue);
		} 
		else 
		{
		    &sorttools::format_string_english (\$formatted_metavalue);
		}
		
		push(@{$self->{'list'}->{$full_doc_OID}},$formatted_metavalue);
		push(@{$self->{'listmetavalue'}->{$full_doc_OID}} ,$metavalue);

		last if ($self->{'onlyfirst'});
	    }
	}
	my $date = $doc_obj->get_metadata_element($thissection,"Date");
	$self->{'reclassify'}->{$full_doc_OID} = [$doc_obj,$date];
    }
}

sub reinit 
{
    my ($self,$classlist_ref) = @_;
    
    my %mtfreq = ();
    my @single_classlist = ();
    my @multiple_classlist = ();

    # find out how often each metavalue occurs
    map 
    { 
	my $mv;
	foreach $mv (@{$self->{'listmetavalue'}->{$_}} )
	{
	    $mtfreq{$mv}++; 
	}
    } @$classlist_ref;

    # use this information to split the list: single metavalue/repeated value
    map
    { 
	my $i = 1;
	my $metavalue;
	foreach $metavalue (@{$self->{'listmetavalue'}->{$_}})
	{
	    if ($mtfreq{$metavalue} >= $self->{'mingroup'})
	    {
		push(@multiple_classlist,[$_,$i,$metavalue]); 
	    } 
	    else
	    {
		push(@single_classlist,[$_,$metavalue]); 
		$metavalue =~ tr/[A-Z]/[a-z]/;
		$self->{'reclassifylist'}->{"Metavalue_$i.$_"} = $metavalue;
	    }
	    $i++;
	}
    } @$classlist_ref;
    
    
    # Setup sub-classifiers for multiple list

    $self->{'classifiers'} = {};

    my $pm;
    foreach $pm ("List", "SectionList")
    {
	my $listname 
	    = &util::filename_cat($ENV{'GSDLHOME'},"perllib/classify/$pm.pm");
	if (-e $listname) { require $listname; }
	else 
	{ 
	    my $outhandle = $self->{'outhandle'};
	    print $outhandle "AZCompactList ERROR - couldn't find classifier \"$listname\"\n"; 
	    die "\n";
	}
    }

    # Create classifiers objects for each entry >= mingroup
    my $metavalue;
    foreach $metavalue (keys %mtfreq)
    {
	if ($mtfreq{$metavalue} >= $self->{'mingroup'})
	{
	    my $listclassobj;
	    my $doclevel = $self->{'doclevel'};
	    my $metaname  = $self->{'metaname'};
	    my @metaname_list = split('/',$metaname);
	    $metaname = shift(@metaname_list);
	    if (@metaname_list==0)
	    {
		my @args;
		push @args, ("-metadata", "$metaname");
		push @args, ("-title", "$metavalue");
		push @args, ("-sort", "Date");

		if ($doclevel =~ m/^top(level)?/i)
		{
		    eval ("\$listclassobj = new List(\@args)"); warn $@ if $@;
		}
		else
		{
		    eval ("\$listclassobj = new SectionList($args)");
		}
	    }
	    else
	    {
		$metaname = join('/',@metaname_list);
		
		my @args;
		push @args, ("-metadata", "$metaname");
		push @args, ("-title", "$metavalue");
		push @args, ("-doclevel", "$doclevel");
		push @args, "-recopt";

		eval ("\$listclassobj = new AZCompactList($args)");
	    }
	    if ($@) {
		my $outhandle = $self->{'outhandle'};
		print $outhandle "$@";
		die "\n";
	    }
	    
	    $listclassobj->init();

	    if (defined $metavalue && $metavalue =~ /\w/) 
	    {
		my $formatted_node = $metavalue;
		if ($self->{'metaname'} =~ m/^Creator(:.*)?$/)
		{
		    &sorttools::format_string_name_english(\$formatted_node);
		} 
		else 
		{
		    &sorttools::format_string_english(\$formatted_node);
		}
		
		$self->{'classifiers'}->{$metavalue} 
		= { 'classifyobj'   => $listclassobj,
		    'formattednode' => $formatted_node };
	    }
	}
    }


    return (\@single_classlist,\@multiple_classlist);
}


sub reclassify 
{
    my ($self,$multiple_cl_ref) = @_;

    # Entries in the current classify list that are "book nodes"
    # should be recursively classified.
    #--
    foreach $dm_pair (@$multiple_cl_ref) 
    {
	my ($doc_OID,$mdoffset,$metavalue) = @$dm_pair;
        my $listclassobj;

	# find metavalue in list of sub-classifiers
	my $found = 0;
	my $node_name;
	foreach $node_name (keys %{$self->{'classifiers'}}) 
	{
	    $resafe_node_name = $node_name;
	    $resafe_node_name =~ s/(\(|\)|\[|\]|\{|\}|\^|\$|\.|\+|\*|\?|\|)/\\$1/g;
	    if ($metavalue =~ m/^$resafe_node_name$/i) 
	    {
		my ($doc_obj,$date) = @{$self->{'reclassify'}->{$doc_OID}};

		## date appears to not be used in classifier call ####

		if ($doc_OID =~ m/^.*\.(\d+)$/)
		{
		    $self->{'classifiers'}->{$node_name}->{'classifyobj'}
		    ->classify($doc_obj, "Section=$1"); 
		}
		else
		{
		    $self->{'classifiers'}->{$node_name}->{'classifyobj'}
		    ->classify($doc_obj); 
		}
		
		$found = 1;
		last;
	    }
	}
	
	if (!$found)
	{
	    print $outhandle "Warning: AZCompactList::reclassify ";
	    print $outhandle "could not find sub-node for $metavalue with doc_OID $doc_OID\n";
	}
    }
}



sub get_reclassify_info 
{
    my $self = shift (@_);
    
    my $node_name;
    foreach $node_name (keys %{$self->{'classifiers'}}) 
    {
        my $classifyinfo 
	    = $self->{'classifiers'}->{$node_name}->{'classifyobj'}
	        ->get_classify_info();
        $self->{'classifiers'}->{$node_name}->{'classifyinfo'} 
	    = $classifyinfo;
        $self->{'reclassifylist'}->{"CLASSIFY.$node_name"} 
	    = $self->{'classifiers'}->{$node_name}->{'formattednode'};
    }
}


sub alpha_numeric_cmp
{
    my ($self,$a,$b) = @_;

    my $title_a = $self->{'reclassifylist'}->{$a};
    my $title_b = $self->{'reclassifylist'}->{$b};

    if ($title_a =~ m/^(\d+(\.\d+)?)/)
    {
	my $val_a = $1;
	if ($title_b =~ m/^(\d+(\.\d+)?)/)
	{
	    my $val_b = $1;
	    if ($val_a != $val_b)
	    {
		return ($val_a <=> $val_b);
	    }
	}
    }
    
    return ($title_a cmp $title_b);
}

sub get_classify_info {
    my $self = shift (@_);

    my @classlist =keys %{$self->{'list'}}; # list all doc oids

    my ($single_cl_ref,$multiple_cl_ref) = $self->reinit(\@classlist);
    $self->reclassify($multiple_cl_ref);
    $self->get_reclassify_info();


#    my @reclassified_classlist 
#	= sort {$self->{'reclassifylist'}->{$a} cmp $self->{'reclassifylist'}->{$b};} keys %{$self->{'reclassifylist'}};

    # alpha_numeric_cmp is slower but handles numbers better ...
    my @reclassified_classlist 
	= sort { $self->alpha_numeric_cmp($a,$b) } keys %{$self->{'reclassifylist'}};


    return $self->splitlist (\@reclassified_classlist);
}

sub get_entry {
    my $self = shift (@_);
    my ($title, $childtype, $metaname, $thistype) = @_;

    # organise into classification structure
    my %classifyinfo = ('childtype'=>$childtype,
                        'Title'=>$title,
                        'contains'=>[],
			'mdtype'=>$metaname);

    $classifyinfo{'thistype'} = $thistype
        if defined $thistype && $thistype =~ /\w/;

    return \%classifyinfo;
}



# splitlist takes an ordered list of classifications (@$classlistref) and 
# splits it up into alphabetical sub-sections.
sub splitlist {
    my $self = shift (@_);
    my ($classlistref) = @_;
    my $classhash = {};

    # top level
    my @metanames = split("/",$self->{'metaname'});
    my $metaname = shift(@metanames);

    my $childtype = "HList";
    $childtype = "VList" if (scalar (@$classlistref) <= $self->{'minnesting'});

    my $classifyinfo;
    if (!defined($self->{'recopt'}))
    {
	my $title = $self->{'title'}; # should always be defined by now....
	$title = $metaname unless defined $title; 
	$classifyinfo 
	    = $self->get_entry ($title, $childtype, $metaname, "Invisible");
    }
    else
    {
	my $title = $self->{'title'};
	$classifyinfo 
	    = $self->get_entry ($title, $childtype, $metaname, "VList");
    }

    # don't need to do any splitting if there are less than 'minnesting' classifications
    if ((scalar @$classlistref) <= $self->{'minnesting'}) {
	foreach $subOID (@$classlistref) {
            if ($subOID =~ /^CLASSIFY\.(.*)$/ 
		&& defined $self->{'classifiers'}->{$1}) 
	    {
###		print STDERR "*** subOID = $subOID\n";

                push (@{$classifyinfo->{'contains'}}, 
		      $self->{'classifiers'}->{$1}->{'classifyinfo'});
            } 
	    else 
	    {
		$subOID =~ s/^Metavalue_(\d+)\.//;
		my $metaname_offset = $1 -1;
		my $oid_rec = {'OID'=>$subOID, 'offset'=>$metaname_offset};
		push (@{$classifyinfo->{'contains'}}, $oid_rec);
	    }
	}
	return $classifyinfo;
    }
	
    # first split up the list into separate A-Z and 0-9 classifications
    foreach $classification (@$classlistref) {
	my $title = $self->{'reclassifylist'}->{$classification};
	$title =~ s/&(.){2,4};//g; # remove any HTML special chars
###	$title =~ s/^\s+//g; # remove a leading spaces
###	$title =~ s/^_+//g; # remove a leading underscores
	$title =~ s/^\W+//g;
###	$title =~ s/^(\'|\`|\"|\:|\()//g; # remove any opening punctutation
	$title =~ s/^(.).*$/$1/;
	$title =~ tr/[a-z]/[A-Z]/;

	if ($title =~ /^[0-9]$/) {$title = '0-9';}
	elsif ($title !~ /^[A-Z]$/) {
	    my $outhandle = $self->{'outhandle'};
	    print $outhandle "AZCompactList: WARNING $classification has badly formatted title ($title)\n";
	}
	$classhash->{$title} = [] unless defined $classhash->{$title};
	push (@{$classhash->{$title}}, $classification);
    }
    $classhash = $self->compactlist ($classhash);

    my @tmparr = ();
    foreach $subsection (sort keys (%$classhash)) {
	push (@tmparr, $subsection);
    }
    
    # if there's a 0-9 section it will have been sorted to the beginning
    # but we want it at the end
    if ($tmparr[0] eq '0-9') {
	shift @tmparr;
	push (@tmparr, '0-9');
    }

    foreach $subclass (@tmparr) 
    {
	my $tempclassify = $self->get_entry($subclass, "VList", $metaname);
	foreach $subsubOID (@{$classhash->{$subclass}}) 
	{
            if ($subsubOID =~ /^CLASSIFY\.(.*)$/ 
		&& defined $self->{'classifiers'}->{$1}) 
	    {
                push (@{$tempclassify->{'contains'}}, 
		      $self->{'classifiers'}->{$1}->{'classifyinfo'});
	    }
	    else
	    {
		$subsubOID =~ s/^Metavalue_(\d+)\.//;
		my $metaname_offset = $1 -1;
		my $oid_rec = {'OID'=>$subsubOID, 'offset'=>$metaname_offset};
		push (@{$tempclassify->{'contains'}}, $oid_rec);
	    }
	}
	push (@{$classifyinfo->{'contains'}}, $tempclassify);
    }

    return $classifyinfo;
}

sub compactlist {
    my $self = shift (@_);
    my ($classhashref) = @_;
    my $compactedhash = {};
    my @currentOIDs = ();
    my $currentfirstletter = "";
    my $currentlastletter = "";
    my $lastkey = "";

    # minimum and maximum documents to be displayed per page.
    # the actual maximum will be max + (min-1).
    # the smallest sub-section is a single letter at present
    # so in this case there may be many times max documents
    # displayed on a page.
    my $min = $self->{'mincompact'}; 
    my $max = $self->{'maxcompact'};

    foreach $subsection (sort keys %$classhashref) {
	if ($subsection eq '0-9') {
	    @{$compactedhash->{$subsection}} = @{$classhashref->{$subsection}};
	    next;
	}
	$currentfirstletter = $subsection if $currentfirstletter eq "";
	if ((scalar (@currentOIDs) < $min) ||
	    ((scalar (@currentOIDs) + scalar (@{$classhashref->{$subsection}})) <= $max)) {
	    push (@currentOIDs, @{$classhashref->{$subsection}});
	    $currentlastletter = $subsection;
	} else {

	    if ($currentfirstletter eq $currentlastletter) {
		@{$compactedhash->{$currentfirstletter}} = @currentOIDs;
		$lastkey = $currentfirstletter;
	    } else {
		@{$compactedhash->{"$currentfirstletter-$currentlastletter"}} = @currentOIDs;
		$lastkey = "$currentfirstletter-$currentlastletter";
	    } 
	    if (scalar (@{$classhashref->{$subsection}}) >= $max) {
		$compactedhash->{$subsection} = $classhashref->{$subsection};
		@currentOIDs = ();
		$currentfirstletter = "";
	    } else {
		@currentOIDs = @{$classhashref->{$subsection}};
		$currentfirstletter = $subsection;
		$currentlastletter = $subsection;
	    }
	}
    }

    # add final OIDs to last sub-classification if there aren't many otherwise
    # add final sub-classification
    if (scalar (@currentOIDs) < $min) {
	my ($newkey) = $lastkey =~ /^(.)/;
	@currentOIDs = (@{$compactedhash->{$lastkey}}, @currentOIDs);
	delete $compactedhash->{$lastkey};
	@{$compactedhash->{"$newkey-$currentlastletter"}} = @currentOIDs;
    } else {
	if ($currentfirstletter eq $currentlastletter) {
	    @{$compactedhash->{$currentfirstletter}} = @currentOIDs;
	} else {
	    @{$compactedhash->{"$currentfirstletter-$currentlastletter"}} = @currentOIDs;
	} 
    }

    return $compactedhash;
}

1;


