/*
 * Copyright (c) 1995-2000, Index Data.
 *
 * Permission to use, copy, modify, distribute, and sell this software and
 * its documentation, in whole or in part, for any purpose, is hereby granted,
 * provided that:
 *
 * 1. This copyright and permission notice appear in all copies of the
 * software and its documentation. Notices of copyright or attribution
 * which appear at the beginning of any file must remain unchanged.
 *
 * 2. The name of Index Data or the individual authors may not be used to
 * endorse or promote products derived from this software without specific
 * prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS, IMPLIED, OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.
 * IN NO EVENT SHALL INDEX DATA BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR
 * NOT ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF
 * LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
 * OF THIS SOFTWARE.
 *
 * $Id: odr.h,v 1.1 2000/08/03 03:10:40 johnmcp Exp $
 */

#ifndef ODR_H
#define ODR_H

#include <stdio.h>
#include <string.h>

#include <yaz/yconfig.h>
#include <yaz/nmem.h>

YAZ_BEGIN_CDECL

#ifndef bool_t
#define bool_t int
#endif

/*
 * Tag modes
 */
#define ODR_NONE -1
#define ODR_IMPLICIT 0
#define ODR_EXPLICIT 1

/*
 * Classes
 */
#define ODR_UNIVERSAL   0
#define ODR_APPLICATION 1
#define ODR_CONTEXT     2
#define ODR_PRIVATE     3

/*
 * UNIVERSAL tags
 */
#define ODR_BOOLEAN     1
#define ODR_INTEGER     2
#define ODR_BITSTRING   3
#define ODR_OCTETSTRING 4
#define ODR_NULL        5
#define ODR_OID         6
#define ODR_ODESC       7
#define ODR_EXTERNAL    8
#define ODR_REAL        9
#define ODR_ENUM        10
#define ODR_SEQUENCE    16
#define ODR_SET         17
#define ODR_NUMERICSTRING   18
#define ODR_PRINTABLESTRING 19
#define ODR_GENERALIZEDTIME 24
#define ODR_GRAPHICSTRING   25
#define ODR_VISIBLESTRING   26
#define ODR_GENERALSTRING   27

/*
 * odr stream directions
 */
#define ODR_DECODE      0
#define ODR_ENCODE      1
#define ODR_PRINT       2

typedef struct odr_oct
{
    unsigned char *buf;
    int len;
    int size;
} Odr_oct;

typedef void Odr_null;
extern Odr_null *ODR_NULLVAL;

typedef Odr_oct Odr_any;

typedef struct odr_bitmask
{
#define ODR_BITMASK_SIZE 256
    unsigned char bits[ODR_BITMASK_SIZE];
    int top;
} Odr_bitmask;

typedef int Odr_oid;   /* terminate by -1 */

typedef struct odr_constack
{
    const unsigned char *base;   /* starting point of data */
    int base_offset;
    int len;                     /* length of data, if known, else -1
                                        (decoding only) */
    const unsigned char *lenb;   /* where to encode length */
    int len_offset;
    int lenlen;                  /* length of length-field */
} odr_constack;

#define ODR_S_SET     0
#define ODR_S_CUR     1
#define ODR_S_END     2

typedef struct {      /* used to be statics in ber_tag... */
    int lclass;
    int ltag;
    int br;
    int lcons;
} Odr_ber_tag;

typedef struct odr
{
    int direction;       /* the direction of this stream */

    int error;           /* current error state (0==OK) */

    int can_grow;         /* are we allowed to reallocate */
    unsigned char *buf;            /* memory handle */
    int size;             /* current buffer size */

    int pos;              /* current position */
    int top;              /* top of buffer (max pos when decoding) */

    const unsigned char *bp; /* position in buffer (decoding) */

    int t_class;         /* implicit tagging (-1==default tag) */
    int t_tag;

    int enable_bias;     /* force choice enable flag */
    int choice_bias;     /* force choice */
    int lenlen;          /* force length-of-lenght (odr_setlen()) */

    FILE *print;         /* output file for direction print */
    int indent;          /* current indent level for printing */

    NMEM mem;            /* memory handle for decoding (primarily) */

    /* stack for constructed types */
#define ODR_MAX_STACK 50
    int stackp;          /* top of stack (-1 == initial state) */
    odr_constack stack[ODR_MAX_STACK];

    Odr_ber_tag odr_ber_tag;
} *ODR;

typedef int (*Odr_fun)(ODR, char **, int, const char *);

typedef struct odr_arm
{
    int tagmode;
    int zclass;
    int tag;
    int which;
    Odr_fun fun;
    char *name;
} Odr_arm;

/*
 * Error control.
 */
#define ONONE           0
#define OMEMORY         1
#define OSYSERR         2
#define OSPACE          3
#define OREQUIRED       4
#define OUNEXPECTED     5
#define OOTHER          6
#define OPROTO          7
#define ODATA           8
#define OSTACK          9
#define OCONLEN        10
#define OLENOV         11

extern char *odr_errlist[];

YAZ_EXPORT int odr_geterror(ODR o);
YAZ_EXPORT void odr_perror(ODR o, char *message);
YAZ_EXPORT void odr_setprint(ODR o, FILE *file);
YAZ_EXPORT ODR odr_createmem(int direction);
YAZ_EXPORT void odr_reset(ODR o);
YAZ_EXPORT void odr_destroy(ODR o);
YAZ_EXPORT void odr_setbuf(ODR o, char *buf, int len, int can_grow);
YAZ_EXPORT char *odr_getbuf(ODR o, int *len, int *size);
YAZ_EXPORT void *odr_malloc(ODR o, int size);
YAZ_EXPORT char *odr_strdup(ODR o, const char *str);
YAZ_EXPORT NMEM odr_extract_mem(ODR o);
YAZ_EXPORT Odr_null *odr_nullval(void);
#define odr_release_mem(m) nmem_destroy(m)
#define ODR_MEM NMEM

#define odr_implicit(o, t, p, cl, tg, opt)\
        (odr_implicit_settag((o), cl, tg), t ((o), (p), (opt), 0) )

#define odr_implicit_tag(o, t, p, cl, tg, opt, name)\
        (odr_implicit_settag((o), cl, tg), t ((o), (p), (opt), name) )

#define odr_explicit(o, t, p, cl, tg, opt)\
        ((int) (odr_constructed_begin((o), (p), (cl), (tg), 0) ? \
        t ((o), (p), (opt), 0) &&\
        odr_constructed_end(o) : opt))

#define odr_explicit_tag(o, t, p, cl, tg, opt, name)\
        ((int) (odr_constructed_begin((o), (p), (cl), (tg), 0) ? \
        t ((o), (p), (opt), name) &&\
        odr_constructed_end(o) : opt))

#define ODR_MASK_ZERO(mask)\
    ((void) (memset((mask)->bits, 0, ODR_BITMASK_SIZE),\
    (mask)->top = -1))

#define ODR_MASK_SET(mask, num)\
    (((mask)->bits[(num) >> 3] |= 0X80 >> ((num) & 0X07)),\
    (mask)->top < (num) >> 3 ? ((mask)->top = (num) >> 3) : 0)

#define ODR_MASK_CLEAR(mask, num)\
    ((mask)->bits[(num) >> 3] &= ~(0X80 >> ((num) & 0X07)))

#define ODR_MASK_GET(mask, num)  ( ((num) >> 3 <= (mask)->top) ? \
    ((mask)->bits[(num) >> 3] & (0X80 >> ((num) & 0X07)) ? 1 : 0) : 0)

/* Private macro.
 * write a single character at the current position - grow buffer if
 * necessary.
 * (no, we're not usually this anal about our macros, but this baby is
 *  next to unreadable without some indentation  :)
 */
#define odr_putc(o, c) \
( \
    ( \
        (o)->pos < (o)->size ? \
        ( \
            (o)->buf[(o)->pos++] = (c), \
            0 \
        ) : \
        ( \
            odr_grow_block((o), 1) == 0 ? \
            ( \
                (o)->buf[(o)->pos++] = (c), \
                0 \
            ) : \
            ( \
                (o)->error = OSPACE, \
                -1 \
            ) \
        ) \
    ) == 0 ? \
    ( \
        (o)->pos > (o)->top ? \
        ( \
            (o)->top = (o)->pos, \
            0 \
        ) : \
        0 \
    ) : \
        -1 \
) \

#define odr_tell(o) ((o)->pos)
#define odr_offset(o) ((o)->bp - (o)->buf)
#define odr_ok(o) (!(o)->error)
#define odr_getmem(o) ((o)->mem)
#define odr_setmem(o, v) ((o)->mem = (v))

#define ODR_MAXNAME 256

YAZ_EXPORT int ber_boolean(ODR o, int *val);
YAZ_EXPORT int ber_tag(ODR o, void *p, int zclass, int tag,
		       int *constructed, int opt);
YAZ_EXPORT int ber_enctag(ODR o, int zclass, int tag, int constructed);
YAZ_EXPORT int ber_dectag(const unsigned char *buf, int *zclass,
			  int *tag, int *constructed);
YAZ_EXPORT int odr_bool(ODR o, int **p, int opt, const char *name);
YAZ_EXPORT int odr_integer(ODR o, int **p, int opt, const char *name);
YAZ_EXPORT int odr_enum(ODR o, int **p, int opt, const char *name);
YAZ_EXPORT int odr_implicit_settag(ODR o, int zclass, int tag);
YAZ_EXPORT int ber_enclen(ODR o, int len, int lenlen, int exact);
YAZ_EXPORT int ber_declen(const unsigned char *buf, int *len);
YAZ_EXPORT void odr_prname(ODR o, const char *name);
YAZ_EXPORT int ber_null(ODR o);
YAZ_EXPORT int odr_null(ODR o, Odr_null **p, int opt, const char *name);
YAZ_EXPORT int ber_integer(ODR o, int *val);
YAZ_EXPORT int odr_constructed_begin(ODR o, void *p, int zclass, int tag,
				     const char *name);
YAZ_EXPORT int odr_constructed_end(ODR o);
YAZ_EXPORT int odr_sequence_begin(ODR o, void *p, int size, const char *name);
YAZ_EXPORT int odr_set_begin(ODR o, void *p, int size, const char *name);
YAZ_EXPORT int odr_sequence_end(ODR o);
YAZ_EXPORT int odr_set_end(ODR o);
YAZ_EXPORT int ber_octetstring(ODR o, Odr_oct *p, int cons);
YAZ_EXPORT int odr_octetstring(ODR o, Odr_oct **p, int opt, const char *name);
YAZ_EXPORT int odp_more_chunks(ODR o, const unsigned char *base, int len);
YAZ_EXPORT int odr_constructed_more(ODR o);
YAZ_EXPORT int odr_bitstring(ODR o, Odr_bitmask **p, int opt,
			     const char *name);
YAZ_EXPORT int ber_bitstring(ODR o, Odr_bitmask *p, int cons);
YAZ_EXPORT int odr_generalstring(ODR o, char **p, int opt, const char *name);
YAZ_EXPORT int ber_oidc(ODR o, Odr_oid *p);
YAZ_EXPORT int odr_oid(ODR o, Odr_oid **p, int opt, const char *name);
YAZ_EXPORT int odr_choice(ODR o, Odr_arm arm[], void *p, void *whichp,
			  const char *name);
YAZ_EXPORT int odr_cstring(ODR o, char **p, int opt, const char *name);
YAZ_EXPORT int odr_sequence_of(ODR o, Odr_fun type, void *p, int *num,
			       const char *name);
YAZ_EXPORT int odr_set_of(ODR o, Odr_fun type, void *p, int *num,
			  const char *name);
YAZ_EXPORT int odr_any(ODR o, Odr_any **p, int opt, const char *name);
YAZ_EXPORT int ber_any(ODR o, Odr_any **p);
YAZ_EXPORT int completeBER(const unsigned char *buf, int len);
YAZ_EXPORT void odr_begin(ODR o);
YAZ_EXPORT void odr_end(ODR o);
YAZ_EXPORT Odr_oid *odr_oiddup(ODR odr, Odr_oid *o);
YAZ_EXPORT Odr_oid *odr_oiddup_nmem(NMEM nmem, Odr_oid *o);
YAZ_EXPORT int odr_grow_block(ODR b, int min_bytes);
YAZ_EXPORT int odr_write(ODR o, unsigned char *buf, int bytes);
YAZ_EXPORT int odr_seek(ODR o, int whence, int offset);
YAZ_EXPORT int odr_dumpBER(FILE *f, const char *buf, int len);
YAZ_EXPORT void odr_choice_bias(ODR o, int what);
YAZ_EXPORT void odr_choice_enable_bias(ODR o, int mode);
YAZ_EXPORT int odr_total(ODR o);
YAZ_EXPORT char *odr_errmsg(int n);
YAZ_EXPORT Odr_oid *odr_getoidbystr(ODR o, char *str);
YAZ_EXPORT Odr_oid *odr_getoidbystr_nmem(NMEM o, char *str);
YAZ_EXPORT int odr_initmember(ODR o, void *p, int size);
YAZ_EXPORT int odr_peektag(ODR o, int *zclass, int *tag, int *cons);
YAZ_EXPORT void odr_setlenlen(ODR o, int len);

typedef struct Odr_external
{
    Odr_oid *direct_reference;       /* OPTIONAL */
    int     *indirect_reference;     /* OPTIONAL */
    char    *descriptor;             /* OPTIONAL */
    int which;
#define ODR_EXTERNAL_single 0
#define ODR_EXTERNAL_octet 1
#define ODR_EXTERNAL_arbitrary 2
    union
    {
	Odr_any  *single_ASN1_type;
	Odr_oct  *octet_aligned; 
	Odr_bitmask *arbitrary;      /* we aren't really equipped for this*/
    } u;
} Odr_external;

YAZ_EXPORT int odr_external(ODR o, Odr_external **p, int opt,
			    const char *name);
YAZ_EXPORT int odr_visiblestring(ODR o, char **p, int opt,
				 const char *name);
YAZ_EXPORT int odr_graphicstring(ODR o, char **p, int opt,
				 const char *name);
YAZ_EXPORT int odr_generalizedtime(ODR o, char **p, int opt,
				   const char *name);
YAZ_END_CDECL

#include <yaz/xmalloc.h>

#endif
