/**********************************************************************
 *
 * userdb.h -- functions to handle a user database
 * Copyright (C) 1999  DigiLib Systems Limited, New Zealand
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/


#ifndef USERDB_H
#define USERDB_H

#include "infodbclass.h"


// the password line in the userinfo_t must be encrypted using
// this function
text_t crypt_text (const text_t &text);

// username_ok tests to make sure a username is ok. a username
// must be at least 2 characters long, but no longer than 30
// characters long. it can contain the characters a-z A-Z 0-9
// . and _
bool username_ok (const text_t &username);

// password_ok tests to make sure a password is ok. a password
// must be at least 3 characters long but no longer than 8 characters
// long. it can contain any character in the range 0x20-0x7e
bool password_ok (const text_t &password);


// information about a single user
struct userinfo_t {
  void clear ();
  userinfo_t () {clear();}
  userinfo_t &operator=(const userinfo_t &x);
  
  text_t username;
  text_t password;
  bool enabled;
  text_t groups; // comma separated list
  text_t comment;
};


// these functions deal with user and key databases that are already open
// they should have been opened with gdbmclass::opendatabase
// as soon as the databases are finished with they should be closed with
// gdbmclass::closedatabase so that another process can access the database
//
// some of the functions need the databases opened with write access, an
// appropriate call would be something like:
//
//   success = keydb.opendatabase (keyfile, GDBM_WRCREAT, 1000);
//
// note: passwords and keys are encrypted, so you can't just compare passwords
// in a userinfo_t with a password given by the user. use check_passwd


// functions dealing with user databases

// returns true on success (in which case userinfo will contain
// the information for this user)
bool get_user_info (gdbmclass &userdb, const text_t &username,
		    userinfo_t &userinfo);
bool get_user_info (const text_t &userdbfile, const text_t &username,
		    userinfo_t &userinfo);

// returns true on success
bool set_user_info (gdbmclass &userdb, const text_t &username,
		    const userinfo_t &userinfo);
bool set_user_info (const text_t &userdbfile, const text_t &username,
		    const userinfo_t &userinfo);

// removes a user from the user database -- forever
void delete_user (gdbmclass &userdb, const text_t &username);
void delete_user (const text_t &userdbfile, const text_t &username);

// gets a list of all the users in the database. returns true
// on success
void get_user_list (gdbmclass &userdb, text_tarray &userlist);

// returns true if the user's password is correct.
bool check_passwd (const userinfo_t &thisuser, const text_t &password);


// functions dealing with databases of temporary keys

// generates a random key for the user, stores it in the database and
// returns it so that it can be used in page generation
// returns "" on failure
text_t generate_key (gdbmclass &keydb, const text_t &username);
text_t generate_key (const text_t &keydbfile, const text_t &username);

// checks to see if there is a key for this particular user in the
// database that hasn't decayed. a short decay is used when group
// is set to administrator
bool check_key (gdbmclass &keydb, const userinfo_t &thisuser,
		const text_t &key, const text_t &group, int keydecay);
bool check_key (const text_t &keydbfile, const userinfo_t &thisuser,
		const text_t &key, const text_t &group, int keydecay);

// remove_old_keys will remove all keys created more than keydecay ago.
// use sparingly, it can be quite an expensive function
void remove_old_keys (const text_t &keydbfile, int keydecay);


#endif
