/**********************************************************************
 *
 * statusaction.cpp -- 
 * Copyright (C) 1999  The New Zealand Digital Library Project
 *
 * A component of the Greenstone digital library software
 * from the New Zealand Digital Library Project at the
 * University of Waikato, New Zealand.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *********************************************************************/

#include "statusaction.h"
#include "fileutil.h"
#include "htmlutils.h"
#include "gsdltools.h"
#include <assert.h>


void statusaction::output_frameset (cgiargsclass &/*args*/, displayclass &disp, 
				    outconvertclass &outconvert, 
				    ostream &textout, ostream &/*logout*/) {
  textout << outconvert << disp << "_status:frameset_\n";
}

void statusaction::output_select (cgiargsclass &/*args*/, displayclass &disp, 
				  outconvertclass &outconvert, 
				  ostream &textout, ostream &/*logout*/) {
  textout << outconvert << disp << "_status:header_(selector)\n"
    "_status:select_\n"
    "_status:footer_\n";
}

void statusaction::output_welcome (cgiargsclass &/*args*/, recptprotolistclass *protos, 
				   displayclass &disp, outconvertclass &outconvert, 
				   ostream &textout, ostream &logout) {

  if (recpt == NULL) return;

  textout << outconvert << disp 
	  << "_status:infoheader_(_titlewelcome_)\n"
	  << "_status:welcome_"
	  << "<center><table width=_pagewidth_>\n"
	  << "<th align=left>abbrev.</th><th align=left>collection</th>"
	  << "<th align=left>public?</th><th align=left>running?</th></tr>\n";

  recptprotolistclass::iterator rprotolist_here = protos->begin();
  recptprotolistclass::iterator rprotolist_end = protos->end();
  while (rprotolist_here != rprotolist_end) {
    if ((*rprotolist_here).p != NULL) {
      text_t protoname = (*rprotolist_here).p->get_protocol_name();
      text_tarray collist;
      comerror_t err;
      (*rprotolist_here).p->get_collection_list (collist, err, logout);
      if (err == noError) {
	text_tarray::iterator collist_here = collist.begin();
	text_tarray::iterator collist_end = collist.end();

	while (collist_here != collist_end) {

	    textout << outconvert << disp
		    << "<tr><td><a href=\"_gwcgi_?_optsite_e=_compressedoptions_&a=status&p=collectioninfo&pr="
		    << protoname
		    << "&c="
		    << *collist_here
		    << "\">"
		    << *collist_here 
		    << "</a></td>";

	  ColInfoResponse_t *cinfo = recpt->get_collectinfo_ptr ((*rprotolist_here).p, *collist_here, logout);
	  if (cinfo != NULL) {
	    text_t collname = *collist_here;
	    text_tmap::iterator it = cinfo->collectionmeta.find("collectionname");
	    if (it != cinfo->collectionmeta.end()) collname = (*it).second;

	    textout << "<td>";
	    if (cinfo->buildDate > 0)
	      textout << outconvert << disp 
		      << "<a href=\"_httppagex_(about)&c=" << *collist_here
		      << "\" target=\\_top>";

	    textout << outconvert << disp << collname;

	    if (cinfo->buildDate > 0) textout << "</a>";

	    textout << "</td>";

	    if (cinfo->isPublic) textout << "<td>yes</td>";
	    else textout << "<td>no</td>";
	    
	    if (cinfo->buildDate > 0)
	      textout << outconvert << "<td>yes</td>";
	    else
	      textout << "<td>no</td>";
	    
	  } else {
	    textout << "<td></td><td></td><td></td>";
	  }

	  textout << "</tr>\n";
	  collist_here ++;
	}
      }
    }
    rprotolist_here ++;
  }

  textout << "</table></center>\n";
  textout << outconvert << disp << "_status:infofooter_\n";
}

void statusaction::output_generalinfo (cgiargsclass &/*args*/, displayclass &disp, 
				       outconvertclass &outconvert, 
				       ostream &textout, ostream &/*logout*/) {
  if (recpt == NULL) return;
  const recptconf &rcinfo = recpt->get_configinfo ();
  
  textout << outconvert << disp << "_status:infoheader_(General Information)\n";
  textout << outconvert 
	  << "<h2>General information</h2>\n"
	  << "<table>\n"
	  << "<tr valign=top><th>gsdlhome</th><td>\"" << rcinfo.gsdlhome
	  << "\"</td></tr>\n"
	  << "<tr valign=top><th>collection</th><td>\"" << rcinfo.collection
	  << "\"</td></tr>\n"
	  << "<tr valign=top><th>collectdir</th><td>\"" << rcinfo.collectdir
	  << "\"</td></tr>\n"
	  << "<tr valign=top><th>httpprefix</th><td>\"" << rcinfo.httpprefix
	  << "\"</td></tr>\n"
	  << "<tr valign=top><th>httpimg</th><td>\"" << rcinfo.httpimg
	  << "\"</td></tr>\n"
	  << "<tr valign=top><th>gwcgi</th><td>\"" << rcinfo.gwcgi
	  << "\"</td></tr>\n";
    
  // macrofiles
  textout << outconvert << "<tr valign=top><th>macrofiles</th><td>";
  text_tset::const_iterator macrohere = rcinfo.macrofiles.begin ();
  text_tset::const_iterator macroend = rcinfo.macrofiles.end ();
  bool macrofirst = true;
  while (macrohere != macroend) {
    if (!macrofirst) textout << outconvert << ", ";
    macrofirst = false;
    textout << outconvert << "\"" << *macrohere << "\"";
    macrohere++;
  }
  textout << outconvert << "</td></tr>\n";
  
  // saveconf
  textout << outconvert << "<tr valign=top><th>saveconf</th><td>\"" << rcinfo.saveconf
	  << "\"</td></tr>\n";

  // usecookies
  textout << outconvert << "<tr valign=top><th>usecookies</th><td>\"";
  if (rcinfo.usecookies) textout << outconvert << "true";
  else textout << outconvert << "false";
  textout << outconvert << "\"</td></tr>\n";
  
  // logcgiargs
  textout << outconvert << "<tr valign=top><th>logcgiargs</th><td>\"";
  if (rcinfo.logcgiargs) textout << outconvert << "true";
  else textout << outconvert << "false";
  textout << outconvert << "\"</td></tr>\n";

  // pageparams
  textout << outconvert << "<tr valign=top><th>pageparams</th><td>";
  text_tmap::const_iterator params_here = rcinfo.pageparams.begin();
  text_tmap::const_iterator params_end = rcinfo.pageparams.end();
  bool params_first = true;
  while (params_here != params_end) {
    if (!params_first) textout << outconvert << ", ";
    params_first = false;
    textout << outconvert << "\"" << (*params_here).first << "\" != \""
	    << (*params_here).second << "\"";
    
    params_here++;
  }
  textout << outconvert << "</td></tr>\n";

  // macroprecedence
  textout << outconvert << "<tr valign=top><th>macroprecedence</th><td>\"" << rcinfo.macroprecedence
	  << "\"</td></tr>\n";

  // arguments
  cgiargsinfoclass *rcargsinfo = recpt->get_cgiargsinfo_ptr ();
  if (rcargsinfo != NULL) {
    textout << outconvert << "<tr valign=top><th>arguments</th><td>";
    
    cgiargsinfoclass::const_iterator argsinfohere = rcargsinfo->begin ();
    cgiargsinfoclass::const_iterator argsinfoend = rcargsinfo->end ();
    bool argsinfofirst = true;
    while (argsinfohere != argsinfoend) {
      if (!argsinfofirst) textout << outconvert << ", ";
      argsinfofirst = false;
      textout << outconvert << "\"" << (*argsinfohere).second.shortname << "\"";
      argsinfohere++;
    }
    
    textout << outconvert << "</td></tr>\n";
  }      
  
  // actions
  actionmapclass *actions = recpt->get_actionmap_ptr();
  if (actions != NULL) {
    textout << outconvert << "<tr valign=top><th>actions</th><td>";
    
    actionptrmap::iterator actionshere = actions->begin ();
    actionptrmap::iterator actionsend = actions->end ();
    bool actionsfirst = true;
    while (actionshere != actionsend) {
      if (!actionsfirst) textout << outconvert << ", ";
      actionsfirst = false;
      assert ((*actionshere).second.a != NULL);
      if ((*actionshere).second.a != NULL) {
	textout << outconvert << "\"" << (*actionshere).second.a->get_action_name() << "\"";
      }
      actionshere++;
    }
    
    textout << outconvert << "</td></tr>\n";
  }
  
  // browsers
  browsermapclass *browsers = recpt->get_browsermap_ptr();
  if (browsers != NULL) {
    textout << outconvert << "<tr valign=top><th>browsers</th><td>";
    
    browserptrmap::iterator browsershere = browsers->begin ();
    browserptrmap::iterator browsersend = browsers->end ();
    bool browsersfirst = true;
    while (browsershere != browsersend) {
      if (!browsersfirst) textout << outconvert << ", ";
      browsersfirst = false;
      assert ((*browsershere).second.b != NULL);
      if ((*browsershere).second.b != NULL) {
	textout << outconvert << "\"" << (*browsershere).second.b->get_browser_name() << "\"";
      }
      browsershere++;
    }
    
    textout << outconvert << "</td></tr>\n";
  }

  // protocols
  recptprotolistclass *protocols = recpt->get_recptprotolist_ptr ();
  if (protocols != NULL) {
    textout << outconvert << "<tr valign=top><th>protocols</th><td>";
    
    recptprotolistclass::iterator protohere = protocols->begin ();
    recptprotolistclass::iterator protoend = protocols->end ();
    bool protofirst = true;
    while (protohere != protoend) {
      if (!protofirst) textout << outconvert << ", ";
      protofirst = false;
      if ((*protohere).p != NULL) {
	textout << outconvert << "\"" << (*protohere).p->get_protocol_name() << "\"";
      }
      protohere++;
    }
    
    textout << outconvert << "</td></tr>\n";
  }
  
  // converters
  convertinfoclass *converters = recpt->get_convertinfo_ptr ();
  if (converters != NULL) {
    textout << outconvert << "<tr valign=top><th>converters</th><td>";
    
    convertinfoclass::iterator converthere = converters->begin ();
    convertinfoclass::iterator convertend = converters->end ();
    bool convertfirst = true;
    while (converthere != convertend) {
      if (!convertfirst) textout << outconvert << ", ";
      convertfirst = false;
      textout << outconvert << "\"" << (*converthere).second.name << "\"";
      converthere++;
    }
    
    textout << outconvert << "</td></tr>\n";
  }
  
  textout << outconvert << disp << "</table>\n_status:infofooter_\n";
}

void statusaction::output_argumentinfo (cgiargsclass &args, displayclass &disp, 
					outconvertclass &outconvert, 
					ostream &textout, ostream &/*logout*/) {
  if (recpt == NULL) return;
  cgiargsinfoclass *rcargsinfo = recpt->get_cgiargsinfo_ptr ();
  if (rcargsinfo == NULL) return;

  textout << outconvert << disp << "_status:infoheader_(Argument Information)\n";
  textout << outconvert
	  << "<h2>Argument information</h2>\n"
	  << "<table>";

  // argument information
  textout << outconvert << "<tr valign=top><th>short name</th><th>long name</th>"
	  << "<th>multiple char?</th>"
	  << "<th>multiple value?</th>"
	  << "<th>default</th><th>default status</th><th>saved args</th>"
	  << "<th>current value</th></tr>\n";
  

  cgiargsinfoclass::const_iterator argsinfohere = rcargsinfo->begin();
  cgiargsinfoclass::const_iterator argsinfoend = rcargsinfo->end();
  text_t *arg_value;
  while (argsinfohere != argsinfoend) {
    const cgiarginfo &ainfo = (*argsinfohere).second;
    textout << outconvert 
	    << "<tr valign=top><td>" << ainfo.shortname << "</td>\n";
    
    textout << outconvert << "<td>" << ainfo.longname << "</td>\n";
    if (ainfo.multiplechar) textout << outconvert << "<td>yes</td>\n";
    else textout << outconvert << "<td>no</td>\n";
    if (ainfo.multiplevalue) textout << outconvert << "<td>yes</td>\n";
    else textout << outconvert << "<td>no</td>\n";
    textout << outconvert << "<td>" << ainfo.argdefault << "</td>\n";
    switch (ainfo.defaultstatus) {
    case cgiarginfo::none: textout << outconvert << "<td>none</td>\n"; break;
    case cgiarginfo::weak: textout << outconvert << "<td>weak</td>\n"; break;
    case cgiarginfo::good: textout << outconvert << "<td>good</td>\n"; break;
    case cgiarginfo::config: textout << outconvert << "<td>config</td>\n"; break;
    case cgiarginfo::imperative: textout << outconvert << "<td>imperative</td>\n"; break;
    }
    switch (ainfo.savedarginfo) {
    case cgiarginfo::mustnot: textout << outconvert << "<td>mustnot</td>\n"; break;
    case cgiarginfo::can: textout << outconvert << "<td>can</td>\n"; break;
    case cgiarginfo::must: textout << outconvert << "<td>must</td>\n"; break;
    }
    
    arg_value = args.getarg (ainfo.shortname);
    if (arg_value == NULL) textout << outconvert << "<td></td></tr>\n";
    else textout << outconvert << "<td>\"" << *arg_value << "\"</td></tr>\n";
    
    argsinfohere ++;
  }
  
  textout << outconvert << disp << "</table>\n_status:infofooter_\n";
}

void statusaction::output_actioninfo (cgiargsclass &/*args*/, displayclass &disp, 
				      outconvertclass &outconvert, 
				      ostream &textout, ostream &/*logout*/) {
  if (recpt == NULL) return;
  actionmapclass *actions = recpt->get_actionmap_ptr();

  textout << outconvert << disp << "_status:infoheader_(Action Information)\n";
  textout << outconvert
	  << "<h2>Action information</h2>\n"
	  << "<table>";

  // action information
  if (actions != NULL) {
    textout << outconvert 
	    << "<tr><th>action name</th><th>cgi arguments</th></tr>\n";
	
    actionptrmap::iterator actionshere = actions->begin ();
    actionptrmap::iterator actionsend = actions->end ();
    while (actionshere != actionsend) {
      assert ((*actionshere).second.a != NULL);
      if ((*actionshere).second.a != NULL) {
	textout << outconvert 
		<< "<tr><td>" << (*actionshere).second.a->get_action_name()
		<< "</td><td>";
	
	cgiargsinfoclass argsinfo = (*actionshere).second.a->getargsinfo();
	cgiargsinfoclass::const_iterator argsinfohere = argsinfo.begin ();
	cgiargsinfoclass::const_iterator argsinfoend = argsinfo.end ();
	bool aifirst = true;
	while (argsinfohere != argsinfoend) {
	  if (!aifirst) textout << outconvert << ", ";
	  aifirst = false;
	  textout << outconvert << (*argsinfohere).second.shortname;
	  argsinfohere++;
	}
	
	textout << outconvert << "</td></tr>\n";
      }
      actionshere++;
    }
  }
  
  textout << outconvert << disp << "</table>\n_status:infofooter_\n";
}

void statusaction::output_browserinfo (cgiargsclass &/*args*/, displayclass &disp, 
				       outconvertclass &outconvert, 
				       ostream &textout, ostream &/*logout*/) {
  if (recpt == NULL) return;
  browsermapclass *browsers = recpt->get_browsermap_ptr();

  textout << outconvert << disp << "_status:infoheader_(Browser Information)\n";
  textout << outconvert
	  << "<h2>Browser information</h2>\n"
	  << "<table>";

  // browser information
  if (browsers != NULL) {
    textout << outconvert 
	    << "<tr><th>browser name</th><th>default formatstring</th></tr>\n";
	
    browserptrmap::iterator browsershere = browsers->begin ();
    browserptrmap::iterator browsersend = browsers->end ();
    while (browsershere != browsersend) {
      assert ((*browsershere).second.b != NULL);
      if ((*browsershere).second.b != NULL) {
	textout << outconvert 
		<< "<tr><td>" << (*browsershere).second.b->get_browser_name()
		<< "</td><td>" << html_safe ((*browsershere).second.b->get_default_formatstring())
		<< "</td></tr>\n";
      }
      browsershere++;
    }
  }
  
  textout << outconvert << disp << "</table>\n_status:infofooter_\n";
}

void statusaction::output_protocolinfo (cgiargsclass &/*args*/, displayclass &disp, 
					outconvertclass &outconvert, 
					ostream &textout, ostream &logout) {
  if (recpt == NULL) return;
  const recptconf &rcinfo = recpt->get_configinfo ();
  bool colspecific = !rcinfo.collection.empty();
  bool unreachablecol = false;

  recptprotolistclass *rprotolist = recpt->get_recptprotolist_ptr ();
  if (rprotolist == NULL) return;

  textout << outconvert << disp << "_status:infoheader_(Protocol Information)\n";
  textout << outconvert
	  << "<h2>Protocol information</h2>\n"
	  << "<table>\n"
	  << "<tr><th>protocol</th><th>collections</th></tr>\n";

  text_t protoname;
  recptprotolistclass::iterator rprotolist_here = rprotolist->begin();
  recptprotolistclass::iterator rprotolist_end = rprotolist->end();
  while (rprotolist_here != rprotolist_end) {
    if ((*rprotolist_here).p != NULL) {
      protoname = (*rprotolist_here).p->get_protocol_name();
      textout << outconvert
	      << "<tr><td>"
	      << protoname
	      << "</td><td>";

      text_tarray collist;
      comerror_t err;
      (*rprotolist_here).p->get_collection_list (collist, err, logout);
      if (err == noError) {
	text_tarray::iterator collist_here = collist.begin();
	text_tarray::iterator collist_end = collist.end();
	bool first = true;
	while (collist_here != collist_end) {
	  if (!first) textout << outconvert << ", ";
	  first = false;

	  if (colspecific && *collist_here != rcinfo.collection) {
	    unreachablecol = true;
	    textout << outconvert << "\"" << *collist_here << "\"";
	  } else {
	    textout << outconvert << disp
		    << "\"<a href=\"_gwcgi_?e=_compressedoptions_&a=status&p=collectioninfo&pr="
		    << protoname
		    << "&c="
		    << *collist_here
		    << "\">"
		    << *collist_here 
		    << "</a>\"";
	  }

	  collist_here++;
	}

      } else {
	textout << outconvert << "Error (" << get_comerror_string (err) 
		<< ") while getting collect list\n";
      }

      textout << outconvert 
	      << "</td></tr>\n";
    }

    rprotolist_here++;
  }

  textout << outconvert << "</table>\n";
  if (unreachablecol) {
    textout << outconvert 
	    << "<b>Warning:</b> the receptionist is running in collection specific\n"
	    << "mode making some of the collections unreachable.\n";
  }
  textout << outconvert << disp << "_status:infofooter_\n";
}

void statusaction::output_collectioninfo (cgiargsclass &args, displayclass &disp, 
					  outconvertclass &outconvert, 
					  ostream &textout, ostream &logout) {
  if (recpt == NULL) return;

  textout << outconvert << disp << "_status:infoheader_(Collection info)\n";
  textout << outconvert << "<h2>Collection info</h2>\n";

  // get the list of protocols
  recptprotolistclass *rprotolist = recpt->get_recptprotolist_ptr ();
  if (rprotolist == NULL) return;

  // look for the desired protocol
  text_t &arg_pr = args["pr"];
  text_t &arg_c = args["c"];
  recptproto *rproto = NULL;
  recptprotolistclass::iterator rprotolist_here = rprotolist->begin();
  recptprotolistclass::iterator rprotolist_end = rprotolist->end();
  while (rprotolist_here != rprotolist_end) {
    rproto = (*rprotolist_here).p;
    if (rproto != NULL && rproto->get_protocol_name() == arg_pr) {
      // see if the protocol has the collection
      bool hascollection;
      comerror_t err;
      rproto->has_collection (arg_c, hascollection, err, logout);
      if (err == noError && hascollection) break;
    }
    rprotolist_here++;
  }
  
  if (rprotolist_here == rprotolist_end) {
    textout << outconvert << "Protocol \"" << arg_pr << "\" with collection \""
	    << arg_c << "\" was not found\n";

  } else {
    // rproto can't be NULL to get here
    ColInfoResponse_t *collectinfo = recpt->get_collectinfo_ptr (rproto, arg_c, logout);
    if (collectinfo != NULL) {
      textout << outconvert << "<table>\n"
	      << "<tr><th>collection name</th><td>\""
	      << collectinfo->shortInfo.name
	      << "\"</td></tr>\n"

	      << "<tr><th>host</th><td>\""
	      << collectinfo->shortInfo.host
	      << "\"</td></tr>\n"

	      << "<tr><th>port</th><td>\""
	      << collectinfo->shortInfo.port
	      << "\"</td></tr>\n"

	      << "<tr><th>is public?</th><td>";
      if (collectinfo->isPublic) textout << outconvert << "true";
      else textout << outconvert << "false";
      textout << outconvert 
	      << "</td></tr>\n"

	      << "<tr><th>is beta?</th><td>";
      if (collectinfo->isBeta) textout << outconvert << "true";
      else textout << outconvert << "false";
      textout << outconvert 
	      << "</td></tr>\n"

	      << "<tr><th>build date</th><td>\""
	      << collectinfo->buildDate
	      << "\"</td></tr>\n"

	      << "<tr><th>interface languages</th><td>";
      text_tarray::iterator languages_here = collectinfo->languages.begin();
      text_tarray::iterator languages_end = collectinfo->languages.end();
      bool languages_first = true;
      while (languages_here != languages_end) {
	if (!languages_first) textout << outconvert << ", ";
	languages_first = false;
	textout << outconvert << "\"" << *languages_here << "\"";
	languages_here++;
      }

      textout << "<tr><th valign=top>collection metadata</th><td><table>\n";
      text_tmap::iterator meta_here = collectinfo->collectionmeta.begin();
      text_tmap::iterator meta_end = collectinfo->collectionmeta.end();
      while (meta_here != meta_end) {
	textout << outconvert << "<tr><td>" << (*meta_here).first
		<< "</td><td>" << (*meta_here).second << "</td></tr>\n";
	meta_here ++;
      }
      textout << "</table></td></tr>\n";

      textout << "<tr><th valign=top>format info</th><td><table>\n";
      text_tmap::iterator format_here = collectinfo->format.begin();
      text_tmap::iterator format_end = collectinfo->format.end();
      while (format_here != format_end) {
	textout << outconvert << "<tr><td>" << (*format_here).first
		<< "</td><td>" << html_safe((*format_here).second) << "</td></tr>\n";
	format_here ++;
      }
      textout << "</table></td></tr>\n";

      textout << "<tr><th valign=top>building info</th><td><table>\n";
      text_tmap::iterator building_here = collectinfo->building.begin();
      text_tmap::iterator building_end = collectinfo->building.end();
      while (building_here != building_end) {
	textout << outconvert << "<tr><td>" << (*building_here).first
		<< "</td><td>" << (*building_here).second << "</td></tr>\n";
	building_here ++;
      }
      textout << "</table></td></tr>\n";

      textout << outconvert 
	      << "</td></tr>\n"

	      << "<tr><th>number of documents</th><td>\""
	      << collectinfo->numDocs
	      << "\"</td></tr>\n"

	      << "<tr><th>number of sections</th><td>\""
	      << collectinfo->numSections
	      << "\"</td></tr>\n"

	      << "<tr><th>number of words</th><td>\""
	      << collectinfo->numWords
	      << "\"</td></tr>\n"

	      << "<tr><th>number of bytes</th><td>\""
	      << collectinfo->numBytes
	      << "\"</td></tr>\n"

	      << "<tr><th>preferred receptionist</th><td>\""
	      << collectinfo->receptionist
	      << "\"</td></tr>\n"

	      << "</table>";
      
    } else {
      textout << "ERROR (statusaction::output_collectioninfo): while getting collect information\n";
    }


    InfoFiltersResponse_t filterinfo;
    InfoFilterOptionsRequest_t filteroptions_request;
    InfoFilterOptionsResponse_t filteroptions;
    comerror_t err;
    rproto->get_filterinfo (arg_c, filterinfo, err, logout);
    if (err == noError) {
      text_tset::iterator filternames_here = filterinfo.filterNames.begin();
      text_tset::iterator filternames_end = filterinfo.filterNames.end();
      while (filternames_here != filternames_end) {
	textout << outconvert 
		<< "<hr>\n"
		<< "<h3>Filter options for \"" << (*filternames_here) << "\"</h3>\n"
		<< "<table>\n"
		<< "<tr><th>option name</th><th>type</th><th>repeatable</th>"
		<< "<th>default value</th><th>valid values</th></tr>\n";

	filteroptions_request.clear();
	filteroptions_request.filterName = *filternames_here;
	rproto->get_filteroptions (arg_c, filteroptions_request,
				   filteroptions, err, logout);
	if (err == noError) {
	  FilterOption_tmap::iterator filteropt_here = 
	    filteroptions.filterOptions.begin();
	  FilterOption_tmap::iterator filteropt_end = 
	    filteroptions.filterOptions.end();
	  while (filteropt_here != filteropt_end) {
	    textout << outconvert
		    << "<tr><td>\""
		    << (*filteropt_here).second.name
		    << "\"</td>\n"
	      
		    << "<td>";
	    text_t type_string;
	    switch ((*filteropt_here).second.type) {
	    case FilterOption_t::booleant: type_string = "boolean"; break;
	    case FilterOption_t::integert: type_string = "integer"; break;
	    case FilterOption_t::enumeratedt: type_string = "enumerated"; break;
	    case FilterOption_t::stringt: type_string = "string"; break;
	    }
	    textout << outconvert
		    << type_string
		    << "</td>\n"
	      
		    << "<td>";
	    text_t repeat_string;
	    switch ((*filteropt_here).second.repeatable) {
	    case FilterOption_t::onePerQuery: repeat_string = "one per query"; break;
	    case FilterOption_t::onePerTerm: repeat_string = "one per term"; break;
	    case FilterOption_t::nPerTerm: repeat_string = "n per term"; break;
	    }
	    textout << outconvert
		    << repeat_string
		    << "</td>\n"
	      
		    << "<td>\""
		    << (*filteropt_here).second.defaultValue
		    << "\"</td>\n"
	      
		    << "<td>";
	    
	    text_tarray::iterator valid_here = 
	      (*filteropt_here).second.validValues.begin();
	    text_tarray::iterator valid_end = 
	      (*filteropt_here).second.validValues.end();
	    bool valid_first = true;
	    while (valid_here != valid_end) {
	      if (!valid_first) textout << outconvert << ", ";
	      valid_first = false;
	      textout << outconvert << "\"" << *valid_here << "\"";
	      valid_here++;
	    }
	    textout << outconvert
		    << "</td></tr>\n";

	    filteropt_here++;
	  }

	  textout << outconvert
		  << "</table>\n";

	} else {
	  textout << outconvert << "Error (" << get_comerror_string (err) 
		  << ") while getting filter option information\n";
	}

	filternames_here++;
      }

    } else {
      textout << outconvert << "Error (" << get_comerror_string (err) 
	      << ") while getting filter information\n";
    }
  }
  
  textout << outconvert << disp << "_status:infofooter_\n";
}

void statusaction::output_usagelog (cgiargsclass &/*args*/, displayclass &disp,
				    outconvertclass &outconvert, 
				    ostream &textout, ostream &/*logout*/) {

  // output last 100 lines of usage.txt

  text_t logfilename = filename_cat (gsdlhome, "etc", "usage.txt");

  textout << outconvert << disp << "_status:infoheader_(Usage log)\n";
  textout << outconvert << "<h2>Usage log</h2>\n";

  textout << outconvert << "<p>The usage log, "
	  << logfilename << ", contains the\n"
	  << "following information:\n"
	  << "<p>(note that if this file is more than 100 lines long only the last 100 lines will\n"
	  << "be displayed on this page)\n\n"
	  << "<pre>\n";

  // note that we're expecting lines to be no more than 1500 characters on
  // average - should fix this file_tail() thing sometime
  textout << outconvert << file_tail (logfilename, 100, 1500);

  textout << outconvert << disp << "</pre>\n"
	  << "_status:infofooter_\n";
}

void statusaction::output_initlog (cgiargsclass &/*args*/, displayclass &disp,
				   outconvertclass &outconvert, 
				   ostream &textout, ostream &/*logout*/) {

  text_t initfilename = filename_cat (gsdlhome, "etc", "initout.txt");
  char *cinitfilename = initfilename.getcstr();
  if (cinitfilename == NULL) return;

  textout << outconvert << disp << "_status:infoheader_(Init log)\n";
  textout << outconvert << "<h2>Init log</h2>\n";

#ifdef GSDL_USE_IOS_H
  ifstream initin (cinitfilename, ios::in | ios::nocreate);
#else
  ifstream initin (cinitfilename, ios::in);
#endif

  delete cinitfilename;
  if (initin) {
    textout << outconvert << "<p>The initialisation error log, "
	    << initfilename << ", contains the\n";
    textout << outconvert << "following information:\n\n";
    text_t errorpage = "<p><pre>\n";
    
    char c;
    initin.get(c);
    while (!initin.eof ()) {
      errorpage.push_back(c);
      initin.get(c);
    }
    
    errorpage += "</pre>\n";
    initin.close();
    textout << outconvert << errorpage;
    
  } else {
    textout << outconvert << "Couldn't read initialisation error log, " 
	    << initfilename << ".\n";
  }

  textout << outconvert << disp << "_status:infofooter_\n";
}

void statusaction::output_errorlog (cgiargsclass &/*args*/, displayclass &disp, 
				    outconvertclass &outconvert, 
				    ostream &textout, ostream &logout) {

  text_t errfilename = filename_cat (gsdlhome, "etc", "errout.txt");
  char *cerrfilename = errfilename.getcstr();
  if (cerrfilename == NULL) return;

  textout << outconvert << disp << "_status:infoheader_(Error log)\n";
  textout << outconvert << "<h2>Error log</h2>\n";
  logout << flush;

#ifdef GSDL_USE_IOS_H
  ifstream errin (cerrfilename, ios::in | ios::nocreate);
#else
  ifstream errin (cerrfilename, ios::in);
#endif

  delete cerrfilename;
  if (errin) {
    textout << outconvert << "<p>The error log, "
		<< errfilename << ", contains the\n";
    textout << outconvert << "following information:\n\n";
    text_t errorpage = "<p><pre>\n";

    char c;
    errin.get(c);
    while (!errin.eof ()) {
      errorpage.push_back(c);
      errin.get(c);
    }
    
    errorpage += "</pre>\n";
    errin.close();
    textout << outconvert << errorpage;

  } else {
    textout << outconvert << "Couldn't read error log, " 
		<< errfilename << ".\n";
  }

  textout << outconvert << disp << "_status:infofooter_\n";
}

void statusaction::output_gsdlsite (cgiargsclass &/*args*/, displayclass &disp, 
				    outconvertclass &outconvert, 
				    ostream &textout, ostream &logout) {

  textout << outconvert << disp << "_status:infoheader_(gsdlsite.cfg)\n";
  textout << outconvert << "<h2>gsdlsite.cfg</h2>\n";
  logout << flush;

#ifdef GSDL_LOCAL_LIBRARY
  text_t gsdlsite_cfg = filename_cat (gsdlhome, "gsdlsite.cfg");
#else
  text_t gsdlsite_cfg = "gsdlsite.cfg";
#endif

  char *gsdlsite_cfgc = gsdlsite_cfg.getcstr();
#ifdef GSDL_USE_IOS_H
  ifstream gsdlsitein (gsdlsite_cfgc, ios::in | ios::nocreate);
#else
  ifstream gsdlsitein (gsdlsite_cfgc, ios::in);
#endif

  delete gsdlsite_cfgc;

  if (gsdlsitein) {
    textout << "<p>The gsdlsite.cfg configuration file contains the\n"
	    << "following information:\n\n";
    text_t gsdlsite = "<p><pre>\n";

    char c;
    gsdlsitein.get(c);
    while (!gsdlsitein.eof ()) {
      gsdlsite.push_back(c);
      gsdlsitein.get(c);
    }
    
    gsdlsite += "</pre>\n";
    gsdlsitein.close();
    textout << outconvert << gsdlsite;

  } else {
    textout << "Couldn't read gsdlsite.cfg configuration file\n";
  }

  textout << outconvert << disp << "_status:infofooter_\n";
}

void statusaction::output_maincfg (cgiargsclass &/*args*/, displayclass &disp, 
				   outconvertclass &outconvert, 
				   ostream &textout, ostream &/*logout*/) {

  text_t maincfgfile = filename_cat (gsdlhome, "etc", "main.cfg");

  textout << outconvert << disp << "_status:infoheader_(main.cfg)\n"
	  << "<h2>main.cfg</h2>\n"
	  << "<p>The main configuration file, "
	  << dm_safe(maincfgfile) << ", contains the following information:<br>\n\n"
	  << "(Note that only users belonging to the \"administrator\" group "
	  << "may edit this file)<br>\n"
	  << "_status:maincfg_<br>\n"
	  << "_status:infofooter_\n";
}

void statusaction::change_maincfg (cgiargsclass &args, displayclass &disp, 
				   outconvertclass &outconvert, 
				   ostream &textout, ostream &logout) {

  // write out the contents of the cfgfile argument to main.cfg
  text_t cfgfile = filename_cat(gsdlhome, "etc", "main.cfg");
  char *cfgfilec = cfgfile.getcstr();
#ifdef __WIN32__
  ofstream cfg_out (cfgfilec, ios::binary);
#else
  ofstream cfg_out (cfgfilec);
#endif
  
  if (cfg_out) {
    // lock the file
    int fd = GSDL_GET_FILEDESC(cfg_out);
    int lock_val = 1;
    GSDL_LOCK_FILE (fd);
    if (lock_val != 0) {
      logout << "statusaction::change_maincfg: Error: Couldn't lock file " << cfgfilec << "\n";
      cfg_out.close();
      textout << outconvert << disp << "_status:changemaincfgfail_";
      
    } else {
      
      outconvertclass text_t2ascii;
      cfg_out << text_t2ascii << args["cfgfile"];
      GSDL_UNLOCK_FILE (fd);
      cfg_out.close();
      textout << outconvert << disp << "_status:changemaincfgsuccess_";
    }
  } else {
    logout << "statusaction::change_maincfg: Error: Couldn't open file " << cfgfilec << "for writing\n";
    textout << outconvert << disp << "_status:changemaincfgfail_";
  }
}

void statusaction::output_errorpage (outconvertclass &outconvert, 
				     ostream &textout, ostream &/*logout*/,
				     text_t message) {
  textout << outconvert 
	  << "<html>\n"
	  << "<head>\n"
	  << "<title>Error</title>\n"
	  << "</head>\n"
	  << "<body bgcolor=\"#ffffff\" text=\"#000000\" link=\"#006666\" "
	  << "alink=\"#cc9900\" vlink=\"#666633\">\n"
	  << "<h2>Oops!</h2>\n" 
	  << message
	  << "\n</body>\n"
	  << "</html>\n";
  return;
}



statusaction::statusaction () {
  disabled = true;
  recpt = NULL;

  // this action uses cgi variable "a"
  cgiarginfo arg_ainfo;
  arg_ainfo.shortname = "a";
  arg_ainfo.longname = "action";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "status";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "p" -- status page
  arg_ainfo.shortname = "p";
  arg_ainfo.longname = "page";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "frameset";
  arg_ainfo.savedarginfo = cgiarginfo::must;
  argsinfo.addarginfo (NULL, arg_ainfo);

  // "pr" -- protocol
  arg_ainfo.shortname = "pr";
  arg_ainfo.longname = "protocol";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::none;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::can;
  argsinfo.addarginfo (NULL, arg_ainfo);

  arg_ainfo.shortname = "cfgfile";
  arg_ainfo.longname = "configuration file contents";
  arg_ainfo.multiplechar = true;
  arg_ainfo.defaultstatus = cgiarginfo::weak;
  arg_ainfo.argdefault = "";
  arg_ainfo.savedarginfo = cgiarginfo::mustnot;
  argsinfo.addarginfo (NULL, arg_ainfo);
}

statusaction::~statusaction () {
}

bool statusaction::check_cgiargs (cgiargsinfoclass &/*argsinfo*/, cgiargsclass &args, 
				  ostream &/*logout*/) {

  // only users belonging to the "administrator" group may edit 
  // the main.cfg file
  if (args["p"] == "maincfg" || args["p"] == "changemaincfg") {
    args["uan"] = "1";
    args["ug"] = "administrator";
  }
  return true;
}

void statusaction::get_cgihead_info (cgiargsclass &/*args*/, recptprotolistclass * /*protos*/,
				     response_t &response, text_t &response_data, 
				     ostream &/*logout*/) {
  response = content;
  response_data = "text/html";
}


void statusaction::define_internal_macros (displayclass &disp, cgiargsclass &args, 
					   recptprotolistclass * /*protos*/, ostream &logout) {

  // define_internal_macros sets the following macros:
  // _maincfgfile_   the contents of the main.cfg configuration file

  if (args["p"] == "maincfg") {

    // read in main.cfg
    text_t maincfg = filename_cat(gsdlhome, "etc", "main.cfg");  
    char *maincfgc = maincfg.getcstr();
    
#ifdef GSDL_USE_IOS_H
    ifstream cfg_ifs (maincfgc, ios::in | ios::nocreate);
#else
    ifstream cfg_ifs (maincfgc, ios::in);
#endif
    
    if (cfg_ifs) {  
      text_t cfgtext;
      char c;
      cfg_ifs.get(c);
      while (!cfg_ifs.eof ()) {
	cfgtext.push_back(c);
	cfg_ifs.get(c);
      }
      cfg_ifs.close();
      
      // define it as a macro
      disp.setmacro("maincfgfile", "status", dm_safe(cfgtext));
      
    } else {
      logout << "statusaction::define_internal_macros: couldn't open configuration file (" 
	     << maincfgc << ") for reading\n";
      disp.setmacro("maincfgfile", "status", "");
    }    
    delete maincfgc;
  }
}

bool statusaction::do_action (cgiargsclass &args, recptprotolistclass *protos, 
			      browsermapclass * /*browsers*/, displayclass &disp, 
			      outconvertclass &outconvert, ostream &textout, 
			      ostream &logout) {
  // make sure the status function is enabled
  if (disabled) {
    textout << outconvert 
	    << "<html>\n"
	    << "<head>\n"
	    << "<title>Status disabled</title>\n"
	    << "</head>\n"
	    << "<body bgcolor=\"#ffffff\" text=\"#000000\" link=\"#006666\" "
	    << "alink=\"#cc9900\" vlink=\"#666633\">\n"
	    << "<h2>Facility disabled</h2>\n" 
	    << "Sorry, the Maintenance and Administration facility is currently disabled\n"
	    << "\n</body>\n"
	    << "</html>\n";
    return true;
  }
  
  // make sure we know about a receptionist
  if (recpt == NULL) {
    output_errorpage (outconvert, textout, logout, 
		      "The status action does not contain information\n"
		      "about any receptionists. The method set_receptionist\n"
		      "was probably not called from the module which instantiated\n"
		      "this status action.\n");
    return true;
  }

  // check to make sure status.dm was read in
  const recptconf &rcinfo = recpt->get_configinfo ();
  text_tset::const_iterator macrohere = rcinfo.macrofiles.begin ();
  text_tset::const_iterator macroend = rcinfo.macrofiles.end ();
  while (macrohere != macroend) {
    if (*macrohere == "status.dm") break;
    macrohere++;
  }
  if (macrohere == macroend) {
    output_errorpage (outconvert, textout, logout, 
		      "The status.dm file was not read in. This macro file is\n" 
		      "needed to display configuration and status information.\n");
    return true;
  }

  // output the required status page
  text_t &arg_p = args["p"];
  if (arg_p == "frameset") output_frameset (args, disp, outconvert, textout, logout);
  else if (arg_p == "select") output_select (args, disp, outconvert, textout, logout);
  else if (arg_p == "welcome") output_welcome (args, protos, disp, outconvert, textout, logout);
  else if (arg_p == "generalinfo") output_generalinfo (args, disp, outconvert, textout, logout);
  else if (arg_p == "argumentinfo") output_argumentinfo (args, disp, outconvert, textout, logout);
  else if (arg_p == "actioninfo") output_actioninfo (args, disp, outconvert, textout, logout);
  else if (arg_p == "browserinfo") output_browserinfo (args, disp, outconvert, textout, logout);
  else if (arg_p == "protocolinfo") output_protocolinfo (args, disp, outconvert, textout, logout);
  else if (arg_p == "collectioninfo") output_collectioninfo (args, disp, outconvert, textout, logout);
  else if (arg_p == "usagelog") output_usagelog (args, disp, outconvert, textout, logout);
  else if (arg_p == "initlog") output_initlog (args, disp, outconvert, textout, logout);
  else if (arg_p == "errorlog") output_errorlog (args, disp, outconvert, textout, logout);
  else if (arg_p == "gsdlsite") output_gsdlsite (args, disp, outconvert, textout, logout);
  else if (arg_p == "maincfg") output_maincfg (args, disp, outconvert, textout, logout);
  else if (arg_p == "changemaincfg") change_maincfg (args, disp, outconvert, textout, logout);
  else {
    output_errorpage (outconvert, textout, logout, 
		      "Unknown page \"" + arg_p + "\".\n");
  }

  return true;
}

void statusaction::configure (const text_t &key, const text_tarray &cfgline) {
  if ((key == "status") && (cfgline.size() == 1) && 
      (cfgline[0] == "true" || cfgline[0] == "on" || cfgline[0] == "enabled")) {
    disabled = false;
  } else {
    // call the parent class to deal with the things which
    // are not dealt with here
    action::configure (key, cfgline);
  }
}
