#!/usr/bin/perl -w

###########################################################################
#
# filecopy.pl --
# A component of the Greenstone digital library software
# from the New Zealand Digital Library Project at the 
# University of Waikato, New Zealand.
#
# Copyright (C) 1999 New Zealand Digital Library Project
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
###########################################################################


# This program will "download" the specified files/directories

BEGIN {
    die "GSDLHOME not set\n" unless defined $ENV{'GSDLHOME'};
    unshift (@INC, "$ENV{'GSDLHOME'}/perllib");
}

use util;
use parsargv;
use File::stat;
use FileHandle;

sub print_usage {
    print STDERR "\n";
    print STDERR "filecopy.pl: Recursively copies files into a collections import directory.\n\n";
    print STDERR "\n  usage: $0 [options] [directories] collection-name\n\n";

    print STDERR "  options:\n";
    print STDERR "   -collectdir directory   Collection directory (defaults to " .
	&util::filename_cat ($ENV{'GSDLHOME'}, "collect") . ")\n";
    print STDERR "   -out                    Filename or handle to print output status to.\n";
    print STDERR "                           The default is STDERR\n\n";
}

sub download_files
{
    my $dirname = pop(@_);
    my $full_importname 
	= &util::filename_cat($collectdir, $dirname, "import");

    # split argv into 2 lists: files and directories
    my (@files,@dirs) = ((),());
    my $a;
    foreach $a (@_)
    {
	$a =~ s/^\"//;
	$a =~ s/\"$//;

	if (-e $a)
	{
	    if (-d $a)
	    {
		push(@dirs,$a);
	    }
	    else
	    {
		push(@files,$a);
	    }
	}
	else
	{
	    print $out "Error: filename '$a' does not exist\n";
	}
    }
    
    if (scalar(@files)>0)
    {
	my $f;	
	foreach $f (@files)
	{
	    my $src_file = $f;
	    my $dst_file = &get_dst_dir ($full_importname,$f);

	    my $do_copy = "no";
	    if (-e $dst_file)
	    {
		# compare file dates
		my $src_stat = stat($src_file);
		my $dst_stat = stat($dst_file);

		if ($src_stat->mtime > $dst_stat->mtime)
		{
		    $do_copy = "yes";
		}
	    }
	    else
	    {
		$do_copy = "yes";
	    }

	    if ($do_copy eq "yes")
	    {
		print $out "Copying $src_file-->$dst_file\n";
		&util::cp($src_file,$dst_file);
	    }
	}
    }


    if (scalar(@dirs)>0)
    {
	my $d;	
	foreach $d (@dirs)
	{
	    my $src_dir = $d;
	    my $dst_dir = &get_dst_dir ($full_importname, $d);
	    if (!-e $dst_dir)
	    {
		# create it if it doesn't exist
		&util::mk_all_dir($dst_dir);
	    }

	    # read in dir
            if (!opendir (INDIR, $d)) 
	    {
                print $out "Error: Could not open directory $d\n";
            } 
	    else 
	    {
                my @sub_files = grep (!/^\.\.?$/, readdir (INDIR));
                closedir (INDIR);

		map { $_ = &util::filename_cat($d,$_); } @sub_files;
		download_files(@sub_files,$dirname);
	    }
	}
    }
}

sub main
{
    if (!parsargv::parse(\@ARGV, 
			 'collectdir/.*/', \$collectdir,
			 'out/.*/STDERR', \$out)) {
	&print_usage();
	die "\n";
    }

    if ($collectdir !~ /\w/) {
	$collectdir = &util::filename_cat ($ENV{'GSDLHOME'}, "collect");
    }

    my $close_out = 0;
    if ($out !~ /^(STDERR|STDOUT)$/i) {
	open (OUT, ">$out") || die "Couldn't open output file $out\n";
	$out = OUT;
	$close_out = 1;
    }
    $out->autoflush(1);

    download_files(@ARGV);

    close OUT if $close_out;
    return 0;
}

sub get_dst_dir {
    my ($full_importname, $dir) = @_;

    if ($ENV{'GSDLOS'} eq "windows") {
	# don't want windows filenames like c:\gsdl\...\import\c:\dir
	$dir =~ s/^[a-z]:[\\\/]//i;
    }
    return &util::filename_cat($full_importname,$dir);
}

&main();
